function [action,NewOrdering] = ReOrderShockDLG(selector,infotype,AllVars,AllShocks,Ordering,CurrINI)
% ReOrderShockDLG: Dialog for reordering shocks used for observed variable scenarios and for conditional
%                  forecasting.
%
% USAGE:
%
%       [action,NewOrdering] = ReOrderShockDLG(selector,infotype,AllVars,AllShocks,Ordering,CurrINI)
%
% REQUIRED INPUT:  selector (string vector), with the possible values "init", "slider", "quit"
%
% OPTIONAL INPUT:  infotype (string vector), taking on the value "Conditional" or "Scenario".
%
%                  AllVars (string matrix), with names of all scenario or conditional variables.
%
%                  AllShocks (string matrix), with the names of all shocks.
%
%                  Ordering (vector), with the current ordering of the shocks.
%
%                  CurrINI (structure), with initialization information.
%
% Optional OUTPUT: action (string vector) taking the value "OK" or "Cancel". Depends on which button was
%                         clicked on.
%
%                  NewOrdering (vector), with row number for the positions in AllVars of the vartype.
%
% NOTE: If selector = "init", then  all optional inputs and outputs are required.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: September 10, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 11-9-2007:  Finished the function.
%
% * 18-10-2007: Updated the documentation.
%
% * 30-10-2007: Updated the documentation.
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==1;
   SV.AllVars = AllVars;
   SV.SplitStringMatrix = '';
   for i=1:size(AllVars,1);
      SV.SplitStringMatrix = strvcat(SV.SplitStringMatrix,' <= ');
   end;
   SV.AllShocks = AllShocks;
   SV.Ordering = Ordering;
   scrsz = get(0,'ScreenSize');
   ImageStructure = ReadImages;
   %
   % try to determine the width and the height
   %
   width = 600;
   height = 85+(15*length(Ordering));
   if height<145;
      height = 145;
   elseif height>500;
      height = 500;
   end;
else;
   ReOrderShockGUI = findobj('Type','figure','Tag','ReOrderShockDLG');
   controls = get(ReOrderShockGUI,'UserData');
   SV = get(controls.cancel,'UserData');
end;

switch selector
   case 'init'
      ReOrderShockGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','ReOrderShockDLG.m', ...
         'MenuBar','none', ...
         'Units','pixels', ...
         'Position',[(scrsz(3)-width)/2 (scrsz(4)-height)/2 width height], ...
         'Tag','ReOrderShockDLG', ...
         'Resize','off', ...
         'Visible','off', ...
         'Name',['Select ' infotype ' Shock Ordering'], ...
         'NumberTitle','off', ...
         'WindowStyle','modal', ...
         'CloseRequestFcn','ReOrderShockDLG quit', ...
         'ToolBar','none');
      %
      if MatlabNumber>=7.0;
         set(ReOrderShockGUI,'DockControl','off');
      end;
      %
      AxesBox([2 2 width-2 height-10],[infotype ' Variables and Shock Ordering'],100,[0.5 1],'on',CurrINI);
      %
      % Create a listbox
      %
      controls.variables = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[20 60 465 height-85], ...
         'FontName','FixedWidth', ...
         'Style','listbox', ...
         'Enable','on', ...
         'Min',0, ...
         'Max',1, ...
         'HorizontalAlignment','left', ...
         'String',[AllVars SV.SplitStringMatrix AllShocks(Ordering,:)], ...
         'Visible','on', ...
         'UserData',Ordering, ...
         'Tag','VariablesListBox',...
         'Value',1);
      %
      % move up buttons
      %
      controls.up = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-95 ((height-80)/2)+60 75 25], ...
         'TooltipString','Move Up', ...
         'CData',ImageStructure.up, ...
         'Enable','on', ...
         'Callback','ReOrderShockDLG reorder', ...
         'Tag','MoveUpButton');
      %
      % move down button
      %
      controls.down = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-95 ((height-140)/2)+60 75 25], ...
         'TooltipString','Move Up', ...
         'CData',ImageStructure.down, ...
         'Enable','on', ...
         'Callback','ReOrderShockDLG reorder', ...
         'Tag','MoveDownButton');
      %
      % Two buttons to select action
      %
      controls.ok = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-255 20 75 25], ...
         'String','OK', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'Callback','ReOrderShockDLG quit', ...
         'UserData','waiting', ...
         'Tag','ButtonOK');
      %
      controls.resetbutton = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-175 20 75 25], ...
         'String','Reset', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','off', ...
         'Callback','ReOrderShockDLG reset', ...
         'Tag','ButtonReset');
      %
      controls.cancel = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-95 20 75 25], ...
         'String','Cancel', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'UserData',SV, ...
         'Callback','ReOrderShockDLG quit', ...
         'Tag','ButtonCancel');
      %
      set(ReOrderShockGUI,'UserData',controls);
   case 'reorder'
      OldValue = get(controls.variables,'Value');
      Ordering = get(controls.variables,'UserData');
      if (gco==controls.up)&(OldValue>1);
         %
         % move current value up
         %
         UpValue = Ordering(OldValue);
         Ordering(OldValue) = Ordering(OldValue-1);
         Ordering(OldValue-1) = UpValue;
         set(controls.variables,'String',[SV.AllVars SV.SplitStringMatrix SV.AllShocks(Ordering,:)],'Value',OldValue-1,'UserData',Ordering);
      elseif (gco==controls.down)&(OldValue<length(Ordering));
         %
         % move current value down
         %
         DownValue = Ordering(OldValue);
         Ordering(OldValue) = Ordering(OldValue+1);
         Ordering(OldValue+1) = DownValue;
         set(controls.variables,'String',[SV.AllVars SV.SplitStringMatrix SV.AllShocks(Ordering,:)],'Value',OldValue+1,'UserData',Ordering);
      end;
      if max(abs(SV.Ordering-Ordering))>0;
         set(controls.resetbutton,'Enable','on');
      else;
         set(controls.resetbutton,'Enable','off');
      end;
   case 'reset'
      set(controls.variables,'String',[SV.AllVars SV.SplitStringMatrix SV.AllShocks(SV.Ordering,:)],'UserData',SV.Ordering,'Value',1);
      set(controls.resetbutton,'Enable','off');
   case 'quit'
      set(controls.ok,'UserData','done');
end;

if strcmp(selector,'init')==1;
   set(ReOrderShockGUI,'Visible','on');
   drawnow;
   if MatlabNumber>=7;
      uicontrol(controls.ok);
   end;
   %
   waitfor(controls.ok,'UserData','done');
   if (isempty(get(ReOrderShockGUI,'CurrentObject'))==1)|(strcmp(get(get(ReOrderShockGUI,'CurrentObject'),'Type'),'uicontrol')==0);
      action = 'Cancel';
   else;
      action = get(get(ReOrderShockGUI,'CurrentObject'),'String');
      if (isempty(action)==1)|(strcmp(action,'OK')==0);
         action = 'Cancel';
      end;
   end;
   SV = get(controls.cancel,'UserData');
   %
   % determine output
   %
   NewOrdering = get(controls.variables,'UserData');
   %
   delete(ReOrderShockGUI);
   drawnow;
   pause(0.02);
end;

%
% end of ReOrderShockDLG.m
%
