function RawPosteriorPlots(ParamSample,NumBurnin,CurrChain,TypeStr,DSGEModel,CurrINI)
% RawPosteriorPlots: Provides plots of the posterior draws over all the draws
%
% USAGE:
%
%       RawPosteriorPlots(ParamSample,NumBurnin,CurrChain,TypeStr,DSGEModel,CurrINI)
%
% REQUIRED INPUT: ParamSample (matrix) of dimension NumIterxNumParam with all the posterior
%                             draws of the original parameters.
%
%                 NumBurnin (integer) with the number of burn-in draws.
%
%                 CurrChain (integer) with the number for the used sampling chain.
%
%                 TypeStr (string vector) that takes the values "Original" or "Transformed".
%
%                 DSGEModel (structure)  whose fields contain all required information
%                             about the model, such as input files, variable and sample
%                             data. Actual observations are not stored in this file.
%
%                 CurrINI (structure) with initialization information.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: December 13, 2006.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 14-12-2006: Added the Chain number to figure name string.
%
% * 18-12-2006: Added an additional drawnow command for refreshing all figures.
%
% * 9-1-2007:   The plots are now automatically stored in the graphs directory.
%
% * 8-2-2007:   Added a second "drawnow" call at the end of the figure loop.
%
% * 28-2-2007:  Added the "TypeStr" string input. Thereby we can plots posterior draws of either
%               the original parameters or of the transformed parameters.
%
% * 9-11-2007:  Updated the documentation.
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 10-3-2008:  Made sure that the "CurrChain" variable affects the filename for the figure.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

[NumIter,NumParam] = size(ParamSample);
%
%
% Try to load the posterior mode estimates
%
ModeFile = [DSGEModel.OutputDirectory '\mode\PosteriorMode-' DSGEModel.NameOfModel '.mat'];
ErrorStr = '';
try;
   ModeData = load(ModeFile);
catch;
   ErrorStr = ['Unable to load the file "' ModeFile '". Message caught is: ' lasterr];
end;
if isempty(ErrorStr)==0;
   About(ErrorStr,'information','YADA - Bad MAT File',120,500,CurrINI);
   return;
end;
%
if strcmp(lower(TypeStr),'original')==1;
   thetaMode = ModeData.thetaMode;
else;
   thetaMode = ModeData.phiMode;
end;
ParameterNames = ModeData.ParameterNames;
%
% Directory where the plots are stored
%
GraphDirectory = [DSGEModel.OutputDirectory '\graphs'];
%
% Compute the number of figures (max 16 plots per graph)
%
NumFigs = ceil(length(thetaMode)/16);
%
if NumFigs==1;
   [NumRows,NumCols] = VillaniConstructOptimalSubplot(NumParam);
else;
   NumRows = 4;
   NumCols = 4;
end;
NumPlots = 0;
pos = [(CurrINI.scrsz(3)-1000)/2 (CurrINI.scrsz(4)-680)/2 1000 650];
Xvalues = (1:NumIter)';
%
% now we can start the loops
%
for i=1:NumFigs;
   if i>1;
      pos(1) = pos(1)+22;
      pos(2) = pos(2)-22;
      if pos(1)+1000>CurrINI.scrsz(3);
         pos(1) = 0;
      end;
      if pos(2)<=32;
         pos(2) = CurrINI.scrsz(4)-700;
      end;
   end;
   FigHandles{i} = figure('Visible','off','Position',pos);
   if NumFigs==1;
      set(FigHandles{i},'Name',['Raw Posterior Draws of ' TypeStr ' Parameters - Chain ' num2str(CurrChain,'%0.0f')]);
   else;
      set(FigHandles{i},'Name',['Raw Posterior Draws of ' TypeStr ' Parameters - Figure ' num2str(i,'%0.0f') ' Chain ' num2str(CurrChain,'%0.0f')]);
   end;
   SetFigureProperties(FigHandles{i},CurrINI);
   %
   k = 0;
   for r=1:NumRows;
      for c=1:NumCols;
         NumPlots = NumPlots+1;
         if NumPlots<=NumParam;
            k = k+1;
            subplot(NumRows,NumCols,k);
            %
            % plot the posterior draws
            %
            dh = plot(Xvalues,ParamSample(:,NumPlots),'-','Color',[0.75 0 0],'LineWidth',1);
            hold('on');
            SetAxesFonts(CurrINI,'Title',StringTrim(ParameterNames.estimated(NumPlots,:)));
            axis('tight');
            ylim = get(gca,'YLim');
            add_on = 0.1*(ylim(2)-ylim(1));
            ylim = [ylim(1)-add_on ylim(2)+add_on];
            %
            set(gca,'YLim',ylim,'XLim',[0 NumIter]);
            %
            % plot the BurnIn value
            %
            bi = plot(NumBurnin*ones(1,2),ylim,'-','Color',[0 0 0]);
            hold('on');
            %
            % plot the posterior mode estimate
            %
            pme = plot([0 NumIter],thetaMode(NumPlots)*ones(1,2),'-','Color',[0 0 0]);
            hold('off');
            drawnow;
         end;
      end;
   end;
   %
   % save it to disk before
   %
   if DirExist(GraphDirectory)==1;
      if NumFigs>1;
         file = [GraphDirectory '\RawPosteriorDraws-' TypeStr '-' num2str(NumIter,'%0.0f') '-' int2str(CurrChain) '-' num2str(i,'%0.0f') '.fig'];
      else;
         file = [GraphDirectory '\RawPosteriorDraws-' TypeStr '-' num2str(NumIter,'%0.0f') '-' int2str(CurrChain) '.fig'];
      end;
      saveas(FigHandles{i},file);
   end;
   %
   % Fix the toolbar cdata
   %
   SetToolbarImages(FigHandles{i},CurrINI.images);
   %
   % Display the current figure
   %
   set(FigHandles{i},'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
   drawnow;
   pause(0.02);
end;
%
clear('ParamSample');

%
% end of RawPosteriorPlots.m
%
