function answer = Query(TextStr,ImageStr,height,NameStr,width,FocusStr,CurrINI,ButtonNameStr,ButtonWidth,ButtonTxtFile)
% Query: Displays a dialog with a Yes and a No button and gives the answer
%        as output.
%
% USAGE:
%
%       answer = Query(TextStr,ImageStr,height,NameStr,width,FocusStr,CurrINI,ButtonNameStr,ButtonWidth,ButtonTxtFile)
%
% REQUIRED INPUT:  TextStr (string matrix) with the text to display on the dialog.
%
%                  ImageStr (string vector) that determines which image is displayed. Valid names
%                           are "information", "question", "warning", "error", and "delete". If some
%                           other string vector is given then the default logo image is chosen.
%
%                  height (integer) with the dialog height in pixel units.
%
% OPTIONAL INPUT:  NameStr (string vector) with the figure name.
%
%                  width (integer) with the dialog width in pixel units. Default is 300 and minimum
%                           is 250.
%
%                  FocusStr (string vector) that takes on the values "yes" or "no", where the latter
%                           is the default. Determines which button gets the focus provided that
%                           the version of Matlab supports setting focus on a control.
%
%                  CurrINI (structure) with initialization information.
%
%                  ButtonNameStr (cell array of strings) of dimension 1x2 with alternative names
%                           for the buttons.
%
%                  ButtonWidth (integer) with the length of the button. Minimum is 75.
%
%                  ButtonTxtFile (string vector) with the full path and name of a file to open
%                           with the TextGUI if view button is clicked on.
%
% REQUIRED OUTPUT: answer (string vector) that is either "Yes" or "No". The default is "No".
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: July 25, 2006.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 11-5-2007:  Updated the documentation.
%
% * 15-5-2007:  Made the Yes and the No buttons 75 pixels wide rather than 50 pixels.
%
% * 9-11-2007:  Updated the documentation.
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 23-5-2008:  Updated the documentation.
%
% * 8-12-2008:  Added the input variables "ButtonNameStr", "ButtonWidth", and "ButtonTxtFile".
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if nargin<7;
   CurrINI = InitializeINIFile;
end;
%
if nargin==3;
   NameStr = 'Information';
end;
if nargin<5;
   width = 300;
end;
%
if width<250;
   width = 250;
end;
%
if nargin<6;
   FocusStr = 'yes';
else;
   if ischar(FocusStr)==1;
      FocusStr = lower(FocusStr);
      if (strcmp(FocusStr,'yes')==0)&(strcmp(FocusStr,'no')==0);
         FocusStr = 'yes';
      end;
   else;
      FocusStr = 'yes';
   end;
end;
if nargin<8;
   FirstBtnStr = 'Yes';
   SecondBtnStr = 'No';
   ThirdBtnStr = '';
else;
   if length(ButtonNameStr)==2;
      FirstBtnStr = ButtonNameStr{1};
      SecondBtnStr = ButtonNameStr{2};
      ThirdBtnStr = '';
   elseif length(ButtonNameStr)==3;
      FirstBtnStr = ButtonNameStr{1};
      SecondBtnStr = ButtonNameStr{2};
      ThirdBtnStr = ButtonNameStr{3};
   else;
      FirstBtnStr = 'Yes';
      SecondBtnStr = 'No';
      ThirdBtnStr = '';
   end;
end;
if nargin<9;
   ButtonWidth = 75;
else;
   if isnumeric(ButtonWidth)==1;
      ButtonWidth = floor(ButtonWidth);
      if (2*ButtonWidth)+5>width;
         ButtonWidth = 75;
      end;
   else;
      ButtonWidth = 75;
   end;
end;
%
ImageStructure = ReadImages;
bgcolor = get(0,'defaultuicontrolbackgroundcolor');
%
if strcmp(ImageStr,'information')==1;
   mylogo = ImageStructure.allimages.information_image;
elseif strcmp(ImageStr,'question')==1;
   mylogo = ImageStructure.allimages.question_image;
elseif strcmp(ImageStr,'warning')==1;
   mylogo = ImageStructure.allimages.warning_image;
elseif strcmp(ImageStr,'error')==1;
   mylogo = ImageStructure.allimages.error_image;
elseif strcmp(ImageStr,'exit')==1;
   mylogo = ImageStructure.allimages.exit_image;
elseif strcmp(ImageStr,'delete')==1;
   mylogo = ImageStructure.allimages.deletebig_image;
else;
   mylogo = ImageStructure.allimages.logo_image;
end;
mylogo = MirrorImage(MakeTrans(mylogo,[160 160 164],bgcolor));

scrsz = get(0,'ScreenSize');

textheight = height-54;
axesheight = (height-90)/2;
if axesheight<0;
   axesheight = 0;
   height = 90;
end;

querydlg = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
   'FileName','Query.m', ...
   'MenuBar','none', ...
   'Units','pixels', ...
   'Position',[(scrsz(3)-width)/2 (scrsz(4)-height)/2 width height], ...
   'Tag','FigQuery', ...
   'Resize','off', ...
   'Visible','off', ...
   'Name',NameStr, ...
   'NumberTitle','off', ...
   'WindowStyle','modal', ...
   'CloseRequestFcn','CloseGUI', ...
   'ToolBar','none');

if MatlabNumber>=7.0;
   set(querydlg,'DockControl','off');
end;

imageaxes = axes('Units','pixels', ...
   'Box','on', ...
   'TickLength',[0 0], ...
   'CameraUpVector',[0 1 0], ...
   'Color',get(0,'defaultuicontrolbackgroundcolor'), ...
   'Position',[10 50+axesheight 39 39], ...
   'XTickLabelMode','manual', ...
   'YTickLabelMode','manual', ...
   'Tag','AxisLogo', ...
   'XColor',get(0,'defaultuicontrolbackgroundcolor'), ...
   'YColor',get(0,'defaultuicontrolbackgroundcolor'), ...
   'ZColor',get(0,'defaultuicontrolbackgroundcolor'));

image('CData',mylogo,'Clipping','off','Parent',imageaxes);

query_txt = uicontrol('Units','pixels', ...
   'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
   'Position',[50 50 width-63 textheight], ...
   'Style','text', ...
   'FontSize',CurrINI.gui_fontsize, ...
   'FontName',CurrINI.gui_fontname, ...
   'FontWeight',CurrINI.gui_fontweight, ...
   'FontAngle',CurrINI.gui_fontangle, ...
   'HorizontalAlignment','left', ...
   'String',' ', ...
   'Tag','Text');

%
% need to fix TextStr string and turn it into a cell str
%
[out_str,npos] = textwrap(query_txt,cellstr(TextStr));
set(query_txt,'String',out_str,'Position',[50 50 width-63 npos(4)]);
if max(size(out_str))==1;
   set(query_txt,'HorizontalAlignment','center');
end;

c_pos = 0;
if (60+npos(4)>=90)&(height>60+npos(4));
   %
   % fix the height of the dialog
   %
   set(querydlg,'Position',[(scrsz(3)-width)/2 (scrsz(4)-(60+npos(4)))/2 width 60+npos(4)]);
   c_pos = 60+npos(4);
elseif (60+npos(4)<90)&(height>60+npos(4));
   set(querydlg,'Position',[(scrsz(3)-width)/2 (scrsz(4)-90)/2 width 90]);
   c_pos = 90;
end;
if c_pos>0;
   axesheight = (c_pos-90)/2;
   set(imageaxes,'Position',[10 50+axesheight 39 39]);
end;
%
% The Yes button
%
myquery.yes = uicontrol('Units','pixels', ...
   'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
   'Position',[(width/2)-ButtonWidth-2.5 12 ButtonWidth 25], ...
   'String',FirstBtnStr, ...
   'FontSize',CurrINI.gui_fontsize, ...
   'FontName',CurrINI.gui_fontname, ...
   'FontWeight',CurrINI.gui_fontweight, ...
   'FontAngle',CurrINI.gui_fontangle, ...
   'Enable','on', ...
   'Callback','CloseGUI', ...
   'UserData','waiting', ...
   'Tag','YesButton');
%
% the NO button
%
myquery.no = uicontrol('Units','pixels', ...
   'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
   'Position',[(width/2)+2.5 12 ButtonWidth 25], ...
   'String',SecondBtnStr, ...
   'FontSize',CurrINI.gui_fontsize, ...
   'FontName',CurrINI.gui_fontname, ...
   'FontWeight',CurrINI.gui_fontweight, ...
   'FontAngle',CurrINI.gui_fontangle, ...
   'Enable','on', ...
   'Callback','CloseGUI', ...
   'Tag','NoButton');
%
% check if we should show a View button
%
if nargin==10;
   if (FileExist(ButtonTxtFile)==1)&(isempty(ThirdBtnStr)==0);
      %
      % check if there's enough room
      %
      if 19.5+(2*ButtonWidth)<=(width/2);
         UData.TxtFile = ButtonTxtFile;
         UData.NameStr = 'GNU General Public License';
         UData.AxesBoxText = 'License';
         myquery.view = uicontrol('Units','pixels', ...
            'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
            'Position',[12 12 ButtonWidth 25], ...
            'String',ThirdBtnStr, ...
            'FontSize',CurrINI.gui_fontsize, ...
            'FontName',CurrINI.gui_fontname, ...
            'FontWeight',CurrINI.gui_fontweight, ...
            'FontAngle',CurrINI.gui_fontangle, ...
            'Enable','on', ...
            'UserData',UData, ...
            'Callback','ShowQueryTxtFile', ...
            'Tag','ViewButton');
      end;
   end;
end;
%
set(querydlg,'UserData',myquery, ...
   'HandleVisibility','callback');
%
set(querydlg,'Visible','on');
drawnow;
if MatlabNumber>=7;
   if strcmp(FocusStr,'yes')==1;
      uicontrol(myquery.yes);
   else;
      uicontrol(myquery.no);
   end;
end;
%
waitfor(querydlg,'waitstatus','inactive');
%
% collect the answer
%
answer = get(get(querydlg,'CurrentObject'),'String');
%
if isempty(answer)==1;
   answer = get(myquery.no,'String');
end;
%
delete(querydlg);
drawnow;
pause(0.02);

%
% end of Query.m
%
