function [action,PredictionEvent] = PredictionEventDLG(selector,AllVars,PredictionEvent,MaxPH,CurrINI)
% PredictionEventDLG: Dialog for setting prediction events.
%
% USAGE:
%
%       [action,PredictionEvent] = PredictionEventDLG(selector,AllVars,PredictionEvent,MaxPH,CurrINI)
%
% REQUIRED INPUT:  selector (string vector), with the possible values "init", "checklowerbound",
%                           "checkupperbound", "changeeventlength", "reset", "changevariable", and "quit"
%
% OPTIONAL INPUT:  AllVars (string matrix), with all the observed variables.
%
%                  PredictionEvent (matrix), with the currently suggested upper and lower bounds and lengths
%                           of the prediction events in periods.
%
%                  MaxPH (integer) with the maximum length of the prediction events.
%
%                  CurrINI (structure), with initialization information.
%
% Optional OUTPUT: action (string vector) taking the value "OK" or "Cancel". Depends on which button was
%                           clicked on.
%
%                  PredictionEvent (matrix), with the revised values for the upper, the lower bounds and the
%                           lengths of the prediction events.
%
%
% NOTE: If selector = "init", then  all optional inputs and outputs are required.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: May 3, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 4-5-2007:   Added the ability to handle upper bounds.
%
% * 9-11-2007:  Updated the documentaion.
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==1;
   SV.AllVars = AllVars;
   SV.PredictionEvent = PredictionEvent;
   SV.CurrINI = CurrINI;
   scrsz = get(0,'ScreenSize');
   %
   % try to determine the width and the height
   %
   width = 630;
   height = 130;
else;
   PredictionEventGUI = findobj('Type','figure','Tag','PredictionEventDLG');
   controls = get(PredictionEventGUI,'UserData');
   SV = get(controls.cancel,'UserData');
   CurrINI = SV.CurrINI;
end;

switch selector
   case 'init'
      PredictionEventGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','PredictionEventDLG.m', ...
         'MenuBar','none', ...
         'Units','pixels', ...
         'Position',[(scrsz(3)-width)/2 (scrsz(4)-height)/2 width height], ...
         'Tag','PredictionEventDLG', ...
         'Resize','off', ...
         'Visible','off', ...
         'Name','Prediction Event Specification', ...
         'NumberTitle','off', ...
         'WindowStyle','modal', ...
         'CloseRequestFcn','PredictionEventDLG quit', ...
         'ToolBar','none');
      %
      if MatlabNumber>=7.0;
         set(PredictionEventGUI,'DockControl','off');
      end;
      %
      AxesBox([2 2 width-2 height-10],'Prediction Event Data Selection',100,[0.5 1],'on',CurrINI);
      %
      controls.variablenametext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[20 85 140 20], ...
         'String','Variable:', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','text', ...
         'Enable','on', ...
         'Tag','VariableNameText');
      %
      controls.variables = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[20 65 140 20], ...
         'String',SV.AllVars, ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','popupmenu', ...
         'Callback','PredictionEventDLG changevariable', ...
         'Value',1, ...
         'Enable','on', ...
         'Tag','VariableNamePopup');
      %
      controls.upperboundtext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[170 85 140 20], ...
         'String','Upper bound:', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','text', ...
         'Enable','on', ...
         'Tag','UpperBoundText');
      %
      controls.upperbound = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[170 65 140 20], ...
         'String',num2str(PredictionEvent(1,1),'%0.12f'), ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','edit', ...
         'Callback','PredictionEventDLG checkupperbound', ...
         'Enable','on', ...
         'Tag','UpperBoundEdit');
      %
      controls.lowerboundtext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[320 85 140 20], ...
         'String','Lower bound:', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','text', ...
         'Enable','on', ...
         'Tag','LowerBoundText');
      %
      controls.lowerbound = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[320 65 140 20], ...
         'String',num2str(PredictionEvent(1,2),'%0.12f'), ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','edit', ...
         'UserData',PredictionEvent, ...
         'Callback','PredictionEventDLG checklowerbound', ...
         'Enable','on', ...
         'Tag','LowerBoundEdit');
      %
      controls.eventlengthtext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[470 85 140 20], ...
         'String','Event length:', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','text', ...
         'Enable','on', ...
         'Tag','EventLengthText');
      %
      EventLengthStr = '1 period';
      for i=2:MaxPH;
         EventLengthStr = [EventLengthStr '|' num2str(i,'%0.0f') ' periods'];
      end;
      %
      controls.eventlength = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[470 65 140 20], ...
         'String',EventLengthStr, ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','popupmenu', ...
         'Callback','PredictionEventDLG changeeventlength', ...
         'Value',PredictionEvent(1,3), ...
         'Enable','on', ...
         'Tag','EventLengthPopup');
      %
      % Two buttons to select action
      %
      controls.ok = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-255 20 75 25], ...
         'String','OK', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'Callback','PredictionEventDLG quit', ...
         'UserData','waiting', ...
         'Tag','ButtonOK');
      %
      controls.reset = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-175 20 75 25], ...
         'String','Reset', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'Callback','PredictionEventDLG reset', ...
         'Tag','ButtonReset');
      %
      controls.cancel = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-95 20 75 25], ...
         'String','Cancel', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'UserData',SV, ...
         'Callback','PredictionEventDLG quit', ...
         'Tag','ButtonCancel');
      %
      set(PredictionEventGUI,'UserData',controls);
   case 'changevariable'
      %
      % when the variable name changes we need to change the value displayed in the edit box
      %
      PredictionEvent = get(controls.lowerbound,'UserData');
      val = get(controls.variables,'Value');
      set(controls.upperbound,'String',num2str(PredictionEvent(val,1),'%0.12f'));
      set(controls.lowerbound,'String',num2str(PredictionEvent(val,2),'%0.12f'));
      set(controls.eventlength,'Value',PredictionEvent(val,3));
   case 'changeeventlength'
      PredictionEvent = get(controls.lowerbound,'UserData');
      val = get(controls.variables,'Value');
      PredictionEvent(val,3) = get(controls.eventlength,'Value');
      set(controls.lowerbound,'UserData',PredictionEvent);
   case 'checklowerbound'
      newval = StringTrim(get(controls.lowerbound,'String'));
      newvalue = str2num(newval);
      PredictionEvent = get(controls.lowerbound,'UserData');
      val = get(controls.variables,'Value');
      if (isempty(newvalue)==0)&(isnumeric(newvalue)==1)&(isreal(newvalue)==1)&(newvalue<PredictionEvent(val,1));
         PredictionEvent(val,2) = newvalue;
         set(controls.lowerbound,'UserData',PredictionEvent);
      else;
         set(controls.lowerbound,'String',num2str(PredictionEvent(val,2),'%0.12f'));
         txt = ['The lower bound value you selected (' newval ') for ' StringTrim(SV.AllVars(val,:)) ' is not valid. It must be smaller than the upper bound and only real numbers or -Inf can be used!'];
         About(txt,'information','Prediction Event - Lower Bound',160,500,CurrINI);
      end;
   case 'checkupperbound'
      newval = StringTrim(get(controls.upperbound,'String'));
      newvalue = str2num(newval);
      PredictionEvent = get(controls.lowerbound,'UserData');
      val = get(controls.variables,'Value');
      if (isempty(newvalue)==0)&(isnumeric(newvalue)==1)&(isreal(newvalue)==1)&(newvalue>PredictionEvent(val,2));
         PredictionEvent(val,1) = newvalue;
         set(controls.lowerbound,'UserData',PredictionEvent);
      else;
         set(controls.upperbound,'String',num2str(PredictionEvent(val,1),'%0.12f'));
         txt = ['The upper bound value you selected (' newval ') for ' StringTrim(SV.AllVars(val,:)) ' is not valid. It must be greater than the lower bound and only real numbers or Inf can be used!'];
         About(txt,'information','Prediction Event - Upper Bound',160,500,CurrINI);
      end;
   case 'reset'
      %
      % resets current value to the original
      %
      val = get(controls.variables,'Value');
      set(controls.upperbound,'String',num2str(SV.PredictionEvent(val,1),'%0.12f'));
      set(controls.lowerbound,'UserData',SV.PredictionEvent,'String',num2str(SV.PredictionEvent(val,2),'%0.12f'));
      set(controls.eventlength,'Value',SV.PredictionEvent(val,3));
   case 'quit'
      set(controls.ok,'UserData','done');
end;

if strcmp(selector,'init')==1;
   set(PredictionEventGUI,'Visible','on');
   drawnow;
   if MatlabNumber>=7;
      uicontrol(controls.ok);
   end;
   %
   waitfor(controls.ok,'UserData','done');
   if (isempty(get(PredictionEventGUI,'CurrentObject'))==1)|(strcmp(get(get(PredictionEventGUI,'CurrentObject'),'Type'),'uicontrol')==0);
      action = 'Cancel';
   else;
      action = get(get(PredictionEventGUI,'CurrentObject'),'String');
      if (isempty(action)==1)|(strcmp(action,'OK')==0);
         action = 'Cancel';
      end;
   end;
   SV = get(controls.cancel,'UserData');
   %
   % determine output
   %
   if strcmp(action,'OK')==1;
      %
      % choose new values
      %
      PredictionEvent = get(controls.lowerbound,'UserData');
   else;
      PredictionEvent = SV.PredictionEvent;
   end;
   %
   delete(PredictionEventGUI);
   drawnow;
   pause(0.02);
end;

%
% end of PredictionEventDLG.m
%
