function PlotMultiANOVA(MPSRF,NumBurnin,NumChains,DSGEModel,CurrINI)
% PlotMultiANOVA: Prepares plot of the multivariate potential scale reduction factor and the
%                 determinants of the between-chain and the within-chain covariance.
%
% USAGE:
%
%       PlotMultiANOVA(MPSRF,NumBurnin,NumChains,DSGEModel,CurrINI)
%
% REQUIRED INPUT: MPSRF (matrix) of dimension NumSeqEst x 4, where NumSeqEst is the number
%                       of sequential estimates. The first column contains the number of
%                       observations included in the sequatial estimation, the second the
%                       Multivariate Potential Scale Reduction Factor, the third column the
%                       determinant of V (the total covariance matrix for the
%                       parameters), and the fourth column the determinant of W (the
%                       within-chain covariance matrix for the parameters).
%
%                 NumBurnin (integer) with the number of burn-in draws.
%
%                 NumChains (integer) with the number of parallel MCMC chains
%
%                 DSGEModel (structure)  whose fields contain all required information
%                       about the model, such as input files, variable and sample data.
%                       Actual observations are not stored in this file.
%
%                 CurrINI (structure) with initialization information.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: April 24, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 9-11-2007:  Updated the documentation.
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% set the data
%
Xvalues = MPSRF(:,1)+NumBurnin;
%
GraphDirectory = [DSGEModel.OutputDirectory '\graphs'];
%
FigHandle =figure('Visible','off', ...
       'Position',[(CurrINI.scrsz(3)-750)/2 (CurrINI.scrsz(4)-550)/2 750 550], ...
       'Tag','MPSRF', ...
       'Name','Multivariate ANOVA - MPSRF');
%
SetFigureProperties(FigHandle,CurrINI);
%
% plot the multivariate potential scale reduction factors against the number of
% draws from the posterior
%
subplot(2,1,1);
MPHandle = plot(Xvalues,MPSRF(:,2),'LineStyle','-','Color',[0.75 0 0],'LineWidth',2);
hold('on');
ylim = get(gca,'YLim');
ylim_2 = ylim(2);
%
NBHandle = plot(NumBurnin*ones(1,2),[1 ylim_2],'LineStyle','-','Color',[0 0 0]);
hold('on');
CLineHandle = plot([0 Xvalues(length(Xvalues))],1.1*ones(1,2),'LineStyle','-','Color',[0 0 0]);
hold('off');
set(gca,'YLim',[1 ylim_2],'XLim',[0 Xvalues(length(Xvalues))],'Children',[MPHandle NBHandle CLineHandle]);
%
% set the title
%
SetAxesFonts(CurrINI,'Title','Sequential estimates of the multivariate potential scale reduction factor (MPSRF)','XLabel',['RWM draw number (with ' num2str(NumBurnin,'%0.0f') ' burn in draws)'],'YLabel','MPSRF');
%
subplot(2,1,2);
WHandle = plot(Xvalues,MPSRF(:,4),'LineStyle','--','Color',[0.75 0 0],'LineWidth',2);
hold('on');
VHandle = plot(Xvalues,MPSRF(:,3),'LineStyle','-','Color',[0 0 0.75],'LineWidth',2);
hold('on');
ylim = get(gca,'YLim');
ylim_2 = 1.1*ylim(2);
%
NBHandle = plot(NumBurnin*ones(1,2),[0 ylim_2],'LineStyle','-','Color',[0 0 0]);
hold('off');
set(gca,'YLim',[0 ylim_2],'XLim',[0 Xvalues(length(Xvalues))],'Children',[VHandle WHandle NBHandle]);
%
% set the title
%
SetAxesFonts(CurrINI,'Title',['Total and within variation in the ' num2str(NumChains,'%0.0f') ' parallel MCMC chains'],'XLabel',['RWM draw number (with ' num2str(NumBurnin,'%0.0f') ' burn in draws)']);
%
legHandle = legend([WHandle VHandle],strvcat('Determinant within variance','Determinant total variance'),2);
SetAxesFonts(CurrINI,legHandle);
%
% save to disk
%
if DirExist(GraphDirectory)==1;
   FigFile = [GraphDirectory '\MANOVA-' DSGEModel.NameOfModel '-' num2str(NumChains,'%0.0f') '.' num2str(NumBurnin,'%0.0f') '.' num2str(Xvalues(length(Xvalues)),'%0.0f') '.fig'];
   saveas(FigHandle,FigFile);
end;
%
% display the graph
%
SetToolbarImages(FigHandle,CurrINI.images);
set(FigHandle,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
drawnow;
pause(0.02);

%
% end of PlotMultiANOVA.m
%
