function [FigHandles,LastPosition,UsedRowsCols] = PlotEditedData(FirstPosition,OriginalData,EditedData,CompData,OriginalDateVector,DateVector,VerticalLinePos,DataNames,NameStr,OriginalColor,EditedColor,CompColor,OriginalLineStyle,EditedLineStyle,CompLineStyle,OriginalPrependStr,EditedPrependStr,CompPrependStr,MatchMaxPlots,MaxPlots,CurrINI)
% PlotEditedData: Plots edited data and the original.
%
% USAGE:
%
%       [FigHandles,LastPosition,UsedRowsCols] = PlotEditedData(FirstPosition,OriginalData,EditedData,CompData,OriginalDateVector,...
%                                       DateVector,VerticalLinePos,DataNames,NameStr,OriginalColor,EditedColor,CompColor,...
%                                       OriginalLineStyle,EditedLineStyle,CompLineStyle,OriginalPrependStr,EditedPrependStr,...
%                                       CompPrependStr,MatchMaxPlots,MaxPlots,CurrINI)
%
% REQUIRED INPUT:  FirstPosition (vector) with lower left corner values in pixels of the first figure.
%                                The vector may be empty or have two elements.
%
%                  OriginalData (matrix) of dimension nxTO with the original data for the plot.
%
%                  EditedData (matrix) of dimension nxTE with the edited data for the plot.
%
%                  CompData (matrix) of dimension nxTC with any comparison data for the plot. Notice
%                                that the number of observations TC cannot exceed that of OriginalData
%                                (if non-empty).
%
%                  OriginalDateVector (vector) of dimention 1xTO with numerical dates for the original
%                                data.
%
%                  DateVector (vector) of dimension 1xTE with numerical dates for the edited data.
%
%                  VerticalLinePos (real) with the position on the X-axis where a vertical line should
%                                be drawn. May be empty.
%
%                  DataNames (string matrix) with n rows holding the names of the variables.
%
%                  NameStr (string vector) with the name for the window of the plots.
%
%                  OriginalColor (vector) of dimension 1x3 with numbers between 0 and 1 for the line
%                                color of the original data.
%
%                  EditedColor (vector) of dimension 1x3 with numbers between 0 and 1 for the line
%                                color of the edited data.
%
%                  CompColor (vector) of dimension 1x3 with numbers between 0 and 1 for the line
%                                color of the comparison data.
%
%                  OriginalLineStyle (string vector) with the line style for the original data.
%
%                  EditedLineStyle (string vector) with the line style for the edited data.
%
%                  CompLineStyle (string vector) with the line style for the comparison data.
%
%                  OriginalPrependStr (string vector) that will be prepended to the names of the original
%                                variables in the legend of the plot.
%
%                  EditedPrependStr (string vector) that will be prepended to the names of the edited
%                                variables in the legend of the plot.
%
%                  CompPrependStr (string vector) that will be prepended to the names of the comparison
%                                variables in the legend of the plot.
%
%                  MatchMaxPlots (boolean) that determines if the MaxPlots vector be matched exactly.
%
%                  MaxPlots (vector) of dimension 1x2 with the maximum number of row and columns graphs
%                                per figure.
%
%                  CurrINI (structure) with initialization information.
%
% REQUIRED OUTPUT: FigHandles (cell array) with handles to the figures.
%
% OPTIONAL OUTPUT: LastPosition (vector) with two values giving the last figure lower left corner position
%                                in pixels.
%
%                  UsedRowsCols (vector) of dimension 2 with the number of rows and columns used for the plots.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: November 8, 2007.
%                        This version: February 10, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 09-11-2007: Added the XLabel fix function.
%
% * 12-11-2007: Made sure that there is only one legend per figure. Added the "LastPosition" output and the
%               "FirstPosition" inputs.
%
% * 13-11-2007: Fixed a bug where the wrong data and names were used when more than one figure was created
%               ("k" was used instead of "NumPlots").
%
% * 09-01-2008: Added the grid lines for the x-axis to all plots.
%
% * 10-01-2008: Changed the last drawnow call to pause(0.02).
%
% * 11-01-2008: Added the input variable "OriginalDateVector" to allow the OriginalData and the EditedData
%               to have different lengths.
%
% * 15-01-2008: Added the input "MaxPlots" for the maximum number of plots per figure. The OriginalData matrix
%               can now also be empty.
%
% * 16-01-2008: The "MaxPlots" variable for the "PlotEditedData" function is now a 1x2 vector with the maximum
%               rows and columns for the subplot function. Made sure that the integer n is not zero.
%
% * 25-01-2008: Added the input variables "CompData", "CompLineStyle", "CompColor", and "CompPrependStr".
%
% * 28-01-2008: Added the input variable "VerticalLinePos".
%
% * 01-02-2008: The variable "EditedData" may now also be empty.
%
% * 17-03-2008: Allow CompData to have fewer columns than OriginalData.
%
% * 23-05-2008: Updated the documentation.
%
% * 31-07-2008: Added the "MatchMaxPlots" variable.
%
% * 27-08-2008: Made sure that only the first value of "VerticalLinePos" is used if it has more than one
%               element. Also, the "Children" property is only reset when the reordering has the correct
%               length.
%
% * 19-08-2008: Added the "UsedRowsCols" output variable.
%
% * 19-12-2008: Updated the documentation.
%
% * 10-02-2008: Made sure that CompData has enough data points if some appear to be missing. The
%               missing ones and then replaced by NaNs.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

n = size(OriginalData,1);
if isempty(OriginalData)==1;
   n = size(EditedData,1);
end;
if n==0;
   n = size(CompData,1);
end;
if isempty(CompData)==1;
   CompPrependStr = '';
else;
   if (size(CompData,2)<length(OriginalDateVector));
      CompData = [CompData NaN*ones(size(CompData,1),length(OriginalDateVector)-size(CompData,2))];
   elseif (size(CompData,2)>length(OriginalDateVector));
      CompData = [];
   end;
   if (size(CompData,1)~=n);
      %
      % the CompData does not have the right dimensions
      %
      CompData = [];
      CompPrependStr = '';
   end;
end;
%
% Compute the number of figures (max 16 plots per graph)
%
MaxPlots = floor(MaxPlots);
NumFigs = ceil(n/(MaxPlots(1)*MaxPlots(2)));
%
if (NumFigs==1)&(MatchMaxPlots==0);
   [NumRows,NumCols] = VillaniConstructOptimalSubplot(n);
else;
   NumRows = MaxPlots(1);
   NumCols = MaxPlots(2);
end;
NumPlots = 0;
if length(FirstPosition)~=2;
   FirstPosition = [(CurrINI.scrsz(3)-1000)/2 (CurrINI.scrsz(4)-650)/2];
end;
pos = [FirstPosition 1000 650];
if pos(1)+1000>CurrINI.scrsz(3);
   pos(1) = 0;
end;
if pos(2)<=32;
   pos(2) = CurrINI.scrsz(4)-700;
end;
CurrINI.xgrid = 1;
for i=1:NumFigs;
   if i>1;
      pos(1) = pos(1)+22;
      pos(2) = pos(2)-22;
      if pos(1)+1000>CurrINI.scrsz(3);
         pos(1) = 0;
      end;
      if pos(2)<=32;
         pos(2) = CurrINI.scrsz(4)-700;
      end;
   end;
   FigHandles{i} = figure('Visible','off','Position',pos);
   if NumFigs==1;
      set(FigHandles{i},'Name',NameStr);
   else;
      set(FigHandles{i},'Name',[NameStr ' - Figure ' num2str(i)]);
   end;
   SetFigureProperties(FigHandles{i},CurrINI);
   k = 0;
   %
   % start looping
   %
   for r=1:NumRows;
      for c=1:NumCols;
         NumPlots = NumPlots+1;
         if NumPlots<=n;
            k = k+1;
            subplot(NumRows,NumCols,k);
            %
            if isempty(CompData)==0;
               CompHandle = plot(OriginalDateVector(1:size(CompData,2)),CompData(NumPlots,:),CompLineStyle,'Color',CompColor,'LineWidth',2);
               hold('on');
            else;
               CompHandle = [];
            end;
            %
            if isempty(OriginalData)==0;
               OriginalHandle = plot(OriginalDateVector,OriginalData(NumPlots,:),OriginalLineStyle,'Color',OriginalColor,'LineWidth',2);
               hold('on');
            else;
               OriginalHandle = [];
            end;
            %
            if isempty(EditedData)==0;
               EditedHandle = plot(DateVector,EditedData(NumPlots,:),EditedLineStyle,'Color',EditedColor,'LineWidth',2);
               hold('on');
            else;
               EditedHandle = [];
            end;
            %
            ylim = get(gca,'YLim');
            AddOn = 0.1*(ylim(2)-ylim(1));
            ylim = [ylim(1)-AddOn ylim(2)+AddOn];
            %
            % check if we have more original data than edited data
            %
            if length(OriginalDateVector)>length(DateVector);
               DatePos = length(OriginalDateVector)-length(DateVector);
               DateValue = (OriginalDateVector(DatePos)+OriginalDateVector(DatePos+1))/2;
               VerticalLineHandle = plot(DateValue*ones(1,2),ylim,'-','Color',[0 0 0],'LineWidth',1);
               hold('on');
            elseif (length(OriginalDateVector)==length(DateVector))&(isempty(VerticalLinePos)==0);
               VerticalLineHandle = plot(VerticalLinePos(1)*ones(1,2),ylim,'-','Color',[0 0 0],'LineWidth',1);
               hold('on');
            else;
               VerticalLineHandle = [];
            end;
            %
            if (ylim(1)<0)&(ylim(2)>0);
               ZeroHandle = plot([min(DateVector(1),OriginalDateVector(1)) max(DateVector(length(DateVector)),OriginalDateVector(length(OriginalDateVector)))],[0 0],'-','Color',[0 0 0]);
               hold('on');
               %
               % make sure the zero line is below the other lines
               %
            else;
               ZeroHandle = [];
            end;
            if length(get(gca,'Children'))==length([EditedHandle OriginalHandle CompHandle VerticalLineHandle ZeroHandle]);
               set(gca,'Children',[EditedHandle OriginalHandle CompHandle VerticalLineHandle ZeroHandle]);
            end;
            set(gca,'YLim',ylim,'XLim',[min(DateVector(1),OriginalDateVector(1)) max(DateVector(length(DateVector)),OriginalDateVector(length(OriginalDateVector)))]);
            %
            % check the Xlabel to make sure only integers are displayed
            %
            SetXTickIntegers(gca);
            %
            % set the title
            %
            SetAxesFonts(CurrINI,'Title',StringTrim(DataNames(NumPlots,:)));
            %
            % now we need to add a legend
            %
            hold('off');
         end;
      end;
   end;
   %
   % create the legend
   %
   LegendHandle = legend([OriginalHandle EditedHandle CompHandle],strvcat(OriginalPrependStr,EditedPrependStr,CompPrependStr));
   SetAxesFonts(CurrINI,LegendHandle);
   %
   % transform units temporarily into pixels
   %
   OldAxesUnits = get(gca,'Units');
   OldLegUnits = get(LegendHandle,'Units');
   set(gca,'Units','pixels');
   set(LegendHandle,'Units','pixels');
   %
   % check if we can move legend
   %
   LegPos = get(LegendHandle,'Position');
   AxPos = get(gca,'Position');
   if LegPos(1)<AxPos(1)+AxPos(3);
      if AxPos(1)+AxPos(3)+10+LegPos(3)<1000;
         LegPos(1) = AxPos(1)+AxPos(3)+10;
         set(LegendHandle,'Position',LegPos);
      end;
   end;
   set(gca,'Units',OldAxesUnits);
   set(LegendHandle,'Units',OldLegUnits);
   %
   % Fix the toolbar cdata
   %
   SetToolbarImages(FigHandles{i},CurrINI.images);
   %
   % Set the current figure's CloseRequest function
   %
   set(FigHandles{i},'CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
end;
if nargout>=2;
   LastPosition = [pos(1) pos(2)];
end;
if nargout==3;
   UsedRowsCols = [NumRows NumCols];
end;

%
% end of PlotEditedData.m
%
