function [action,NewValues] = ParameterScenarioDLG(selector,ParameterValues,ParameterScenarioValues,ParameterBounds,ParameterNames,PType,CurrINI)
% ParameterScenarioDLG: Presents a dialog for changing the values of certain parameters in for parameter scenarios.
%
% USAGE:
%
%       [action,NewValues] = ParameterScenarioDLG(selector,ParameterValues,ParameterScenarioValues,ParameterBounds,ParameterNames,PType,CurrINI)
%
% REQUIRED INPUT:  selector (string vector) that takes on the possible values "init", "checkedit", "changeparameter", "reset", or "quit".
%
% OPTIONAL INPUT:  ParameterValues (vector) with the original values of the parameters. This option is required if selector is "init".
%
%                  ParameterScenarioValues (vector) with the alternative values of the parameters. This option is required if selector is "init".
%
%                  ParameterBounds (matrix) with upper and lower bounds for the parameters.
%
%                  ParameterNames (string matrix) with the names of the parameters. This option is required if selector is "init".
%
%                  PType (string vector) that takes on the values "Posterior Mode" or "Initial Values". This option is required if selector is "init".
%
%                  CurrINI (structure) with initialization information. This option is required if selector is "init".
%
% REQUIRED OUTPUT: action (string vector) that takes on the values "OK" or "Cancel".
%
%                  NewValues (vector) with the new alternative parameter values.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: October 26, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 30-10-2007: Updated the documenation.
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 19-3-2008:  Added a missing "set(controls.cancel,'UserData',SV);" command when edited data have
%               been accepted.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==1;
   SV.CurrINI = CurrINI;
   scrsz = get(0,'ScreenSize');
   SV.ParameterScenarioValues = ParameterScenarioValues;
   SV.OrigParameterScenarioValues = ParameterScenarioValues;
   SV.ParameterValues = ParameterValues;
   SV.ParameterBounds = ParameterBounds;
   %
   % try to determine the width and the height
   %
   width = 540;
   height = 130;
else;
   ParameterScenarioGUI = findobj('Type','figure','Tag','ParameterScenarioDLG');
   controls = get(ParameterScenarioGUI,'UserData');
   SV = get(controls.cancel,'UserData');
   CurrINI = SV.CurrINI;
end;

switch selector
   case 'init'
      ParameterScenarioGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','ParameterScenarioDLG.m', ...
         'MenuBar','none', ...
         'Units','pixels', ...
         'Position',[(scrsz(3)-width)/2 (scrsz(4)-height)/2 width height], ...
         'Tag','ParameterScenarioDLG', ...
         'Resize','off', ...
         'Visible','off', ...
         'Name',['Parameter Scenarios - ' PType], ...
         'NumberTitle','off', ...
         'WindowStyle','modal', ...
         'CloseRequestFcn','ParameterScenarioDLG quit', ...
         'ToolBar','none');
      %
      if MatlabNumber>=7.0;
         set(ParameterScenarioGUI,'DockControl','off');
      end;
      %
      AxesBox([2 2 width-2 height-10],'Parameter Scenario Specification',100,[0.5 1],'on',CurrINI);
      %
      controls.parameternametext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[20 85 140 20], ...
         'String','Parameter:', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','text', ...
         'Enable','on', ...
         'Tag','ParameterNameText');
      %
      controls.parameters = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[20 65 140 20], ...
         'String',ParameterNames, ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','popupmenu', ...
         'Callback','ParameterScenarioDLG changeparameter', ...
         'Value',1, ...
         'Enable','on', ...
         'Tag','ParameterNamePopup');
      %
      controls.oldparametertext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[170 85 160 20], ...
         'String','Original parameter value:', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','text', ...
         'Enable','on', ...
         'Tag','OldParameterText');
      %
      controls.oldparametervalue = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[170 65 160 20], ...
         'String',num2str(ParameterValues(1),'%0.12f'), ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','edit', ...
         'Enable','off', ...
         'Tag','OldParameterEdit');
      %
      controls.newparametertext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[340 85 160 20], ...
         'String','Parameter value for scenario:', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','text', ...
         'Enable','on', ...
         'Tag','NewParameterText');
      %
      controls.newparametervalue = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[340 65 160 20], ...
         'String',num2str(ParameterScenarioValues(1),'%0.12f'), ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','edit', ...
         'CallBack','ParameterScenarioDLG checkedit', ...
         'Enable','on', ...
         'Tag','NewParameterEdit');
      %
      % Two buttons to select action
      %
      controls.ok = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-255 20 75 25], ...
         'String','OK', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'Callback','ParameterScenarioDLG quit', ...
         'UserData','waiting', ...
         'Tag','ButtonOK');
      %
      controls.reset = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-175 20 75 25], ...
         'String','Reset', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'Callback','ParameterScenarioDLG reset', ...
         'Tag','ButtonReset');
      %
      controls.cancel = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-95 20 75 25], ...
         'String','Cancel', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'UserData',SV, ...
         'Callback','ParameterScenarioDLG quit', ...
         'Tag','ButtonCancel');
      %
      set(ParameterScenarioGUI,'UserData',controls);
   case 'changeparameter'
      val = get(controls.parameters,'Value');
      set(controls.oldparametervalue,'String',num2str(SV.ParameterValues(val),'%0.12f'));
      set(controls.newparametervalue,'String',num2str(SV.ParameterScenarioValues(val),'%0.12f'));
   case 'checkedit'
      val = get(controls.parameters,'Value');
      parvalue = str2num(get(controls.newparametervalue,'String'));
      if (isnumeric(parvalue)==1)&(isnan(parvalue)==0)&(isinf(parvalue)==0);
         %
         % compare the value with the bounds
         %
         AddStr = '';
         WithinBounds = 1;
         if SV.ParameterBounds(val,3)==0;
            if (parvalue<=SV.ParameterBounds(val,1))|(parvalue>=SV.ParameterBounds(val,2));
               WithinBounds = 0;
               AddStr = ' The lower and the upper bounds are both open.';
            end;
         elseif SV.ParameterBounds(val,3)==1;
            if (parvalue<SV.ParameterBounds(val,1))|(parvalue>=SV.ParameterBounds(val,2));
               WithinBounds = 0;
               AddStr = ' The lower bound is open, while the upper bound is closed.';
            end;
         elseif SV.ParameterBounds(val,3)==2;
            if (parvalue<SV.ParameterBounds(val,1))|(parvalue>SV.ParameterBounds(val,2));
               WithinBounds = 0;
               AddStr = ' The lower and the upper bound are both closed.';
            end;
         end;
         if WithinBounds==1;
            %
            % the new value is accepted
            %
            SV.ParameterScenarioValues(val) = parvalue;
            set(controls.newparametervalue,'String',num2str(parvalue,'%0.12f'));
            set(controls.cancel,'UserData',SV);
         else;
            set(controls.newparametervalue,'String',num2str(SV.ParameterScenarioValues(val),'%0.12f'));
            txt = ['The value you have selected "' num2str(parvalue) '" doesn''t belong to the support of its distribution. The lower bound of the support is given by "' num2str(SV.ParameterBounds(val,1),'%0.12f') '", while the upper bound is "' num2str(SV.ParameterBounds(val,2),'%0.12f') '".' AddStr];
            About(txt,'information',get(ParameterScenarioGUI,'Name'),300,500,CurrINI);
         end;
      else;
         set(controls.newparametervalue,'String',num2str(SV.ParameterScenarioValues(val),'%0.12f'));
         txt = ['You can only select finite numerical values for the parameters.'];
         About(txt,'information',get(ParameterScenarioGUI,'Name'),200,500,CurrINI);
      end;
   case 'reset'
      %
      % resets current value to the original
      %
      val = get(controls.parameters,'Value');
      SV.ParameterScenarioValues(val) = SV.OrigParameterScenarioValues(val);
      set(controls.newparametervalue,'String',num2str(SV.ParameterScenarioValues(val),'%0.12f'));
      set(controls.cancel,'UserData',SV);
   case 'quit'
      set(controls.ok,'UserData','done');
end;

if strcmp(selector,'init')==1;
   set(ParameterScenarioGUI,'Visible','on');
   drawnow;
   if MatlabNumber>=7;
      uicontrol(controls.ok);
   end;
   %
   waitfor(controls.ok,'UserData','done');
   if (isempty(get(ParameterScenarioGUI,'CurrentObject'))==1)|(strcmp(get(get(ParameterScenarioGUI,'CurrentObject'),'Type'),'uicontrol')==0);
      action = 'Cancel';
   else;
      action = get(get(ParameterScenarioGUI,'CurrentObject'),'String');
      if (isempty(action)==1)|(strcmp(action,'OK')==0);
         action = 'Cancel';
      end;
   end;
   SV = get(controls.cancel,'UserData');
   %
   % determine the remaining output
   %
   if strcmp(action,'OK')==1;
      %
      % choose new values
      %
      NewValues = SV.ParameterScenarioValues;
   else;
      NewValues = SV.OrigParameterScenarioValues;
   end;
   %
   delete(ParameterScenarioGUI);
   drawnow;
   pause(0.02);
end;

%
% end of ParameterScenarioDLG.m
%
