function MultistepForecastDLG(selector,DSGEModel,CurrINI,ForecastResults,NameStr,ModelStr,AnnualStr)
% MultistepForecastDLG: Displays multistep forecasts of the observed variables
%
% USAGE:
%
%       MultistepForecastDLG(selector,DSGEModel,CurrINI,ForecastResults,NameStr,ModelStr)
%
% REQUIRED INPUT: selector (string vector) taking on the values "init", "showgraph", and "done".
%
% OPTIONAL INPUT: DSGEModel (structure) whose fields contain all required information
%                           about the model, such as input files, variable and sample
%                           data. Actual observations are not stored in this file.
%
%                 CurrINI (structure), with initialization information.
%
%                 ForecastResults (structure) of dimension equal to the frequency of parameter re-estimation
%                           when forecasting or to the number of times we can compute 1-step ahead forecast
%                           errors (whichever is the smallest). The fields in this vector structure are
%                           given by the matrices "Yhat", the forecasts Y(T+h|T), "Y", the observed
%                           variables Y(T+h), "Error", the forecast errors, "RMSE", the theoretical root
%                           mean squared errors for Y(T+h), and "DateVector" containing the dates. The
%                           number of columns of the matrices and vectors are equal to the maximum possible
%                           desired forecast horizon.
%
%                 NameStr (string vector) taking on the values "Posterior Mode" or "Initial Values".
%
%                 ModelStr (string vector) taking on the values "DSGE" or "BVAR".
%
%                 AnnualStr (string vector) taking on the values "original" or "annual"
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: January 8, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 9-1-2007:   Finished the function.
%
% * 16-1-2007:  Added the input "ModelStr".
%
% * 22-1-2007:  Added the input "AnnualStr". Plots of annualized data are now also made possible.
%
% * 23-1-2007:  Changed the Name property string for the dialog.
%
% * 24-1-2007:  Fixed the XLim max-value in the forecast error subplot.
%
% * 25-1-2007:  Made sure that the "Annualize" function receives the relevant terms of the DSGEModel.annualscale
%               vector.
%
% * 13-11-2007: Updated the documentation.
%
% * 2-1-2008:   Removed code using "DSGEModel.InitializeKalman".
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==1;
   DSGEModel.ForecastResults = ForecastResults;
   DSGEModel.NameStr = NameStr;
   DSGEModel.ModelStr = ModelStr;
   DSGEModel.AnnualStr = AnnualStr;
else
   MultistepForecastGUI = findobj('Type','figure','Tag','MultistepForecastDLG');
   ForecastControls = get(MultistepForecastGUI,'UserData');
   CurrINI = get(ForecastControls.varnames,'UserData');
   DSGEModel = get(ForecastControls.variables,'UserData');
end;

%
% Set screen size directly instead of using
% get(0,'ScreenSize')!
%

switch selector;
   case 'init';
      MultistepForecastGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','MultistepForecastDLG.m', ...
         'MenuBar','none', ...
         'PaperUnits','points', ...
         'Units','pixels', ...
         'Position',[(CurrINI.scrsz(3)-500)/2 (CurrINI.scrsz(4)-120)/2 500 120], ...
         'Tag','MultistepForecastDLG', ...
         'Visible','off', ...
         'Resize','off', ...
         'Name',['Multistep Forecasts With ' ModelStr ' Model - ' NameStr], ...
         'NumberTitle','off', ...
         'CloseRequestFcn','MultistepForecastDLG done', ...
         'ToolBar','none');

      if MatlabNumber>=7.0;
         set(MultistepForecastGUI,'DockControl','off');
      end;
      %
      % Check wehere the sample of data actually begins
      %
      [NewStartYear,NewStartPeriod] = AdjustSampleStart(DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.DataFrequency,DSGEModel.KalmanFirstObservation-1);
      %
      % Outer axis
      %
      AxesBox([2 2 498 110],'Graphics',45,[0.5 1],'on',CurrINI);
      
      ForecastControls.sample = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 50 360 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'String',['Estimation Sample: ' NewStartYear ':' NewStartPeriod ' - ' DSGEModel.SubEndYear ':' DSGEModel.SubEndPeriod ' (Max: ' num2str(size(ForecastResults(1).Yhat,2),'%0.0f') ' periods)'], ...
         'HorizontalAlignment','left', ...
         'Tag','SampleText');
      %
      % Select variable
      %
      ForecastControls.variables = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 16 180 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'UserData',DSGEModel, ...
         'String','Select variable', ...
         'HorizontalAlignment','left', ...
         'Tag','ShockText');
      %
      ForecastControls.varnames = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[210 20 155 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',DSGEModel.VariableNames, ...
         'HorizontalAlignment','center', ...
         'Tag','ShcoksPopup',...
         'Value',1);
      %
      if strcmp(ModelStr,'BVAR')==1;
         set(ForecastControls.varnames,'String',DSGEModel.VariableNames(DSGEModel.BVARY,:));
      end;
      %
      % Create graph button
      %
      ForecastControls.show = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[395 55 90 20], ...
         'String','Display', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','MultistepForecastDLG showgraph', ...
         'Enable','on', ...
         'Tag','Done');
      %
      %  Done button
      %
      ForecastControls.done = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[395 20 90 20], ...
         'String','Done', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','MultistepForecastDLG done', ...
         'Enable','on', ...
         'UserData','waiting', ...
         'Tag','Done');

      set(MultistepForecastGUI,'UserData',ForecastControls,...
         'HandleVisibility','callback');
   case 'showgraph'
      VariableValue = get(ForecastControls.varnames,'Value');
      VariableStr = get(ForecastControls.varnames,'String');
      CurrVariable = StringTrim(VariableStr(VariableValue,:));
      if VariableValue<size(VariableStr,1);
         set(ForecastControls.varnames,'Value',VariableValue+1);
      else;
         set(ForecastControls.varnames,'Value',1);
      end;
      %
      % create a vector with dates
      %
      DateVector = MakeDateVector(DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.EndYear,DSGEModel.EndPeriod,DSGEModel.DataFrequency);
      [FirstPeriod,LastPeriod] = CreateSubSample(DSGEModel);
      %
      % create the figure
      %
      FigHandle =figure('Visible','off', ...
                'Position',[(CurrINI.scrsz(3)-750)/2 (CurrINI.scrsz(4)-500)/2 750 500], ...
                'Tag',[CurrVariable '-' DSGEModel.ModelStr], ...
                'Name',['Multistep Forecasts in ' DSGEModel.ModelStr ' Model - ' CurrVariable]);
      %
      SetFigureProperties(FigHandle,CurrINI);
      %
      % create a subplot
      %
      subplot(2,1,1);
      %
      % first we plot the observed variable
      %
      if strcmp(DSGEModel.AnnualStr,'original')==1;
         %
         % we're using the original data
         %
         if strcmp(DSGEModel.ModelStr,'DSGE')==1;
            YHandle = plot(DateVector,DSGEModel.Y(VariableValue,FirstPeriod:size(DSGEModel.Y,2)),'.-','Color',[0.75 0 0],'LineWidth',2);
         else;
            YHandle = plot(DateVector,DSGEModel.Y(DSGEModel.BVARY(VariableValue),FirstPeriod:size(DSGEModel.Y,2)),'.-','Color',[0.75 0 0],'LineWidth',2);
         end;
      else;
         %
         % we're using the annualized data
         %
         Y = Annualize(DSGEModel.Y(:,FirstPeriod:size(DSGEModel.Y,2)),DSGEModel.annual,DSGEModel.annualscale);
         if strcmp(DSGEModel.ModelStr,'DSGE')==1;
            YHandle = plot(DateVector(1,max(DSGEModel.annual):length(DateVector)),Y(VariableValue,:),'.-','Color',[0.75 0 0],'LineWidth',2);
         else;
            YHandle = plot(DateVector(1,max(DSGEModel.annual):length(DateVector)),Y(DSGEModel.BVARY(VariableValue),:),'.-','Color',[0.75 0 0],'LineWidth',2);
         end;
      end;
      hold('on');
      %
      if strcmp(DSGEModel.AnnualStr,'original')==1;
         %
         % we're using the original data
         %
         for i=1:length(DSGEModel.ForecastResults);
            Yhathandle = plot(DSGEModel.ForecastResults(i).DateVector,DSGEModel.ForecastResults(i).Yhat(VariableValue,:),'-','Color',(i/length(DSGEModel.ForecastResults))*[0 0 1],'LineWidth',2);
            hold('on');
         end;
      else;
         %
         % we're using the annualized data
         %
         for i=1:length(DSGEModel.ForecastResults);
            Yhathandle = plot(DSGEModel.ForecastResults(i).DateVector,DSGEModel.ForecastResults(i).AnnYhat(VariableValue,:),'-','Color',(i/length(DSGEModel.ForecastResults))*[0 0 1],'LineWidth',2);
            hold('on');
         end;
      end;
      %
      ylim = get(gca,'YLim');
      add_on = 0.1*(ylim(2)-ylim(1));
      ylim = [ylim(1)-add_on ylim(2)+add_on];
      if strcmp(DSGEModel.AnnualStr,'original')==1;
         set(gca,'YLim',ylim,'XLim',[min(DateVector) max(DateVector)]);
      else;
         set(gca,'YLim',ylim,'XLim',[DateVector(1,max(DSGEModel.annual)) max(DateVector)]);
      end;
      kids = get(gca,'Children');
      if size(kids,2)>size(kids,1);
         kids = kids';
      end;
      FirstPos = DSGEModel.ForecastResults(1).DateVector(1);
      LastPos = 0;
      for i=1:length(DateVector);
         if isequal(FirstPos,DateVector(i))==1;
            LastPos = i-1;
            break;
         end;
      end;
      %
      % add a vertical line from the forecast point
      %
      if LastPos>0;
         fp = plot(DateVector(LastPos)*[1 1],ylim,':','Color',[0 0 0]);
         hold('on');
      else;
         fp = [];
      end;
      %
      if (ylim(1)<0)&(ylim(2)>0);
         %
         % add a zero line
         %
         zl = plot([min(DateVector) max(DateVector)],[0 0],'-','Color',[0 0 0]);
      else;
         zl = [];
      end;
      %
      set(gca,'Children',[kids;zl;fp]);
      %
      SetAxesFonts(CurrINI,'Title',[CurrVariable ' (' DSGEModel.AnnualStr ' data) and out-of-sample forecast paths - ' lower(DSGEModel.NameStr) ' of the parameters']);
      hold('off');
      %
      % graph the forecast errors
      %
      subplot(2,1,2);
      if strcmp(DSGEModel.AnnualStr,'original')==1;
         %
         % we're using the original data
         %
         for i=1:length(DSGEModel.ForecastResults);
            Errhandle = plot(DSGEModel.ForecastResults(i).DateVector,DSGEModel.ForecastResults(i).Error(VariableValue,:),'-','Color',(i/length(DSGEModel.ForecastResults))*[0 0 1],'LineWidth',2);
            hold('on');
         end;
      else;
         %
         % we're using the annualized data
         %
         for i=1:length(DSGEModel.ForecastResults);
            Errhandle = plot(DSGEModel.ForecastResults(i).DateVector,DSGEModel.ForecastResults(i).AnnError(VariableValue,:),'-','Color',(i/length(DSGEModel.ForecastResults))*[0 0 1],'LineWidth',2);
            hold('on');
         end;
      end;
      ylim = get(gca,'YLim');
      add_on = 0.1*(ylim(2)-ylim(1));
      ylim = [ylim(1)-add_on ylim(2)+add_on];
      set(gca,'YLim',ylim,'XLim',[floor(min(DSGEModel.ForecastResults(1).DateVector)) max(DSGEModel.ForecastResults(length(DSGEModel.ForecastResults)).DateVector)]);
      %
      if (ylim(1)<0)&(ylim(2)>0);
         %
         % add a zero line
         %
         kids = get(gca,'Children');
         if size(kids,2)>size(kids,1);
            kids = kids';
         end;
         zl = plot([floor(min(DSGEModel.ForecastResults(1).DateVector)) max(DSGEModel.ForecastResults(length(DSGEModel.ForecastResults)).DateVector)],[0 0],'-','Color',[0 0 0]);
         set(gca,'Children',[kids;zl]);
      end;
      %
      % fix the X-ticks
      %
      xtick = get(gca,'XTick');
      xtix = (floor(min(xtick)):floor(max(xtick)));
      set(gca,'XTick',xtix);
      SetAxesFonts(CurrINI,'Title',['Out-of-sample forecast error paths - ' DSGEModel.AnnualStr ' data']);
      hold('off');
      %
      SetToolbarImages(FigHandle,CurrINI.images);
      set(FigHandle,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
      drawnow;
      pause(0.02);
   case 'done'
      set(ForecastControls.done,'UserData','done');
      delete(MultistepForecastGUI);
      drawnow;
      pause(0.02);
end;
%
if strcmp(selector,'init')==1;
   set(MultistepForecastGUI,'Visible','on');
   drawnow;
   if MatlabNumber>=7;
      uicontrol(ForecastControls.done);
   end;
   waitfor(ForecastControls.done,'UserData','done');
end;

%
% end of MultistepForecastDLG.m
%
