function DataStructure = MakeToolbar(DataStructure,ParentWindow,ToolbarFieldNames,ButtonData,Buttons)
% MakeToolbar: Creates a toolbar with push and toggle buttons.
%
% USAGE: DataStructure = MakeToolbar(DataStructure,ParentWindow,ToolbarFieldNames,ButtonData,Buttons)
%
% REQUIRED INPUT: DataStructure (structure) with handles which may be empty.
%
%                 ParentWindow (handle) to the parent window.
%
%                 ToolbarFieldNames (cell array) with string vectors of names for the
%                          buttons.  The length of the array is number of buttons
%                          plus one (for the toolbar, the first entry).
%
%                 ButtonData (structure) with 2 fields: "cdata" and "userdata".
%                          The "cdata" field is used to set an icon for the button.
%                          The "userdata" field sets a "UserData" property value
%                          for the button.
%
%                 Buttons (structure) with the following fields and entries:
%                          type:          (cell array) with text string push or toggle.
%                          tooltipstr:    (cell array) with strings for tooltips.
%                          callbackstr:   (cell array) with callback strings.
%                          callbackstr:   (cell array) with callback strings.
%                          separatorstr:  (cell array) with on/off string entries.
%                          statestr:      (cell array) with on/off entries (only used for
%                                         toggle buttons).
%                          tagstr:        (cell array) with tag strings.
%
% REQUIRED OUTPUT: DataStructure (structure) a structure containing all handles of the created
%                          objects+any handles passed to the function.
%
%
% OPTIONAL OUTPUT: StatusHandle (handle) to the text control of the statusbar.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: July 21, 2006.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 19-5-2008:  Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize error data
%
origStruct = DataStructure;
%
% check if ParentWindow is valid
%
if (ishandle(ParentWindow)==0)|(strcmp(get(ParentWindow,'Type'),'figure')==0);
   ParentWindow = get(0,'CurrentWindow');
end;
%
% set up a toolbar
%
ToolbarHandle = uitoolbar('Parent',ParentWindow, ...
         'HandleVisibility','off', ...
         'Interruptible','on', ...
         'Clipping','on');
%
% check if we should initialize the structure for output.
%
if (isempty(DataStructure)==1)|(isstruct(DataStructure)==0);
   DataStructure = struct(ToolbarFieldNames{1},ToolbarHandle);
else;
   DataStructure = setfield(DataStructure,ToolbarFieldNames{1},ToolbarHandle);
end;
%
% try to set up all the buttons on the toolbar.
%
for i=1:length(Buttons.type);
   if strcmp(Buttons.type{i},'push')==1;
      %
      % create push button
      %
      ButtonHandle = uipushtool('Parent',ToolbarHandle, ...
                         'TooltipString',Buttons.tooltipstr{i}, ...
                         'ClickedCallback',Buttons.callbackstr{i}, ...
                         'CData',ButtonData(i).cdata, ...
                         'Separator',Buttons.separatorstr{i}, ...
                         'Interruptible','on', ...
                         'UserData',ButtonData(i).userdata, ...
                         'Clipping','on', ...
                         'Enable',Buttons.enablestr{i}, ...
                         'Tag',Buttons.tagstr{i});
   elseif strcmp(Buttons.type{i},'toggle')==1;
      %
      % create toggle button
      %
      ButtonHandle = uitoggletool('Parent',ToolbarHandle, ...
                         'TooltipString',Buttons.tooltipstr{i}, ...
                         'ClickedCallback',Buttons.callbackstr{i}, ...
                         'CData',ButtonData(i).cdata, ...
                         'Separator',Buttons.separatorstr{i}, ...
                         'State',Buttons.statestr{i}, ...
                         'Interruptible','on', ...
                         'UserData',ButtonData(i).userdata, ...
                         'Clipping','on', ...
                         'Enable',Buttons.enablestr{i}, ...
                         'Tag',Buttons.tagstr{i});
   else;
      DataStructure = origStruct;
      return;
   end;
   DataStructure = setfield(DataStructure,ToolbarFieldNames{1+i},ButtonHandle);
end;

%
% end of MakeToolbar.m
%
