function IteratedParamsGraphDLG(selector,CurrINI,ParameterNames,thetaEstimates)
% IteratedParamsGraphDLG: Displays estimates of the original theta parameters throughout the optimization
%                         iterations.
%
% USAGE:
%
%       IteratedParamsGraphDLG(selector,CurrINI,ParameterNames,thetaEstimates)
%
% REQUIRED INPUT: selector (string vector) taking on the values "init", "showgraph", and "done".
%
% OPTIONAL INPUT: CurrINI (structure), with initialization information.
%
%                 ParameterNames (string matrix) whose rows give the names of the estimated parameters
%
%                 thetaEstimates (pxiter matrix) with the p parameters in the rows and the iter
%                                iterations in the columns.
%
% NOTE: When selector = "init", then all optional input arguments are required!
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: November 24, 2006.
%                        Thsi version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 9-11-2007:  Updated the documentation.
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==1;
   CurrINI.thetaEstimates = thetaEstimates;
else
   IPGraphGUI = findobj('Type','figure','Tag','IPGraphDlg');
   IPcontrol = get(IPGraphGUI,'UserData');
   CurrINI = get(IPcontrol.parameternames,'UserData');
end;

%
% Set screen size directly instead of using
% get(0,'ScreenSize')!
%

switch selector;
   case 'init';
      IPGraphGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','IteratedParamsGraphDLG.m', ...
         'MenuBar','none', ...
         'PaperUnits','points', ...
         'Units','pixels', ...
         'Position',[(CurrINI.scrsz(3)-400)/2 (CurrINI.scrsz(4)-120)/2 400 120], ...
         'Tag','IPGraphDlg', ...
         'Visible','off', ...
         'Resize','off', ...
         'Name','Graph Iterated Parameter Estimates', ...
         'NumberTitle','off', ...
         'CloseRequestFcn','IteratedParamsGraphDLG done', ...
         'ToolBar','none');

      if MatlabNumber>=7.0;
         set(IPGraphGUI,'DockControl','off');
      end;

      %
      % Outer axis
      %

      AxesBox([2 2 398 110],'Graphics',45,[0.5 1],'on',CurrINI);

      IPcontrol.sample = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 50 260 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'String',['Iterated parameter estimates from iteration 0 to ' num2str(size(thetaEstimates,2)-1,'%0.0f')], ...
         'HorizontalAlignment','left', ...
         'Tag','SampleText');

      %
      % Select parameter
      %
      IPcontrol.variables = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 16 126 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'String','Select parameter:', ...
         'HorizontalAlignment','left', ...
         'Tag','ParameterText');
      %
      IPcontrol.parameternames = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[150 20 115 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',ParameterNames, ...
         'HorizontalAlignment','center', ...
         'Tag','ShcoksPopup',...
         'Value',1);
      %
      % Create graph button
      %
      IPcontrol.show = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[295 55 90 20], ...
         'String','Display', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','IteratedParamsGraphDLG showgraph', ...
         'Enable','on', ...
         'Tag','Done');
      %
      %  Done button
      %
      IPcontrol.done = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[295 20 90 20], ...
         'String','Done', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','IteratedParamsGraphDLG done', ...
         'Enable','on', ...
         'UserData','waiting', ...
         'Tag','Done');

      set(IPGraphGUI,'UserData',IPcontrol,...
         'HandleVisibility','callback');
   case 'showgraph'
      ParamValue = get(IPcontrol.parameternames,'Value');
      ParamStr = get(IPcontrol.parameternames,'String');
      CurrParam = StringTrim(ParamStr(ParamValue,:));
      if ParamValue<size(ParamStr,1);
         set(IPcontrol.parameternames,'Value',ParamValue+1);
      else;
         set(IPcontrol.parameternames,'Value',1);
      end;
      %
      thetaEstimate = CurrINI.thetaEstimates(ParamValue,:);
      NumIter = (0:1:length(thetaEstimate)-1);
      %
      % prepare the figure
      %
      FigHandle =figure('Visible','off', ...
                'Position',[(CurrINI.scrsz(3)-650)/2 (CurrINI.scrsz(4)-500)/2 650 500], ...
                'Tag',CurrParam, ...
                'Name',['Iterated Parameter Estimate - ' CurrParam]);
      %
      SetFigureProperties(FigHandle,CurrINI);
      %
      %
      %
      LineHandle = plot(NumIter,thetaEstimate,'.-','Color',[0.75 0 0],'LineWidth',2);
      SetAxesFonts(CurrINI,'Title',CurrParam,'XLabel','Iteration');
      axis('tight');
      %
      % check the Y-axis
      %
      ylim = get(gca,'YLim');
      add_on = 0.1*(ylim(2)-ylim(1));
      ylim = [ylim(1)-add_on ylim(2)+add_on];
      set(gca,'YLim',ylim);
      if (ylim(1)<0)&(ylim(2)>0);
         hold('on');
         plot([NumIter(1) NumIter(length(NumIter))],[0 0],'-','Color',[0 0 0]);
         %
         % make sure the zero line is displayed below the observed variable
         %
         kids = get(gca,'Children');
         kids = [kids(2) kids(1)]';
         set(gca,'Children',kids);
      end;
      %
      % display the graph
      %
      SetToolbarImages(FigHandle,CurrINI.images);
      set(FigHandle,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
      drawnow;
      pause(0.02);
   case 'done'
      set(IPcontrol.done,'UserData','done');
      delete(IPGraphGUI);
      drawnow;
      pause(0.02);
end;
%
if strcmp(selector,'init')==1;
   set(IPGraphGUI,'Visible','on');
   drawnow;
   if MatlabNumber>=7;
      uicontrol(IPcontrol.done);
   end;
   waitfor(IPcontrol.done,'UserData','done');
end;

%
% end of IteratedParamsGraphDLG.m
%
