function [action,InitialStateValues] = InitializeStateVarsDLG(selector,AllVars,InitialStateValues,CurrINI)
% InitializeStateVarsDLG: Dialog for setting initial values for potential state variables.
%
% USAGE:
%
%       [action,InitialStateValues] = InitializeStateVarsDLG(selector,AllVars,InitialStateValues,CurrINI)
%
% REQUIRED INPUT:  selector (string vector), with the possible values "init", "checkvalue", "reset",
%                          "changevariable", and "quit"
%
% OPTIONAL INPUT:  AllVars (string matrix), with all the possible state variable and state shock names. Given
%                          by the AiM output.
%
%                  InitialStateValues (vector), with the currently suggested values for the state variables.
%
%                  CurrINI (structure), with initialization information.
%
% Optional OUTPUT: action (string vector) taking the value "OK" or "Cancel". Depends on which button was
%                         clicked on.
%
%                  InitialStateValues (vector), with the revised values for the state variables.
%
%
% NOTE: If selector = "init", then  all optional inputs and outputs are required.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: November 20, 2006.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 22-11-2006: Updated the documentation.
%
% * 23-11-2006: Check if current object is empty after waitfor function call.
%
% * 9-11-2007:  Updated the documentation.
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==1;
   SV.AllVars = AllVars;
   SV.InitialStateValues = InitialStateValues;
   SV.CurrINI = CurrINI;
   scrsz = get(0,'ScreenSize');
   %
   % try to determine the width and the height
   %
   width = 500;
   height = 130;
else;
   InitializeStateVarsGUI = findobj('Type','figure','Tag','InitializeStateVarsDLG');
   controls = get(InitializeStateVarsGUI,'UserData');
   SV = get(controls.cancel,'UserData');
   CurrINI = SV.CurrINI;
end;

switch selector
   case 'init'
      InitializeStateVarsGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','InitializeStateVarsDLG.m', ...
         'MenuBar','none', ...
         'Units','pixels', ...
         'Position',[(scrsz(3)-width)/2 (scrsz(4)-height)/2 width height], ...
         'Tag','InitializeStateVarsDLG', ...
         'Resize','off', ...
         'Visible','off', ...
         'Name','Set Initial State Values', ...
         'NumberTitle','off', ...
         'WindowStyle','modal', ...
         'CloseRequestFcn','InitializeStateVarsDLG quit', ...
         'ToolBar','none');
      %
      if MatlabNumber>=7.0;
         set(InitializeStateVarsGUI,'DockControl','off');
      end;
      %
      AxesBox([2 2 width-2 height-10],'Initial State Values Selection',100,[0.5 1],'on',CurrINI);
      %
      controls.variablenametext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[20 60 75 20], ...
         'String','Variable:', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','text', ...
         'Enable','on', ...
         'Tag','VariableNameText');
      %
      controls.variables = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[100 65 (width/2)-105 20], ...
         'String',SV.AllVars, ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','popupmenu', ...
         'Callback','InitializeStateVarsDLG changevariable', ...
         'Value',1, ...
         'Enable','on', ...
         'Tag','VariableNamePopup');
      %
      controls.valuetext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[(width/2)+5 60 75 20], ...
         'String','Value:', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','text', ...
         'Enable','on', ...
         'Tag','ValueText');
      %
%      disp([(width/2)+85 65 (width/2)-105 20])
      controls.value = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[(width/2)+85 65 (width/2)-105 20], ...
         'String',num2str(InitialStateValues(1),'%0.12f'), ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'HorizontalAlignment','left', ...
         'Style','edit', ...
         'UserData',InitialStateValues, ...
         'Callback','InitializeStateVarsDLG checkvalue', ...
         'Enable','on', ...
         'Tag','valueEdit');
      %
      % Two buttons to select action
      %
      controls.ok = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-255 20 75 25], ...
         'String','OK', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'Callback','InitializeStateVarsDLG quit', ...
         'UserData','waiting', ...
         'Tag','ButtonOK');
      %
      controls.reset = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-175 20 75 25], ...
         'String','Reset', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'Callback','InitializeStateVarsDLG reset', ...
         'Tag','ButtonReset');
      %
      controls.cancel = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[width-95 20 75 25], ...
         'String','Cancel', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Enable','on', ...
         'UserData',SV, ...
         'Callback','InitializeStateVarsDLG quit', ...
         'Tag','ButtonCancel');
      %
      set(InitializeStateVarsGUI,'UserData',controls);
   case 'changevariable'
      %
      % when the variable name changes we need to change the value displayed in the edit box
      %
      InitialStateValues = get(controls.value,'UserData');
      val = get(controls.variables,'Value');
      set(controls.value,'String',num2str(InitialStateValues(val),'%0.12f'));
   case 'checkvalue'
      %
      % in case the height of the dialog is inadequate to show all controls.
      % This requires more than 200 variables in the AiM model!
      %
      newval = StringTrim(get(controls.value,'String'));
      newvalue = str2num(newval);
      InitialStateValues = get(controls.value,'UserData');
      if (isempty(newvalue)==0)&(isnumeric(newvalue)==1)&(isreal(newvalue)==1);
         val = get(controls.variables,'Value');
         InitialStateValues(val) = newvalue;
         set(controls.value,'UserData',InitialStateValues);
      else;
         val = get(controls.variables,'Value');
         set(controls.value,'String',num2str(InitialStateValues(val),'%0.12f'));
         txt = ['The initial value you selected (' newval ') for ' StringTrim(SV.AllVars(val,:)) ' is not valid. Only real numbers can be used!'];
         About(txt,'information','YADA - Set Initial State Values',160,500,CurrINI);
      end;
   case 'reset'
      %
      % resets all values to the original
      %
      val = get(controls.variables,'Value');
      set(controls.value,'UserData',SV.InitialStateValues,'String',num2str(SV.InitialStateValues(val),'%0.12f'));
   case 'quit'
      set(controls.ok,'UserData','done');
end;

if strcmp(selector,'init')==1;
   set(InitializeStateVarsGUI,'Visible','on');
   drawnow;
   if MatlabNumber>=7;
      uicontrol(controls.ok);
   end;
   %
   waitfor(controls.ok,'UserData','done');
   if (isempty(get(InitializeStateVarsGUI,'CurrentObject'))==1)|(strcmp(get(get(InitializeStateVarsGUI,'CurrentObject'),'Type'),'uicontrol')==0);
      action = 'Cancel';
   else;
      action = get(get(InitializeStateVarsGUI,'CurrentObject'),'String');
      if (isempty(action)==1)|(strcmp(action,'OK')==0);
         action = 'Cancel';
      end;
   end;
   SV = get(controls.cancel,'UserData');
   %
   % determine output
   %
   if strcmp(action,'OK')==1;
      %
      % choose new values
      %
      InitialStateValues = get(controls.value,'UserData');
   else;
      InitialStateValues = SV.InitialStateValues;
   end;
   %
   delete(InitializeStateVarsGUI);
   drawnow;
   pause(0.02);
end;

%
% end of InitializeStateVarsDLG.m
%
