function HistDecompDLG(selector,CurrINI,DSGEModel,HistDecompStructure,NameStr)
% HistDecompDLG: Displays historical forecast error decompositions.
%
% USAGE:
%
%       HistDecompDLG(selector,CurrINI,DSGEModel,HistDecompStructure,NameStr)
%
% REQUIRED INPUT: selector (string vector) taking on the values "init", "showgraph", and "done".
%
% OPTIONAL INPUT: CurrINI (structure), with initialization information.
%
%                 DSGEModel (structure) whose fields contain all required information
%                           about the model, such as input files, variable and sample
%                           data. Actual observations are not stored in this file.
%
%                 HistDecompStructure (structure), with the fields "epstth", an nx(T-h) matrix with
%                                   h-steps ahead forecast errors, "wthT", an nx(T-h) or empty matrix
%                                   with the smoothed measurement errors, "KsiErr", an nx(T-h) matrix
%                                   with the state projection error influence, "etathT", and nx(T-h)Xq
%                                   matrix with structural error terms, and "KeepVar" a vector with
%                                   indices that gives the non-zero columns of B0.
%
%                 NameStr (string vector) taking on the values "Posterior Mode", "Initial Values"
%                           or "Distribution".
%
% NOTE: When selector = "init", then all optional input arguments are required!
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: November 15, 2006.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 16-11-2006: Finished the function.
%
% * 4-1-2007:   Changed the width of the display and the window name.
%
% * 26-4-2007:  Updated the documentation.
%
% * 9-11-2007:  Added the call to "SetXTickIntegers".
%
% * 2-1-2008:   Removed code using "DSGEModel.InitializeKalman".
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 15-1-2008:  Added the code for selecting the sample to display.
%
% * 7-5-2008:   Fixed a bug where the wrong shocks were selected when multiple
%               figures are displayed. The q-variable is now initialized outside
%               the figure loop.
%
% * 8-5-2008:   Added code that make it feasible to display results based on posterior
%               distributions.
%
% * 13-5-2008:  Added the final code for using the posterior distributions. The option to
%               show the legend was also added.
%
% * 23-5-2008:  Updated the documentation.
%
% * 11-7-2008:  Changed the ordering of colors for confidence bands.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==1;
   DSGEModel.HistDecompStructure = HistDecompStructure;
   DSGEModel.NameStr = NameStr;
   if strcmp(lower(NameStr),'distribution')==1;
      AddHeight = 30;
   else;
      AddHeight = 0;
   end;
else
   HistGraphGUI = findobj('Type','figure','Tag','HistDecompDLG');
   HistControls = get(HistGraphGUI,'UserData');
   CurrINI = get(HistControls.varnames,'UserData');
   DSGEModel = get(HistControls.variables,'UserData');
end;

%
% Set screen size directly instead of using
% get(0,'ScreenSize')!
%

switch selector;
   case 'init';
      HistGraphGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','HistDecompDLG.m', ...
         'MenuBar','none', ...
         'PaperUnits','points', ...
         'Units','pixels', ...
         'Position',[(CurrINI.scrsz(3)-500)/2 (CurrINI.scrsz(4)-120-AddHeight)/2 500 120+AddHeight], ...
         'Tag','HistDecompDLG', ...
         'Visible','off', ...
         'Resize','off', ...
         'Name',['Forecast Error Decompositions - ' NameStr], ...
         'NumberTitle','off', ...
         'CloseRequestFcn','HistDecompDLG done', ...
         'ToolBar','none');

      if MatlabNumber>=7.0;
         set(HistGraphGUI,'DockControl','off');
      end;
      %
      % Check wehere the sample of data actually begins
      %
      [NewStartYear,NewStartPeriod] = AdjustSampleStart(DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.DataFrequency,DSGEModel.KalmanFirstObservation-1);
      %
      OldDSGEModel = DSGEModel;
      DSGEModel.SubBeginYear = NewStartYear;
      DSGEModel.SubBeginPeriod = NewStartPeriod;
      %
      [FirstPeriod,LastPeriod,SampleStrMat] = CreateSubSample(DSGEModel);
      %
      if size(SampleStrMat,1)>length(DSGEModel.HistDecompStructure.epstth);
         SampleStrMat = SampleStrMat(size(SampleStrMat,1)-length(DSGEModel.HistDecompStructure.epstth)+1:size(SampleStrMat,1),:);
      end;
      %
      DSGEModel = OldDSGEModel;
      %
      % Outer axis
      %
      AxesBox([2 2 498 110+AddHeight],['Graphics - ' num2str(DSGEModel.HistDecompStructure.ForecastHorizon,'%0.0f') '-' lower(DSGEModel.HistDecompStructure.FreqStr) ' ahead forecasts'],200,[0.5 1],'on',CurrINI);
      %
      % Sample selection controls
      %
      HistControls.sampletxt = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 50+AddHeight 360 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'String','Selected Sample:', ...
         'HorizontalAlignment','left', ...
         'Tag','SampleText');
      %
      % sample start and sample end
      %
      HistControls.samplebeg = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[185 55+AddHeight 87 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'String',SampleStrMat, ...
         'HorizontalAlignment','center', ...
         'Callback','HistDecompDLG samplebegin', ...
         'Tag','SampleBeginPopup',...
         'Value',1);
      %
      HistControls.sampleend = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[278 55+AddHeight 87 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'String',SampleStrMat, ...
         'HorizontalAlignment','center', ...
         'Callback','HistDecompDLG sampleend', ...
         'Tag','SampleEndPopup',...
         'Value',size(SampleStrMat,1));
      %
%      HistControls.sample = uicontrol('Units','pixels', ...
%         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
%         'Position',[24 50 360 20], ...
%         'FontSize',CurrINI.gui_fontsize, ...
%         'FontName',CurrINI.gui_fontname, ...
%         'FontWeight',CurrINI.gui_fontweight, ...
%         'FontAngle',CurrINI.gui_fontangle, ...
%         'Style','text', ...
%         'String',['Estimation Sample: ' NewStartYear ':' NewStartPeriod ' - ' DSGEModel.SubEndYear ':' DSGEModel.SubEndPeriod ' (' num2str(DSGEModel.HistDecompStructure.ForecastHorizon,'%0.0f') '-' lower(DSGEModel.HistDecompStructure.FreqStr) ' ahead forecasts)'], ...
%         'HorizontalAlignment','left', ...
%         'Tag','SampleText');
      %
      % Select variable
      %
      HistControls.variables = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 20+AddHeight 160 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'UserData',DSGEModel, ...
         'String','Select variable', ...
         'HorizontalAlignment','left', ...
         'Tag','ShockText');
      %
      HistControls.varnames = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[185 25+AddHeight 180 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',DSGEModel.VariableNames, ...
         'HorizontalAlignment','center', ...
         'Tag','ShcoksPopup',...
         'Value',1);
      %
      %
      if AddHeight>0;
         HistControls.legendcheck = uicontrol('Units','pixels', ...
            'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
            'Position',[24 20 160 20], ...
            'FontSize',CurrINI.gui_fontsize, ...
            'FontName',CurrINI.gui_fontname, ...
            'FontWeight',CurrINI.gui_fontweight, ...
            'FontAngle',CurrINI.gui_fontangle, ...
            'Style','checkbox', ...
            'String','Show legend', ...
            'HorizontalAlignment','left', ...
            'Value',0, ...
            'Tag','ShowLegendCheckbox');
      else;
         HistControls.legendcheck = [];
      end;
      %
      % Create graph button
      %
      HistControls.show = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[395 55 90 20], ...
         'String','Display', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','HistDecompDLG showgraph', ...
         'Enable','on', ...
         'Tag','Done');
      %
      %  Done button
      %
      HistControls.done = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[395 20 90 20], ...
         'String','Done', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','HistDecompDLG done', ...
         'Enable','on', ...
         'UserData','waiting', ...
         'Tag','Done');

      set(HistGraphGUI,'UserData',HistControls,...
         'HandleVisibility','callback');
   case 'samplebegin'
      %
      % check to make sure that the sample begins before it ends!
      %
      samplebeg = get(HistControls.samplebeg,'Value');
      sampleend = get(HistControls.sampleend,'Value');
      if sampleend<samplebeg;
         set(HistControls.sampleend,'Value',samplebeg);
      end;
   case 'sampleend'
      %
      % check to make sure that the sample begins before it ends
      %
      samplebeg = get(HistControls.samplebeg,'Value');
      sampleend = get(HistControls.sampleend,'Value');
      if sampleend<samplebeg;
         set(HistControls.samplebeg,'Value',sampleend);
      end;
   case 'showgraph'
      VariableValue = get(HistControls.varnames,'Value');
      CurrVariable = StringTrim(DSGEModel.VariableNames(VariableValue,:));
      if VariableValue<size(DSGEModel.VariableNames,1);
         set(HistControls.varnames,'Value',VariableValue+1);
      else;
         set(HistControls.varnames,'Value',1);
      end;
      %
      % determine how many plots we should make
      %
      NumPlots = 1+length(DSGEModel.HistDecompStructure.KeepVar);
      PlotMeasureError = 0;
      if rank(DSGEModel.HistDecompStructure.R)>0;
         if DSGEModel.HistDecompStructure.R(VariableValue,VariableValue)>0;
            NumPlots = NumPlots+1;
            PlotMeasureError = 1;
         end;
      end;
      if isfield(DSGEModel.HistDecompStructure,'Quantiles')==1;
         NumPlots = NumPlots+1;
         AddPlots = 1;
      else;
         AddPlots = 0;
      end;
      %
      ShockNames = DSGEModel.StateShockNames(DSGEModel.HistDecompStructure.KeepVar,:);
      NumFigs = ceil(NumPlots/16);
      if NumFigs==1;
         [NumRows,NumCols] = VillaniConstructOptimalSubplot(NumPlots);
      else;
         NumRows = 4;
         NumCols = 4;
      end;
      %
      % Create the date vector
      %
      DateVector = MakeDateVector(DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.SubEndYear,DSGEModel.SubEndPeriod,DSGEModel.DataFrequency);
      %
      if length(DateVector)>length(DSGEModel.HistDecompStructure.epstth);
         DateVector = DateVector(1,length(DateVector)-length(DSGEModel.HistDecompStructure.epstth)+1:length(DateVector));
      end;
      %
      StartSample = get(HistControls.samplebeg,'Value');
      EndSample = get(HistControls.sampleend,'Value');
      %
      % We're ready to start the plots
      %
      NumPlotsMade = 0;
      pos = [(CurrINI.scrsz(3)-800)/2 (CurrINI.scrsz(4)-600)/2 800 600];
      q = 1;
      for i=1:NumFigs;
         if i>1;
            pos(1) = pos(1)+22;
            pos(2) = pos(2)-22;
            if pos(1)+pos(3)>CurrINI.scrsz(3);
               pos(1) = 0;
            end;
            if pos(2)+pos(4)>CurrINI.scrsz(4)-80;
               pos(2) = CurrINI.scrsz(4)-80-pos(4);
            end;
         end;
         FigHandles{i} = figure('Visible','off','Position',pos);
         if NumFigs==1;
            set(FigHandles{i},'Name',[num2str(DSGEModel.HistDecompStructure.ForecastHorizon,'%0.0f') '-' DSGEModel.HistDecompStructure.FreqStr ' Ahead Historical Decomposition - ' CurrVariable]);
         else;
            set(FigHandles{i},'Name',[num2str(DSGEModel.HistDecompStructure.ForecastHorizon,'%0.0f') '-' DSGEModel.HistDecompStructure.FreqStr ' Ahead Historical Decomposition for ' CurrVariable ' - Figure ' num2str(i)]);
         end;
         SetFigureProperties(FigHandles{i},CurrINI);
         %
         k = 0;
         for r=1:NumRows;
            for c=1:NumCols;
               NumPlotsMade = NumPlotsMade+1;
               if NumPlotsMade<=NumPlots;
                  k = k+1;
                  subplot(NumRows,NumCols,k);
                  Handles = [];
                  LegendStr = '';
                  if (AddPlots==1)&(NumPlotsMade==1);
                     PartErrHandle = [];
                     PartErrStr = [int2str(DSGEModel.HistDecompStructure.ForecastHorizon) '-' lower(DSGEModel.HistDecompStructure.FreqStr)  ' ahead forecast '];
                     %
                     % create the confidence bounds
                     %
                     NumBounds = floor(length(DSGEModel.HistDecompStructure.Quantiles)/2);
                     for nb=1:NumBounds;
                        Lower = DSGEModel.HistDecompStructure.Quantiles(nb).epstth(VariableValue,StartSample:EndSample);
                        Upper = DSGEModel.HistDecompStructure.Quantiles(length(DSGEModel.Percentiles)-nb+1).epstth(VariableValue,StartSample:EndSample);
                        PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
                        ConfBound = DSGEModel.HistDecompStructure.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.HistDecompStructure.Quantiles(nb).percent;
                        ConfPatch = patch([DateVector(StartSample:EndSample) fliplr(DateVector(StartSample:EndSample))],[Upper fliplr(Lower)],PatchColor);
                        set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
                        hold('on');
                        Handles = [ConfPatch Handles];
                        LegendStr = strvcat([num2str(ConfBound,'%0.0f') ' confidence'],LegendStr);
                     end;
                     set(gca,'Layer','Top');
                  end;
                  %
                  if NumPlotsMade==1+AddPlots;
                     %
                     % check if we should add error bands
                     %
                     if AddPlots==1;
                        for nb=1:NumBounds;
                           Lower = DSGEModel.HistDecompStructure.Quantiles(nb).KsiErr(VariableValue,StartSample:EndSample);
                           Upper = DSGEModel.HistDecompStructure.Quantiles(length(DSGEModel.Percentiles)-nb+1).KsiErr(VariableValue,StartSample:EndSample);
                           PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
                           ConfBound = DSGEModel.HistDecompStructure.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.HistDecompStructure.Quantiles(nb).percent;
                           ConfPatch = patch([DateVector(StartSample:EndSample) fliplr(DateVector(StartSample:EndSample))],[Upper fliplr(Lower)],PatchColor);
                           set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
                           hold('on');
                           Handles = [ConfPatch Handles];
                           LegendStr = strvcat([num2str(ConfBound,'%0.0f') ' confidence'],LegendStr);
                        end;
                        set(gca,'Layer','Top');
                     end;
                     %
                     % plot state prediction error
                     %
                     PartErrHandle = plot(DateVector(StartSample:EndSample),DSGEModel.HistDecompStructure.KsiErr(VariableValue,StartSample:EndSample),'--','Color',[0 0 0.75],'LineWidth',2);
                     hold('on');
                     PartErrStr = 'State error and';
                  elseif ((NumPlotsMade<NumPlots)&(NumPlotsMade>1+AddPlots))|((NumPlotsMade==NumPlots)&(PlotMeasureError==0));
                     %
                     % check if we should add error bands
                     %
                     if AddPlots==1;
                        for nb=1:NumBounds;
                           Lower = DSGEModel.HistDecompStructure.Quantiles(nb).etathT(VariableValue,StartSample:EndSample,q);
                           Upper = DSGEModel.HistDecompStructure.Quantiles(length(DSGEModel.Percentiles)-nb+1).etathT(VariableValue,StartSample:EndSample,q);
                           PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
                           ConfBound = DSGEModel.HistDecompStructure.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.HistDecompStructure.Quantiles(nb).percent;
                           ConfPatch = patch([DateVector(StartSample:EndSample) fliplr(DateVector(StartSample:EndSample))],[Upper fliplr(Lower)],PatchColor);
                           set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
                           hold('on');
                           Handles = [ConfPatch Handles];
                        end;
                        set(gca,'Layer','Top');
                     end;
                     %
                     % plot the shock term
                     %
                     PartErrHandle = plot(DateVector(StartSample:EndSample),DSGEModel.HistDecompStructure.etathT(VariableValue,StartSample:EndSample,q),'--','Color',[0 0 0.75],'LineWidth',2);
                     hold('on');
                     PartErrStr = [StringTrim(ShockNames(q,:)) ' and'];
                     q = q+1;
                  elseif (NumPlotsMade==NumPlots)&(PlotMeasureError==1);
                     %
                     % check if we should add error bands
                     %
                     if AddPlots==1;
                        for nb=1:NumBounds;
                           Lower = DSGEModel.HistDecompStructure.Quantiles(nb).wthT(VariableValue,StartSample:EndSample);
                           Upper = DSGEModel.HistDecompStructure.Quantiles(length(DSGEModel.Percentiles)-nb+1).wthT(VariableValue,StartSample:EndSample);
                           PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
                           ConfBound = DSGEModel.HistDecompStructure.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.HistDecompStructure.Quantiles(nb).percent;
                           ConfPatch = patch([DateVector(StartSample:EndSample) fliplr(DateVector(StartSample:EndSample))],[Upper fliplr(Lower)],PatchColor);
                           set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
                           hold('on');
                           Handles = [ConfPatch Handles];
                        end;
                        set(gca,'Layer','Top');
                     end;
                     %
                     % plot the measurement error
                     %
                     PartErrHandle = plot(DateVector(StartSample:EndSample),DSGEModel.HistDecompStructure.wthT(VariableValue,StartSample:EndSample),'--','Color',[0 0 0.75],'LineWidth',2);
                     hold('on');
                     PartErrStr = 'Measurement error and';
                  end;
                  ErrHandle = plot(DateVector(StartSample:EndSample),DSGEModel.HistDecompStructure.epstth(VariableValue,StartSample:EndSample),'.-','Color',[0.75 0 0],'LineWidth',2);
                  hold('on');
                  SetAxesFonts(CurrINI,'Title',[PartErrStr ' error for ' CurrVariable]);
                  axis('tight');
                  %
                  % fix the yaxis
                  %
                  ylim = get(gca,'YLim');
                  addY = 0.1*(ylim(2)-ylim(1));
                  ylim = ylim+[-addY addY];
                  set(gca,'YLim',ylim);
                  %
                  % add a zero line if necessary
                  %
                  if (ylim(2)>0)&(ylim(1)<0);
                     ZeroLine = plot([DateVector(StartSample) DateVector(EndSample)],[0 0],'-','Color',[0 0 0]);
                  else;
                     ZeroLine = [];
                  end;
                  kids = [PartErrHandle ErrHandle ZeroLine Handles]';
                  set(gca,'Children',kids);
                  hold('off');
                  %
                  if (isempty(LegendStr)==0)&(ishandle(HistControls.legendcheck)==1)&(get(HistControls.legendcheck,'Value')==1);
                     if isempty(PartErrHandle)==1;
                        legend([ErrHandle Handles],strvcat('Mean error',LegendStr),2);
                     else;
                        legend([ErrHandle PartErrHandle Handles],strvcat('Mean error','Mean state error',LegendStr),2);
                     end;
                  end;
                  %
                  SetXTickIntegers(gca);
               end;
            end;
         end;
         SetToolbarImages(FigHandles{i},CurrINI.images);
         set(FigHandles{i},'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
         drawnow;
         pause(0.02);
      end;
   case 'done'
      set(HistControls.done,'UserData','done');
      delete(HistGraphGUI);
      drawnow;
      pause(0.02);
end;
%
if strcmp(selector,'init')==1;
   set(HistGraphGUI,'Visible','on');
   drawnow;
   if MatlabNumber>=7;
      uicontrol(HistControls.done);
   end;
   waitfor(HistControls.done,'UserData','done');
end;

%
% end of HistDecompDLG.m
%
