function GraphSimulatedData(selector,CurrINI,DSGEModel,SimulatedYData,NameStr)
% GraphSimulatedData: Displays impulse response functions
%
% USAGE:
%
%       GraphSimulatedData(selector,CurrINI,DSGEModel,IRStructure,NameStr)
%
% REQUIRED INPUT: selector (string vector) taking on the values "init", "showgraph", and "done".
%
% OPTIONAL INPUT: CurrINI (structure), with initialization information.
%
%                 DSGEModel (structure) whose fields contain all required information
%                           about the model, such as input files, variable and sample
%                           data. Actual observations are not stored in this file.
%
%                 SimulatedYData (matrix) with the simulated data.
%
%                 NameStr (string vector) taking on the values "Posterior Mode" or "Initial Values".
%
% NOTE: When selector = "init", then all optional input arguments are required!
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: July 11, 2008.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==1;
   DSGEModel.NameStr = NameStr;
   DSGEModel.SimulatedYData = SimulatedYData;
else
   SimulatedDataGUI = findobj('Type','figure','Tag','SimulatedDataDlg');
   SDcontrol = get(SimulatedDataGUI,'UserData');
   CurrINI = get(SDcontrol.variables,'UserData');
   DSGEModel = get(SDcontrol.variablestext,'UserData');
end;

%
% Set screen size directly instead of using
% get(0,'ScreenSize')!
%

switch selector;
   case 'init';
      SimulatedDataGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','GraphSimulatedData.m', ...
         'MenuBar','none', ...
         'PaperUnits','points', ...
         'Units','pixels', ...
         'Position',[(CurrINI.scrsz(3)-500)/2 (CurrINI.scrsz(4)-110)/2 500 110], ...
         'Tag','SimulatedDataDlg', ...
         'Visible','off', ...
         'Resize','off', ...
         'Name',['Simulated Data - ' NameStr], ...
         'NumberTitle','off', ...
         'CloseRequestFcn','GraphSimulatedData done', ...
         'ToolBar','none');

      if MatlabNumber>=7.0;
         set(SimulatedDataGUI,'DockControl','off');
      end;

      %
      % Outer axis
      %

      AxesBox([2 2 498 100],'Simulated Data Graphics',45,[0.5 1],'on',CurrINI);

      SDcontrol.sample = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 55 260 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'String',['Simulated data from observation 1 to ' int2str(size(DSGEModel.SimulatedYData,2))], ...
         'HorizontalAlignment','left', ...
         'Tag','SampleText');

      if strcmp(DSGEModel.DataFrequency,'q')==1;
         set(SDcontrol.sample,'String',strrep(get(SDcontrol.sample,'String'),'year','quarter'));
      elseif strcmp(DSGEModel.DataFrequency,'m')==1;
         set(SDcontrol.sample,'String',strrep(get(SDcontrol.sample,'String'),'year','month'));
      end;
      %
      % Select shock
      %
      SDcontrol.variablestext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 20 176 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'UserData',DSGEModel, ...
         'String','Select observed variable:', ...
         'HorizontalAlignment','left', ...
         'Tag','ShockText');
      %
      SDcontrol.variables = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[200 25 165 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',DSGEModel.VariableNames, ...
         'HorizontalAlignment','center', ...
         'Tag','ShocksPopup',...
         'Value',1);
      %
      % Create graph button
      %
      SDcontrol.show = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[395 55 90 20], ...
         'String','Display', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','GraphSimulatedData showgraph', ...
         'Enable','on', ...
         'Tag','Done');
      %
      %  Done button
      %
      SDcontrol.done = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[395 20 90 20], ...
         'String','Done', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','GraphSimulatedData done', ...
         'Enable','on', ...
         'UserData','waiting', ...
         'Tag','Done');

      set(SimulatedDataGUI,'UserData',SDcontrol,...
         'HandleVisibility','callback');
   case 'showgraph'
      VariableValue = get(SDcontrol.variables,'Value');
      CurrVariable = StringTrim(DSGEModel.VariableNames(VariableValue,:));
      if VariableValue<size(DSGEModel.VariableNames,1);
         set(SDcontrol.variables,'Value',VariableValue+1);
      else;
         set(SDcontrol.variables,'Value',1);
      end;
      %
      FigHandle = figure('Visible','off', ...
                  'Position',[(CurrINI.scrsz(3)-650)/2 (CurrINI.scrsz(4)-500)/2 650 500], ...
                  'Tag',CurrVariable, ...
                  'Name',['Simulated Data - ' CurrVariable]);
      %
      SetFigureProperties(FigHandle,CurrINI);
      %
      % plot the simulated data
      %
      if size(DSGEModel.SimulatedYData,2)~=DSGEModel.T;
         Dates = (1:size(DSGEModel.SimulatedYData,2));
      else;
         Dates = MakeDateVector(DSGEModel.BeginYear,DSGEModel.BeginPeriod,DSGEModel.EndYear,DSGEModel.EndPeriod,DSGEModel.DataFrequency);
      end;
      SDataHandle = plot(Dates,DSGEModel.SimulatedYData(VariableValue,:),'-','Color',[0 0 0.75],'LineWidth',2);
      hold('on');
      %
      if size(DSGEModel.SimulatedYData,2)==DSGEModel.T;
         DataHandle = plot(Dates,DSGEModel.Y(VariableValue,:),'.-','Color',[0.75 0 0],'LineWidth',2);
         hold('on');
      else;
         DataHandle = [];
      end;
      axis('tight');
      %
      SetAxesFonts(CurrINI,'Title',CurrVariable);
      %
      % see if we should add a zero line
      %
      ylim = get(gca,'YLim');
      add_on = 0.1*(ylim(2)-ylim(1));
      ylim = [ylim(1)-add_on ylim(2)+add_on];
      set(gca,'YLim',ylim);
      if (ylim(1)<0)&(ylim(2)>0);
         %
         % plot a zero line
         %
         ZeroHandle = plot([Dates(1) Dates(length(Dates))],[0 0],'-','Color',[0 0 0]);
         set(gca,'Children',[DataHandle SDataHandle ZeroHandle]');
      end;
      %
      % check if we should add a legend
      %
      if size(DSGEModel.SimulatedYData,2)==DSGEModel.T;
         legHandle = legend([DataHandle SDataHandle],'Actual Data','Simulated Data');
         SetAxesFonts(CurrINI,legHandle);
      end;
      %
      hold('off');
      %
      SetXTickIntegers(gca);
      %
      % display the graph
      %
      SetToolbarImages(FigHandle,CurrINI.images);
      set(FigHandle,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
      drawnow;
      pause(0.02);
   case 'done'
      set(SDcontrol.done,'UserData','done');
      delete(SimulatedDataGUI);
      drawnow;
      pause(0.02);
end;
%
if strcmp(selector,'init')==1;
   set(SimulatedDataGUI,'Visible','on');
   drawnow;
   if MatlabNumber>=7;
      uicontrol(SDcontrol.done);
   end;
   waitfor(SDcontrol.done,'UserData','done');
end;

%
% end of GraphSimulatedData.m
%
