function GraphParameterScenarioDLG(selector,Y,YScenario,OriginalShocks,ScenarioShocks,ShockNames,DateVector,BreakPeriod,AnnualizeData,AnnualStr,NameStr,DSGEModel,CurrINI)
% GraphParameterScenarioDLG: Plots observed variable path against scenario paths using original shocks but
%                            scenario parameter values or plots of smooth estimates of the economic shocks
%                            under the origional and the scenario parameter values.
%
% USAGE:
%
%       GraphParameterScenarioDLG(selector,Y,YScenario,OriginalShocks,ScenarioShocks,ShockNames,DateVector,BreakPeriod,AnnualizeData,AnnualStr,NameStr,DSGEModel,CurrINI)
%
% REQUIRED INPUT: selector (string vector) that takes on the values "init", "showgraph", or "done".
%
% OPTIONAL INPUT: Y (matrix) of dimension nxT with the observed variables for the estimation period (minus
%                   data observations due to annualization). Required if selector is "init".
%
%                 YScenario (matrix) of dimension nxT with the scenario paths for the observed variables.
%                   Required if selector is "init".
%
%                 OriginalShocks (matrix) of dimension nx(T+h) with smooth estimates of the economic shocks,
%                   using the original parameter values. The integer h is equal to 0 if data has not been
%                   annualized and the data frequency minus 1 if data has been annualized. Required if
%                   selector is "init".
%
%                 ScenarioShocks (matrix) of dimension nx(T+h) with smooth estimates of the economic shocks,
%                   using the scenario parameter values. Required if selector is "init".
%
%                 ShockNames (string matrix) with names of the nonzero economic shocks.Required if selector
%                   is "init".
%
%                 DateVector (vector) of dimension T+h with numerical date values. Rquired if selector is "init".
%
%                 BreakPeriod (integer) with the position in the sample when the parameter break is assumed
%                   to occur.
%
%                 AnnualizeData (boolean) that takes on the value 1 if data has been annualized and 0 otherwise.
%                   Required if selector is "init".
%
%                 AnnualStr (string vector) that takes on the values "Original" or "Annualized". Required if
%                   selector is "init".
%
%                 NameStr (string vector) that takes on the values "Posterior Mode" or "Initial Values".
%
%                 DSGEModel (structure) whose fields contain all required information about the model, such
%                   as input files, variable and sample data. Actual observations are not stored in this file.
%                   Required if selector is "init".
%
%                 CurrINI (structure) with initialization information. Required if selector is "init".
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: November 2, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 9-11-2007:  Added the call to "SetXTickIntegers".
%
% * 12-11-2007: Fixed a bug where "ShockScenariosGUI" was called rather than "GraphParameterScenarioGUI" when
%               attempting to set the "DockControl" property of the figure to "off".
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 14-1-2008:  Added the code that makes it possible to select the sample to display.
%
% * 23-5-2008:  Updated the documentation.
%
% * 25-7-2008:  Added the input variable "BreakPeriod".
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==1;
   CurrINI.Y = Y;
   CurrINI.YScenario = YScenario;
   CurrINI.OriginalShocks = OriginalShocks;
   CurrINI.ScenarioShocks = ScenarioShocks;
   CurrINI.DateVector = DateVector;
   CurrINI.AnnualizeData = AnnualizeData;
   CurrINI.AnnualStr = AnnualStr;
   CurrINI.ShockNames = ShockNames;
   CurrINI.VariableNames = DSGEModel.VariableNames;
   CurrINI.NameStr = NameStr;
   CurrINI.BreakPeriod = BreakPeriod;
else;
   GraphParameterScenarioGUI = findobj('Type','figure','Tag','GraphParameterScenarioDLG');
   controls = get(GraphParameterScenarioGUI,'UserData');
   CurrINI = get(controls.variables,'UserData');
end;
%
% provide the switch part
%
switch selector
   case 'init'
      GraphParameterScenarioGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','GraphParameterScenarioDLG.m', ...
         'MenuBar','none', ...
         'PaperUnits','points', ...
         'Units','pixels', ...
         'Position',[(CurrINI.scrsz(3)-500)/2 (CurrINI.scrsz(4)-120)/2 500 120], ...
         'Tag','GraphParameterScenarioDLG', ...
         'Visible','off', ...
         'Resize','off', ...
         'Name',['Graph Parameter Scenarios - ' AnnualStr ' Data'], ...
         'NumberTitle','off', ...
         'CloseRequestFcn','GraphParameterScenarioDLG done', ...
         'ToolBar','none');
      %
      if MatlabNumber>=7.0;
         set(GraphParameterScenarioGUI,'DockControl','off');
      end;
      %
      % Outer axis
      %
      AxesBox([2 2 498 110],'Graphics',45,[0.5 1],'on',CurrINI);
      %
      %
      if strcmp(AnnualStr,'Original')==1;
         if DSGEModel.KalmanFirstObservation>1;
            [NewStartYear,NewStartPeriod] = AdjustSampleStart(DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.DataFrequency,DSGEModel.KalmanFirstObservation-1);
         else;
            NewStartYear = DSGEModel.SubBeginYear;
            NewStartPeriod = DSGEModel.SubBeginPeriod;
         end;
      else;
         [NewStartYear,NewStartPeriod] = AdjustSampleStart(DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.DataFrequency,max(DSGEModel.annual)+DSGEModel.KalmanFirstObservation-2);
      end;
      %
      OldDSGEModel = DSGEModel;
      DSGEModel.SubBeginYear = NewStartYear;
      DSGEModel.SubBeginPeriod = NewStartPeriod;
      [FirstPeriod,LastPeriod,SubSampleStrMat] = CreateSubSample(DSGEModel);
      %
      CurrINI.SubSampleStrMat = SubSampleStrMat;
      %
      if strcmp(AnnualStr,'Original')==1;
         CurrINI.ShockSubSampleStrMat = CurrINI.SubSampleStrMat;
      else;
         %
         % create a specific subsample string matrix for the shocks
         %
         DSGEModel = OldDSGEModel;
         [NewStartYear,NewStartPeriod] = AdjustSampleStart(DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.DataFrequency,DSGEModel.KalmanFirstObservation-1);
         DSGEModel.SubBeginYear = NewStartYear;
         DSGEModel.SubBeginPeriod = NewStartPeriod;
         [FirstPeriod,LastPeriod,CurrINI.ShockSubSampleStrMat] = CreateSubSample(DSGEModel);
      end;
      %
      DSGEModel = OldDSGEModel;
      %
      %
      % Sample selection controls
      %
      controls.sampletxt = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 50 360 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'String','Selected Sample:', ...
         'HorizontalAlignment','left', ...
         'Tag','SampleText');
      %
      % sample start and sample end
      %
      controls.samplebeg = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[185 55 87 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',SubSampleStrMat, ...
         'HorizontalAlignment','center', ...
         'UserData',1, ...
         'Callback','GraphParameterScenarioDLG samplebegin', ...
         'Tag','SampleBeginPopup',...
         'Value',1);
      %
      controls.sampleend = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[278 55 87 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',SubSampleStrMat, ...
         'HorizontalAlignment','center', ...
         'UserData',0, ...
         'Callback','GraphParameterScenarioDLG sampleend', ...
         'Tag','SampleEndPopup',...
         'Value',size(SubSampleStrMat,1));
      %
%      controls.sample = uicontrol('Units','pixels', ...
%         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
%         'Position',[24 50 360 20], ...
%         'FontSize',CurrINI.gui_fontsize, ...
%         'FontName',CurrINI.gui_fontname, ...
%         'FontWeight',CurrINI.gui_fontweight, ...
%         'FontAngle',CurrINI.gui_fontangle, ...
%         'Style','text', ...
%         'String',['Selected Sample: ' NewStartYear ':' NewStartPeriod ' - ' DSGEModel.SubEndYear ':' DSGEModel.SubEndPeriod], ...
%         'HorizontalAlignment','left', ...
%         'Tag','SampleText');
      %
      % select variable or shock to plot
      %
      controls.variablestext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 16 160 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'String','Select variable or shock:', ...
         'HorizontalAlignment','left', ...
         'Tag','ObsVarText');
      %
      controls.variables = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[185 20 180 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',strvcat(DSGEModel.VariableNames,ShockNames), ...
         'Callback','GraphParameterScenarioDLG checksample', ...
         'HorizontalAlignment','center', ...
         'Tag','ObsVarPopup',...
         'Value',1);
      %
      % Create graph button
      %
      controls.show = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[395 55 90 20], ...
         'String','Display', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','GraphParameterScenarioDLG showgraph', ...
         'Enable','on', ...
         'Tag','Done');
      %
      %  Done button
      %
      controls.done = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[395 20 90 20], ...
         'String','Done', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','GraphParameterScenarioDLG done', ...
         'Enable','on', ...
         'UserData','waiting', ...
         'Tag','Done');
      %
      % store controls in figure userdata
      %
      set(GraphParameterScenarioGUI,'UserData',controls,...
         'HandleVisibility','callback');
   case 'samplebegin'
      %
      % check to make sure that the sample begins before it ends!
      %
      samplebeg = get(controls.samplebeg,'Value');
      sampleend = get(controls.sampleend,'Value');
      if sampleend<samplebeg;
         set(controls.sampleend,'Value',samplebeg);
      end;
      %
      % the user has manually attempted to change the sample. This affects how
      % StartStart=1 is used when moving from a short to a long sample when
      % annualized data are displayed for observed variables.
      %
      set(controls.sampleend,'UserData',1);
   case 'sampleend'
      %
      % check to make sure that the sample begins before it ends
      %
      samplebeg = get(controls.samplebeg,'Value');
      sampleend = get(controls.sampleend,'Value');
      if sampleend<samplebeg;
         set(controls.samplebeg,'Value',sampleend);
      end;
      %
      % the user has manually attempted to change the sample. This affects how
      % StartStart=1 is used when moving from a short to a long sample when
      % annualized data are displayed for observed variables.
      %
      set(controls.sampleend,'UserData',1);
   case 'checksample'
      if CurrINI.AnnualizeData==1;
         [n,T] = size(CurrINI.Y);
         OldType = get(controls.samplebeg,'UserData');
         NewValue = get(controls.variables,'Value');
         if NewValue<=n;
            set(controls.samplebeg,'UserData',1);
         else;
            set(controls.samplebeg,'UserData',0);
         end;
         NewType = get(controls.samplebeg,'UserData');
         if OldType~=NewType;
            AddOn = size(CurrINI.OriginalShocks,2)-T;
            StartSample = get(controls.samplebeg,'Value');
            EndSample = get(controls.sampleend,'Value');
            if NewType==1;
               %
               % change to short sample
               %
               if StartSample>AddOn;
                  ShortStartValue = StartSample-AddOn;
               else;
                  ShortStartValue = StartSample;
               end;
               if EndSample>AddOn;
                  ShortEndValue = EndSample-AddOn;
               else;
                  ShortEndValue = EndSample;
               end;
               set(controls.samplebeg,'String',CurrINI.SubSampleStrMat,'Value',ShortStartValue);
               set(controls.sampleend,'String',CurrINI.SubSampleStrMat,'Value',ShortEndValue);
            else;
               %
               % change to long sample
               %
               if (StartSample==1)&(get(controls.sampleend,'UserData')==0);
                  set(controls.samplebeg,'String',CurrINI.ShockSubSampleStrMat,'Value',StartSample);
               else;
                  set(controls.samplebeg,'String',CurrINI.ShockSubSampleStrMat,'Value',StartSample+AddOn);
               end;
               set(controls.sampleend,'String',CurrINI.ShockSubSampleStrMat,'Value',EndSample+AddOn);
            end;
         end;
      end;
   case 'showgraph'
      [n,T] = size(CurrINI.Y);
      q = size(CurrINI.OriginalShocks,1);
      CurrValue = get(controls.variables,'Value');
      AllVariables = get(controls.variables,'String');
      CurrVariable = StringTrim(AllVariables(CurrValue,:));
      if CurrValue<n+q;
         NewValue = CurrValue+1;
      else;
         NewValue = 1;
      end;
      set(controls.variables,'Value',NewValue);
      OldType = get(controls.samplebeg,'UserData');
      if NewValue<=n;
         set(controls.samplebeg,'UserData',1);
      else;
         set(controls.samplebeg,'UserData',0);
      end;
      NewType = get(controls.samplebeg,'UserData');
      %
      % get the data to plot
      %
      DateVector = CurrINI.DateVector;
      if CurrINI.BreakPeriod>1;
         BreakPos = (DateVector(CurrINI.BreakPeriod-1)+DateVector(CurrINI.BreakPeriod))/2;
      else;
         BreakPos = DateVector(CurrINI.BreakPeriod);
      end;
      if CurrValue<=n;
         %
         % we shall plot a variable
         %
         y = CurrINI.Y(CurrValue,:);
         yscen = CurrINI.YScenario(CurrValue,:);
         if CurrINI.AnnualizeData==1;
            DateVector = DateVector(length(DateVector)-T+1:length(DateVector));
         end;
         FigStr = [CurrINI.AnnualStr ' Data ' CurrVariable];
         yMarker = '.-';
         TitleStr = [CurrVariable ' and data for scenario parameters with smooth estimates of shocks at ' lower(CurrINI.NameStr)];
         LegendStr = strvcat([CurrINI.AnnualStr ' Data'],'Scenario Data');
      else;
         %
         % we shall plot a shock
         %
         y = CurrINI.OriginalShocks(CurrValue-n,:);
         yscen = CurrINI.ScenarioShocks(CurrValue-n,:);
         FigStr = ['Economic Shock ' CurrVariable];
         yMarker = ':';
         TitleStr = ['Scenario estimates of ' CurrVariable ' relative to smooth estimates at ' lower(CurrINI.NameStr)];
         LegendStr = strvcat(['Smooth estimates at ' lower(CurrINI.NameStr)],'Smooth estimates for scenario');
      end;
      %
      StartSample = get(controls.samplebeg,'Value');
      EndSample = get(controls.sampleend,'Value');
      if StartSample>EndSample;
         OldEndSample = EndSample;
         EndSample = StartSample;
         StartSample = OldEndSample;
      end;
      %
      % check if we should reset the sample controls
      %
      if (CurrINI.AnnualizeData==1)&(OldType~=NewType);
         AddOn = size(CurrINI.OriginalShocks,2)-T;
         if NewType==1;
            %
            % change to short sample
            %
            if StartSample>AddOn;
               ShortStartValue = StartSample-AddOn;
            else;
               ShortStartValue = StartSample;
            end;
            if EndSample>AddOn;
               ShortEndValue = EndSample-AddOn;
            else;
               ShortEndValue = EndSample;
            end;
            set(controls.samplebeg,'String',CurrINI.SubSampleStrMat,'Value',ShortStartValue);
            set(controls.sampleend,'String',CurrINI.SubSampleStrMat,'Value',ShortEndValue);
         else;
            %
            % change to long sample
            %
            if (StartSample==1)&(get(controls.sampleend,'UserData')==0);
               set(controls.samplebeg,'String',CurrINI.ShockSubSampleStrMat,'Value',StartSample);
            else;
               set(controls.samplebeg,'String',CurrINI.ShockSubSampleStrMat,'Value',StartSample+AddOn);
            end;
            set(controls.sampleend,'String',CurrINI.ShockSubSampleStrMat,'Value',EndSample+AddOn);
         end;
      end;
      %
      % create the figure
      %
      FigHandle =figure('Visible','off', ...
                'Position',[(CurrINI.scrsz(3)-650)/2 (CurrINI.scrsz(4)-500)/2 650 500], ...
                'Tag',CurrVariable, ...
                'Name',['Parameter Scenario - ' FigStr]);
      %
      SetFigureProperties(FigHandle,CurrINI);
      %
      % the original data plot
      %
      yHandle = plot(DateVector(StartSample:EndSample),y(StartSample:EndSample),yMarker,'Color',[1 0 0],'LineWidth',2);
      hold('on');
      yScenHandle = plot(DateVector(StartSample:EndSample),yscen(StartSample:EndSample),'--','Color',[0 0 0.75],'LineWidth',2);
      hold('on');
      SetAxesFonts(CurrINI,'Title',TitleStr);
      axis('tight');
      %
      ylim = get(gca,'YLim');
      AddOn = 0.1*(ylim(2)-ylim(1));
      ylim = [ylim(1)-AddOn ylim(2)+AddOn];
      set(gca,'YLim',ylim,'XLim',[DateVector(StartSample) DateVector(EndSample)]);
      %
      % check if we should display the break position
      %
      if (BreakPos>DateVector(StartSample))&(BreakPos<DateVector(EndSample));
         BreakLine = plot(BreakPos*ones(1,2),ylim,':','Color',[0 0 0]);
         hold('on');
      else;
         BreakLine = [];
      end;
      %
      % check if we should add a horizontal zero line
      %
      if (ylim(1)<0)&(ylim(2)>0);
         ZeroLine = plot([DateVector(StartSample) DateVector(EndSample)],[0 0],'-','Color',[0 0 0]);
         %
         %
         kids = get(gca,'Children');
         kids = [yScenHandle yHandle ZeroLine BreakLine]';
         set(gca,'Children',kids);
      end;
      %
      % add a legend
      %
      legHandle = legend([yHandle yScenHandle],LegendStr);
      SetAxesFonts(CurrINI,legHandle);
      hold('off');
      %
      SetXTickIntegers(gca);
      %
      % display the graph
      %
      SetToolbarImages(FigHandle,CurrINI.images);
      set(FigHandle,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
      drawnow;
      pause(0.02);
   case 'done'
      set(controls.done,'UserData','done');
      delete(GraphParameterScenarioGUI);
      drawnow;
      pause(0.02);
end;
%
if strcmp(selector,'init')==1;
   set(GraphParameterScenarioGUI,'Visible','on');
   drawnow;
   if MatlabNumber>=7;
      uicontrol(controls.done);
   end;
   waitfor(controls.done,'UserData','done');
end;

%
% end of GraphParameterScenarioDLG.m
%
