function GraphLogLikelihood(lnLt,DSGEModel,CurrINI,NameStr)
% GraphLogLikelihood: Plots the log-likelihood function over time when it is evaluated at a given parameter value
%
% USAGE:
%
%       GraphLogLikelihood(lnLt,DSGEModel,CurrINI,NameStr)
%
% REQUIRED INPUT:  lnLt (1xT vector) with the values of the log-likelihood function based on the Kalman filter
%                       calculations.
%
%                  DSGEModel (structure) whose fields contain all required information about the model,
%                       such as input files, variable and sample data.
%
%                  CurrINI (structure), with initialization information.
%
%                  NameStr (string vector), that takes on the values "Posterior Mode" or "Initial"
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: November 17, 2006.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 9-11-2007:  Updated the documentation.
%
% * 21-12-2007: Added a drawnow call after making the figure visibile. The function "SetXTickIntegers"
%               is now also called.
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% create a vector with dates
%
DateVector = MakeDateVector(DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.SubEndYear,DSGEModel.SubEndPeriod,DSGEModel.DataFrequency);
%
% fix it if the length of the data is too short
%
if length(DateVector)>length(lnLt);
   DateVector = DateVector(1,length(DateVector)-length(lnLt)+1:length(DateVector));
end;
%
% create the figure
%
FigHandle = figure('Visible','off', ...
                   'Position',[(CurrINI.scrsz(3)-650)/2 (CurrINI.scrsz(4)-500)/2 650 500], ...
                   'Tag','LogLikelihood', ...
                   'Name',['Log-Likelihood Function - ' NameStr]);
%
SetFigureProperties(FigHandle,CurrINI);
%
subplot(2,1,1);
%
% start up a plot
%
LineHandle = plot(DateVector,lnLt,'-','Color',[0.75 0 0],'LineWidth',2);
SetAxesFonts(CurrINI,'Title',['Log-likelihood function ln p(y(t)|x(t),..,y(t-1),...) at ' lower(NameStr)]);
axis('tight');
%
% see if we should add a zero line
%
ylim = get(gca,'YLim');
add_on = 0.1*(ylim(2)-ylim(1));
ylim = [ylim(1)-add_on ylim(2)+add_on];
set(gca,'YLim',ylim);
if (ylim(1)<0)&(ylim(2)>0);
   hold('on');
   plot([DateVector(1) DateVector(length(DateVector))],[0 0],'-','Color',[0 0 0]);
   %
   % make sure the zero line is displayed below the observed variable
   %
   kids = get(gca,'Children');
   kids = [kids(2) kids(1)]';
   set(gca,'Children',kids);
end;
hold('off');
%
% Now we try to display the accumulated log-likelihood
%
subplot(2,1,2)
%
% start up a plot
%
LineHandle = plot(DateVector,cumsum(lnLt),'-','Color',[0.75 0 0],'LineWidth',2);
SetAxesFonts(CurrINI,'Title',['Accumulated log-likelihood function at ' lower(NameStr)]);
axis('tight');
SetXTickIntegers(gca);
%
% display the graphs
%
SetToolbarImages(FigHandle,CurrINI.images);
set(FigHandle,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
drawnow;
pause(0.02);

%
% end of GraphLogLikelihood.m
%
