function GraphEditDataDLG(selector,DLGName,VariableNames,VariableData,ComparisonData,Transformation,DTStatus,DateVector,SampleStrMat,annual,annualscale,DataFrequency,CurrINI)
% GraphEditDataDLG: Presents a dialog for selecting edited data to plot.
%
% USAGE:
%
%       GraphEditDataDLG(selector,DLGName,VariableNames,VariableData,ComparisonData,Transformation,DTStatus,DateVector,SampleStrMat,annual,annualscale,DataFrequency,CurrINI)
%
% REQUIRED INPUT: selector (string vector) that defines the sub-function to run.
%
% OPTIONAL INPUT: DLGName (string vector) with name for the dialog. Required if selector is "init".
%
%                 VariableNames (string matrix) of dimension cxc1 with the names of the variables. Required if
%                               selector is "init".
%                 
%                 VariableData (matrix) of dimension Txc with the data for the variables. Required  if
%                               selector is "init".
%
%                 ComparisonData (matrix) of dimension Txc (or empty) with comparison data for the variables.
%                               Required if selector is "init".
%
%                 Transformation (structure) with fields given by the names in VariableNames, each with
%                               subfields  "fcn", "initial", and "x".
%
%                 DTStatus (boolean) that indicates if the data transformation checkbox should be set to
%                               checked (1) or not (0).
%
%                 DateVector (vector) with numerical dates for the data. Required if selector is "init".
%
%                 SampleStrMat (string matrix) with T rows and whose elements give the sample start and end
%                               periods.
%
%                 annual (vector) indicating how to annualize the variables.
%
%                 annualscale (vector) indicating which scale factors to use.
%
%                 DataFrequency (string vector) with the frequency of the data (q, m, or a).
%
%                 CurrINI (structure) with initialization information.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: October 9, 2007.
%                        This version: February 26, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 10-10-2007: Made sure the X and Y limits are set correctly for the graph.
%
% * 22-10-2007: Allowed for first differences. The "DataFrequency" input is now also included. The user defined
%               transformations are now also considered through the input variable "Transformation".
%
% * 23-10-2007: The function now calls "TransformData" for data transformations.
%
% * 24-10-2007: Increased the maximum height of the About dialog when the TransformData function returns
%               a very long error message.
%
% * 30-10-2007: Updated the documentation.
%
% * 6-11-2007:  Added the input "ComparisonData".
%
% * 7-11-2007:  Added the input "SampleStrMat".
%
% * 9-11-2007:  Added the call to "SetXTickIntegers".
%
% * 12-11-2007: Added the input variable "DTStatus".
%
% * 14-11-2007: Added the 2 new inputs to the calls to the "TransformData" function.
%
% * 20-11-2007: Removed the input variables "FirstObs" and "LastObs".
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 31-1-2008:  Changed line colors and line markers.
%
% * 1-2-2008:   Switched the ordering of the lines in the legend. The "Current Data" is now above
%               the "Comparison Data".
%
% * 25-4-2008:  Changed the colors for the current data (now red) and the comparison data (now blue).
%               The colors therefore mathc those used when running a scenario.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
% * 26-02-2009: Removed the graph comparison data option when this data is an empty matrix
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==1;
   Controls.VariableNames = VariableNames;
   Controls.VariableData = VariableData;
   Controls.DateVector = DateVector;
   Controls.annual = annual;
   Controls.annualscale = annualscale;
   Controls.DataFrequency = DataFrequency;
   Controls.SampleStrMat = SampleStrMat;
   Controls.Transformation = Transformation;
   Controls.ComparisonData = ComparisonData;
   if isempty(ComparisonData)==1;
      AddHeight = 0;
   else;
      AddHeight = 20;
   end;
else;
   GraphGUI = findobj('Type','figure','Tag','GraphEditDataDLG');
   Controls = get(GraphGUI,'UserData');
   CurrINI = get(Controls.show,'UserData');
end;
%
switch selector;
   case 'init';
      GraphGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','GraphEditDataDLG.m', ...
         'MenuBar','none', ...
         'PaperUnits','points', ...
         'Units','pixels', ...
         'Position',[(CurrINI.scrsz(3)-500)/2 (CurrINI.scrsz(4)-185-AddHeight)/2 500 185+AddHeight], ...
         'Tag','GraphEditDataDLG', ...
         'Visible','off', ...
         'Resize','off', ...
         'Name',['Graph Data - ' DLGName], ...
         'NumberTitle','off', ...
         'CloseRequestFcn','GraphEditDataDLG done', ...
         'ToolBar','none');
      %
      % Outer axis
      %
      AxesBox([2 2 498 175+AddHeight],'Graphics',45,[0.5 1],'on',CurrINI);
      %
      % set the selected sample
      %
      Controls.sample = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 130+AddHeight 360 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'String','Selected Sample:', ...
         'HorizontalAlignment','left', ...
         'Tag','SampleText');
      %
      % sample start and sample end
      %
      Controls.samplebeg = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[185 135+AddHeight 87 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',SampleStrMat, ...
         'HorizontalAlignment','center', ...
         'Callback','GraphEditDataDLG samplebegin', ...
         'Tag','SampleBeginPopup',...
         'Value',1);
      %
      Controls.sampleend = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[278 135+AddHeight 87 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',SampleStrMat, ...
         'HorizontalAlignment','center', ...
         'Callback','GraphEditDataDLG sampleend', ...
         'Tag','SampleEndPopup',...
         'Value',size(SampleStrMat,1));
      %
      % Variable selection controls
      %
      Controls.variablestext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 95+AddHeight 180 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'String','Select variable:', ...
         'HorizontalAlignment','left', ...
         'Tag','ObsVarText');
      %
      Controls.variables = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[185 100+AddHeight 180 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'String',VariableNames, ...
         'HorizontalAlignment','center', ...
         'Tag','ObsVarPopup',...
         'Value',1);
      %
      Controls.transformationcheck = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 68+AddHeight 360 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','checkbox', ...
         'Value',DTStatus, ...
         'CallBack','GraphEditDataDLG fixchecks', ...
         'String','Transform the data with user defined functions', ...
         'HorizontalAlignment','left', ...
         'Tag','TransformCheckBox');
      %
      if isempty(Transformation)==1;
         set(Controls.transformationcheck,'Enable','off','Value',0);
      end;
      %
      Controls.annualize = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 44+AddHeight 360 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','checkbox', ...
         'Value',0, ...
         'String','Annualize the data', ...
         'HorizontalAlignment','left', ...
         'Tag','AnnualizeCheckBox');
      %
      Controls.firstdiffs = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 20+AddHeight 360 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','checkbox', ...
         'Value',0, ...
         'String','Difference the data', ...
         'HorizontalAlignment','left', ...
         'Tag','DifferenceCheckBox');
      %
      if get(Controls.transformationcheck,'Value')==1;
         set(Controls.firstdiffs,'Enable','off');
      end;
      %
      if isempty(ComparisonData)==0;
         Controls.comparisondatacheck = uicontrol('Units','pixels', ...
            'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
            'Position',[24 16 360 20], ...
            'FontSize',CurrINI.gui_fontsize, ...
            'FontName',CurrINI.gui_fontname, ...
            'FontWeight',CurrINI.gui_fontweight, ...
            'FontAngle',CurrINI.gui_fontangle, ...
            'Style','checkbox', ...
            'Enable','on', ...
            'Value',1, ...
            'String','Display comparison data in graphs', ...
            'HorizontalAlignment','left', ...
            'Tag','ComparisonDataCheckBox');
      end;
      %
      % display button
      %
      Controls.show = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[395 55 90 20], ...
         'String','Display', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','GraphEditDataDLG showgraph', ...
         'UserData',CurrINI, ...
         'Enable','on', ...
         'Tag','Done');
      %
      %  Done button
      %
      Controls.done = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[395 20 90 20], ...
         'String','Done', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','GraphEditDataDLG done', ...
         'Enable','on', ...
         'UserData','waiting', ...
         'Tag','Done');
      %
      set(GraphGUI,'UserData',Controls,'HandleVisibility','callback');
   case 'samplebegin'
      %
      % check to make sure that the sample begins before it ends!
      %
      samplebeg = get(Controls.samplebeg,'Value');
      sampleend = get(Controls.sampleend,'Value');
      if sampleend<samplebeg;
         set(Controls.sampleend,'Value',samplebeg);
      end;
   case 'sampleend'
      %
      % check to make sure that the sample begins before it ends
      %
      samplebeg = get(Controls.samplebeg,'Value');
      sampleend = get(Controls.sampleend,'Value');
      if sampleend<samplebeg;
         set(Controls.samplebeg,'Value',sampleend);
      end;
   case 'showgraph'
      CurrValue = get(Controls.variables,'Value');
      NumValues = size(Controls.VariableNames,1);
      CurrVariable = StringTrim(Controls.VariableNames(CurrValue,:));
      %
      % reset the control
      %
      if CurrValue<NumValues
         set(Controls.variables,'Value',CurrValue+1);
      else;
         set(Controls.variables,'Value',1);
      end;
      %
      % prepare the data for display
      %
      y = Controls.VariableData(:,CurrValue);
      if (isfield(Controls,'comparisondatacheck')==1)&(get(Controls.comparisondatacheck,'Value')==1);
         x = Controls.ComparisonData(:,CurrValue);
      else;
         x = [];
      end;
      %
      DateVector = Controls.DateVector;
      DateVectorSampleBeg = DateVector(get(Controls.samplebeg,'Value'));
      DateVectorSampleEnd = DateVector(get(Controls.sampleend,'Value'));
      CurrTitle = CurrVariable;
      if get(Controls.transformationcheck,'Value')==1;
         %
         % try to run the appropriate transformations
         %
         OldSize = length(y);
         [y,BadTrans,ErrorStr,HeaderStr,FcnStr] = TransformData(y,Controls.Transformation,CurrVariable,get(Controls.annualize,'Value'));
         if BadTrans==0;
            NewSize = length(y);
            if NewSize>OldSize;
               y = y(NewSize-OldSize+1:NewSize);
            elseif NewSize<OldSize;
               DateVector = DateVector(OldSize-NewSize+1:OldSize);
            end;
            if get(Controls.annualize,'Value')==0;
               CurrTitle = ['Tranformation of ' CurrTitle];
            else;
               CurrTitle = ['Annual tranformation of ' CurrTitle];
            end;
            if isempty(x)==0;
               %
               % transform the x data and check if it is performed without error
               %
               [x,BadTrans,ErrorStr,HeaderStr,FcnStr] = TransformData(x,Controls.Transformation,CurrVariable,get(Controls.annualize,'Value'));
               if BadTrans==1;
                  About(ErrorStr,'error',HeaderStr,300,500,CurrINI);
                  x = [];
               end;
            end;
         else;
            About(ErrorStr,'error',HeaderStr,300,500,CurrINI);
            return;
         end;
      else;
         if get(Controls.firstdiffs,'Value')==1;
            OldSize = length(y);
            if get(Controls.annualize,'Value')==1;
               if strcmp(lower(Controls.DataFrequency),'q')==1;
                  y = y(5:OldSize)-y(1:OldSize-4);
                  if isempty(x)==0;
                     x = x(5:OldSize)-x(1:OldSize-4);
                  end;
                  UScoreStr = '_{4}';
               elseif strcmp(lower(Controls.DataFrequency),'m')==1;
                  y = y(13:OldSize)-y(1:OldSize-12);
                  if isempty(x)==0;
                     x = x(13:OldSize)-x(1:OldSize-12);
                  end;
                  UScoreStr = '_{12}';
               else;
                  y = y(2:OldSize)-y(1:OldSize-1);
                  if isempty(x)==0;
                     x = x(2:OldSize)-x(1:OldSize-1);
                  end;
                  UScoreStr = '';
               end;
               NewSize = length(y);
               DateVector = DateVector(OldSize-NewSize+1:length(DateVector));
               CurrTitle = ['\Delta' UScoreStr ' ' CurrTitle];
            else;
               y = y(2:OldSize)-y(1:OldSize-1);
               if isempty(x)==0;
                  x = x(2:OldSize)-x(1:OldSize-1);
               end;
               NewSize = length(y);
               DateVector = DateVector(OldSize-NewSize+1:length(DateVector));
               CurrTitle = ['\Delta ' CurrTitle];
            end;
         else;
            if get(Controls.annualize,'Value')==1;
               OldSize = length(y);
               y = Annualize(y',Controls.annual(CurrValue),Controls.annualscale(CurrValue))';
               if isempty(x)==0;
                  x = Annualize(x',Controls.annual(CurrValue),Controls.annualscale(CurrValue))';
               end;
               NewSize = length(y);
               DateVector = DateVector(OldSize-NewSize+1:length(DateVector));
               CurrTitle = ['Annualized ' CurrTitle];
            end;
         end;
      end;
      %
      % create the figure
      %
      FigHandle = figure('Visible','off', ...
                 'Position',[(CurrINI.scrsz(3)-650)/2 (CurrINI.scrsz(4)-500)/2 650 500], ...
                 'Tag',CurrVariable, ...
                 'Name',['Graph of ' CurrVariable]);
      %
      SetFigureProperties(FigHandle,CurrINI);
      %
      % check if we should plot the comparison data
      %
      if isempty(x)==0;
         xHandle = plot(DateVector,x,'-.','Color',[0 0 0.75],'LineWidth',2);
         hold('on');
      end;
      %
      yHandle = plot(DateVector,y,'+-','Color',[0.75 0 0],'LineWidth',2);
      axis('tight');
      SetAxesFonts(CurrINI,'Title',CurrTitle);
      %
      % check if we should add a zero line
      %
      ylim = get(gca,'YLim');
      addY = 0.1*(ylim(2)-ylim(1));
      ylim = [ylim(1)-addY ylim(2)+addY];
      if (ylim(1)<0)&(ylim(2)>0);
         %
         % add a horiozntal zero line
         %
         hold('on');
         plot([DateVector(1) DateVector(length(DateVector))],[0 0],'-','Color',[0 0 0]);
         %
         % make sure the zero line is displayed below the observed variable
         %
         kids = get(gca,'Children');
         if isempty(x)==0;
            kids = kids([2 3 1]);
         else;
            kids = kids([2 1]);
         end;
         set(gca,'Children',kids);
      end;
      set(gca,'YLim',ylim,'XLim',[max(DateVectorSampleBeg,DateVector(1)) DateVectorSampleEnd]);
      %
      % check if we should set up a legend
      %
      if isempty(x)==0;
         legHandle = legend([yHandle xHandle],'Current Data','Comparison Data');
         SetAxesFonts(CurrINI,legHandle);
      end;
      hold('off');
      %
      % check if the X label should be fixed
      %
      SetXTickIntegers(gca);
      %
      % display the graph
      %
      SetToolbarImages(FigHandle,CurrINI.images);
      set(FigHandle,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
      drawnow;
      pause(0.02);
   case 'fixchecks'
      if get(Controls.transformationcheck,'Value')==1;
         set(Controls.firstdiffs,'Enable','off','Value',0);
      else;
         set(Controls.firstdiffs,'Enable','on');
      end;
   case 'done'
      set(Controls.done,'UserData','done');
end;

if strcmp(selector,'init')==1;
   set(GraphGUI,'Visible','on');
   drawnow;
   drawnow;
   %
   % run the waitfor function
   %
   waitfor(Controls.done,'UserData','done');
   %
   % delete the dialog
   %
   delete(GraphGUI);
   drawnow;
   pause(0.02);
end;

%
% end of GraphEditDataDLG.m
%
