function GraphDataDLG(selector,CurrINI,DSGEModel,AnnualStr)
% GraphDataDLG: Displays graphs of the observed data.
%
% USAGE:
%
%       GraphDataDLG(selector,CurrINI,DSGEModel)
%
% REQUIRED INPUT: selector (string vector) taking on the values "init", "showgraph", and "done".
%
% OPTIONAL INPUT: CurrINI (structure), with initialization information.
%
%                 DSGEModel (structure) whose fields contain all required information
%                           about the model, such as input files, variable and sample
%                           data. Actual observations are not stored in this file.
%
%                 AnnualStr (string) taking on the values "Original", "Annualized", or "Transformed".
%
% NOTE: When selector = "init", then all optional input arguments are required!
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: August 18, 2006.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 23-8-2006:  Minor fixes.
%
% * 29-8-2006:  Added the call to SetToolbarImages.
%
% * 24-11-2006: Added an extra drawnow after setting figure display to 'visible'.
%
% * 3-1-2007:   Fixed the text string for selecting variable.
%
% * 23-1-2007:  Added the input variable "AnnualStr".
%
% * 25-1-2007:  Made sure that the "Annualize" function receives the relevant terms of the DSGEModel.annualscale
%               vector.
%
% * 2-11-2007:  Updated the documentation.
%
% * 9-11-2007:  Added ther call to "SetXTickIntegers".
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 13-5-2008:  The "AnnualStr" input can now take on the value "Transformed".
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==0;
   GraphGUI = findobj('Type','figure','Tag','GraphDataDLG');
   controls = get(GraphGUI,'UserData');
   CurrINI = get(controls.variables,'UserData');
   DSGEModel = get(controls.variablestext,'UserData');
else;
   if strcmp(AnnualStr,'Transformed')==1;
      %
      % transform the data
      %
      [YTStatus,YData,ErrorStr,HeaderStr] = CalcTransformData(DSGEModel.VariableNames,DSGEModel.Y',DSGEModel.YTransformation);
      if YTStatus==1;
         DSGEModel.TransformedY = YData';
         if size(DSGEModel.TransformedY,2)<size(DSGEModel.Y,2);
            DSGEModel.TransformedY = [NaN*ones(DSGEModel.n,size(DSGEModel.Y,2)-size(DSGEModel.TransformedY,2)) DSGEModel.TransformedY];
         end;
      else;
         About(ErrorStr,'information',HeaderStr,200,500,CurrINI);
         return;
      end;
   else;
      DSGEModel.TransformedY = [];
   end;
end;

%
% Set screen size directly instead of using
% get(0,'ScreenSize')!
%

switch selector;
   case 'init';
      GraphGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','GraphDataDLG.m', ...
         'MenuBar','none', ...
         'PaperUnits','points', ...
         'Units','pixels', ...
         'Position',[(CurrINI.scrsz(3)-500)/2 (CurrINI.scrsz(4)-120)/2 500 120], ...
         'Tag','GraphDataDLG', ...
         'Visible','off', ...
         'Resize','off', ...
         'Name',['Graph Observed Variables - ' AnnualStr ' Data'], ...
         'NumberTitle','off', ...
         'CloseRequestFcn','GraphDataDLG done', ...
         'ToolBar','none');

      if MatlabNumber>=7.0;
         set(GraphGUI,'DockControl','off');
      end;
      DSGEModel.AnnualStr = AnnualStr;
      %
      % Outer axis
      %
      AxesBox([2 2 498 110],'Graphics',45,[0.5 1],'on',CurrINI);
      %
      if (strcmp(AnnualStr,'Original')==1)|(strcmp(AnnualStr,'Transformed')==1);
         NewStartYear = DSGEModel.SubBeginYear;
         NewStartPeriod = DSGEModel.SubBeginPeriod;
      else;
         [NewStartYear,NewStartPeriod] = AdjustSampleStart(DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.DataFrequency,max(DSGEModel.annual)-1);
      end;
      %
      OldDSGEModel = DSGEModel;
      DSGEModel.SubBeginYear = NewStartYear;
      DSGEModel.SubBeginPeriod = NewStartPeriod;
      %
      [FirstPeriod,LastPeriod,SampleStrMat] = CreateSubSample(DSGEModel);
      %
      DSGEModel = OldDSGEModel;
      %
      % Sample selection controls
      %
      controls.sampletxt = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 50 360 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'String','Selected Sample:', ...
         'HorizontalAlignment','left', ...
         'Tag','SampleText');
      %
      % sample start and sample end
      %
      controls.samplebeg = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[185 55 87 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'String',SampleStrMat, ...
         'HorizontalAlignment','center', ...
         'Callback','GraphDataDLG samplebegin', ...
         'Tag','SampleBeginPopup',...
         'Value',1);
      %
      controls.sampleend = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[278 55 87 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'String',SampleStrMat, ...
         'HorizontalAlignment','center', ...
         'Callback','GraphDataDLG sampleend', ...
         'Tag','SampleEndPopup',...
         'Value',size(SampleStrMat,1));
      %
%      controls.sample = uicontrol('Units','pixels', ...
%         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
%         'Position',[24 50 360 20], ...
%         'FontSize',CurrINI.gui_fontsize, ...
%         'FontName',CurrINI.gui_fontname, ...
%         'FontWeight',CurrINI.gui_fontweight, ...
%         'FontAngle',CurrINI.gui_fontangle, ...
%         'Style','text', ...
%         'String',['Selected Sample: ' NewStartYear ':' NewStartPeriod ' - ' DSGEModel.SubEndYear ':' DSGEModel.SubEndPeriod], ...
%         'HorizontalAlignment','left', ...
%         'Tag','SampleText');
      %
      % Select shock
      %
      controls.variablestext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 16 160 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'UserData',DSGEModel, ...
         'String','Select variable:', ...
         'HorizontalAlignment','left', ...
         'Tag','ObsVarText');
      %
      controls.variables = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[185 20 180 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',DSGEModel.VariableNames, ...
         'HorizontalAlignment','center', ...
         'Tag','ObsVarPopup',...
         'Value',1);
      %
      % Create graph button
      %
      controls.show = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[395 55 90 20], ...
         'String','Display', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','GraphDataDLG showgraph', ...
         'Enable','on', ...
         'Tag','Done');
      %
      %  Done button
      %
      controls.done = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[395 20 90 20], ...
         'String','Done', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','GraphDataDLG done', ...
         'Enable','on', ...
         'UserData','waiting', ...
         'Tag','Done');

      set(GraphGUI,'UserData',controls,...
         'HandleVisibility','callback');
   case 'samplebegin'
      %
      % check to make sure that the sample begins before it ends!
      %
      samplebeg = get(controls.samplebeg,'Value');
      sampleend = get(controls.sampleend,'Value');
      if sampleend<samplebeg;
         set(controls.sampleend,'Value',samplebeg);
      end;
   case 'sampleend'
      %
      % check to make sure that the sample begins before it ends
      %
      samplebeg = get(controls.samplebeg,'Value');
      sampleend = get(controls.sampleend,'Value');
      if sampleend<samplebeg;
         set(controls.samplebeg,'Value',sampleend);
      end;
   case 'showgraph'
      VariableValue = get(controls.variables,'Value');
      CurrVariable = StringTrim(DSGEModel.VariableNames(VariableValue,:));
      if VariableValue<size(DSGEModel.VariableNames,1);
         set(controls.variables,'Value',VariableValue+1);
      else;
         set(controls.variables,'Value',1);
      end;
      %
      % create a vector with dates
      %
      DateVector = MakeDateVector(DSGEModel.SubBeginYear,DSGEModel.SubBeginPeriod,DSGEModel.SubEndYear,DSGEModel.SubEndPeriod,DSGEModel.DataFrequency);
      %
      % determine the sub-sample from the full sample
      %
      [FirstPeriod,LastPeriod] = CreateSubSample(DSGEModel);
      %
      % create the figure
      %
      FigHandle =figure('Visible','off', ...
                'Position',[(CurrINI.scrsz(3)-650)/2 (CurrINI.scrsz(4)-500)/2 650 500], ...
                'Tag',CurrVariable, ...
                'Name',['Graph of ' CurrVariable]);
      %
      SetFigureProperties(FigHandle,CurrINI);
      %
      StartSample = get(controls.samplebeg,'Value');
      EndSample = get(controls.sampleend,'Value');
      %
      % start up a plot
      %
      if strcmp(DSGEModel.AnnualStr,'Original')==1;
         y = DSGEModel.Y(VariableValue,FirstPeriod:LastPeriod);
      elseif strcmp(DSGEModel.AnnualStr,'Transformed')==1;
         y = DSGEModel.TransformedY(VariableValue,FirstPeriod:LastPeriod);
      else;
         Y = Annualize(DSGEModel.Y(:,FirstPeriod:LastPeriod),DSGEModel.annual,DSGEModel.annualscale);
         y = Y(VariableValue,:);
         DateVector = DateVector(1,length(DateVector)-length(y)+1:length(DateVector));
      end;
      LineHandle = plot(DateVector(StartSample:EndSample),y(StartSample:EndSample),'.-','Color',[0.75 0 0],'LineWidth',2);
      if (strcmp(DSGEModel.AnnualStr,'Original')==1)|((DSGEModel.annual(VariableValue)==1)&(strcmp(DSGEModel.AnnualStr,'Transformed')==0));
         SetAxesFonts(CurrINI,'Title',CurrVariable);
      elseif strcmp(DSGEModel.AnnualStr,'Transformed')==1;
         SetAxesFonts(CurrINI,'Title',['Transformed ' CurrVariable]);
      else;
         SetAxesFonts(CurrINI,'Title',['Annualized ' CurrVariable]);
      end;
      axis('tight');
      %
      % see if we should add a zero line
      %
      ylim = get(gca,'YLim');
      add_on = 0.1*(ylim(2)-ylim(1));
      ylim = [ylim(1)-add_on ylim(2)+add_on];
      set(gca,'YLim',ylim);
      if (ylim(1)<0)&(ylim(2)>0);
         hold('on');
         plot([DateVector(StartSample) DateVector(EndSample)],[0 0],'-','Color',[0 0 0]);
         %
         % make sure the zero line is displayed below the observed variable
         %
         kids = get(gca,'Children');
         kids = [kids(2) kids(1)]';
         set(gca,'Children',kids);
      end;
      hold('off');
      %
      SetXTickIntegers(gca);
      %
      % display the graph
      %
      SetToolbarImages(FigHandle,CurrINI.images);
      set(FigHandle,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
      drawnow;
      pause(0.02);
   case 'done'
      set(controls.done,'UserData','done');
      delete(GraphGUI);
      drawnow;
      pause(0.02);
end;
%
if strcmp(selector,'init')==1;
   set(GraphGUI,'Visible','on');
   drawnow;
   if MatlabNumber>=7;
      uicontrol(controls.done);
   end;
   waitfor(controls.done,'UserData','done');
end;

%
% end of GraphDataDLG.m
%
