function [status,file] = FileBrowse(file,header,UIHandle,filetype,CurrINI)
% FileBrowse: Browse for a file
%
% USAGE:
%
%       file = FileBrowse(file,header,handle,filetype)
%
% REQUIRED INPUTS:  file (string vector) can be a directoty or a valid file
%
%                   header (string vector) Header string of File open dialog
%
%                   UIHandle (real number) handle to the control where the file should
%                            be stored in a string
%
%                   filetype (string vector) the type of files to browse for
%
%                   CurrINI (structure)
%
% REQUIRED OUTPUTS: status (boolean) is 1 if the selected file exists, 0 if cancel
%                            button for pushed or file does not exist
%
% OPTIONAL OUTPUT:  file (string) name of file that was selected.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: July 25, 2006.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 26-7-2006:  Added m-files to selection type, with special considerations.
%
% * 1-8-2006:   Added the possibility of browsing for DSGE Model files and retrieving the selected file directly.
%
% * 2-8-2006:   Swapped filetype txt to aim, thereby using .aim as an extension for AiM Model Files.
%
% * 9-11-2007:  Updated the documentation.
%
% * 13-5-2008:  The filetype "m" can now also include p-files.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

status = 0;
if (isempty(UIHandle)==0)&(ishandle(UIHandle)==0);
   return;
end;
%
% set up patters
%
if strcmp(filetype,'aim')==1;
   %
   % we shall primarily browse for text files
   %
   if MatlabNumber>=6;
      if FileExist(file)==1;
         [default_path,default_file,default_ext] = fileparts(file);
         default_file = file;
         if (strcmp(lower(default_ext),'.txt')==0)&(strcmp(lower(default_ext),'.aim')==0);
            default_type = [upper(strrep(default_ext,'.','')) ' Files (*' default_ext ')'];
            file = {['*' default_ext], default_type; ...
                    ['*.aim;*.txt;*' default_ext], ['Model Files (*.aim, *.txt, *' default_ext ')']; ...
                    '*.aim', 'AiM Model Files (*.aim)'; ...
                    '*.txt', 'Text Files (*.txt)'; ...
                    '*.*', 'All Files (*.*)'};
         else;
            file = {'*.aim;*.txt', 'Model Files (*.aim, *.txt)'; ...
                    '*.aim', 'AiM Model Files (*.aim)'; ...
                    '*.txt', 'Text Files (*.txt)'; ...
                    '*.*', 'All Files (*.*)'};
         end;
      else;
         file = {'*.aim;*.txt', 'Model Files (*.aim, *.txt)'; ...
                 '*.aim', 'AiM Model Files (*.aim)'; ...
                 '*.txt', 'Text Files (*.txt)'; ...
                 '*.*', 'All Files (*.*)'};
         default_file = '';
      end;
   else;
      if FileExist(file)==0;
         file = [pwd '\*.txt'];
      end;
   end;
elseif strcmp(filetype,'m')==1;
   %
   % we shall browse only for matlab m files
   %
   if MatlabNumber>=6;
      if FileExist(file)==1;
         default_file = file;
      else;
         default_file = '';
      end;
      file = {'*.m', 'Matlab M-Files (*.m)'; ...
              '*.p', 'Matlab P-Files (*.p)'};
   else;
      if FileExist(file)==0;
         file = [pwd '\*.m;*.p'];
      end;
   end;
elseif strcmp(filetype,'dsge')==1;
   %
   % we shall browse only for DSGE model files
   %
   if MatlabNumber>=6;
      if FileExist(file)==1;
         default_file = file;
      else;
         default_file = '';
      end;
      file = {'*.dsge', 'DSGE Model Files (*.dsge)'};
   else;
      if FileExist(file)==0;
         file = [pwd '\*.dsge'];
      end;
   end;
else;
   %
   % we shall primarily browse for spreadsheets
   %
   if MatlabNumber>=6;
      if FileExist(file)==1;
         default_file = file;
      else;
         default_file = '';
      end;
      file = {'*.wk1;*.xls', 'Spreadsheet Files (*.wk1, *.xls)'; ...
                 '*.wk1', 'Lotus 1-2-3 Spreadsheet (*.wk1)'; ...
                 '*.xls', 'Excel Spreadsheet (*.xls)'};
   else;
      if FileExist(file)==0;
         file = [pwd '\*.wk1'];
      end;
   end;
end;
%
% open the fileopen dialog
%
if MatlabNumber>=6;
   [filename,filepath] = uigetfile(file,header,default_file);
else;
   [filename,filepath] = uigetfile(file,header);
end;
drawnow;
drawnow;
if (isnumeric(filename)==0)&(FileExist([filepath filename])==1);
   [new_path,new_file,new_ext] = fileparts([filepath filename]);
   if (MatlabNumber<6)&(strcmp(lower(new_ext),'.xls')==1);
      if nargout==2;
         file = '';
      end;
      txt = 'Your version of Matlab does not support Excel spreadsheets. Please use a different file format.';
      About(txt,'information','Invalid File Format',100,400,CurrINI);
   elseif (strcmp(filetype,'m')==1)&(strcmp(lower(new_ext),'.m')==0)&(strcmp(lower(new_ext),'.p')==0);
      if nargout==2;
         file = '';
      end;
      txt = 'The file must be either a Matlab m-file or a p-file since it will be run by YADA.';
      About(txt,'information','Invalid File Format',100,400,CurrINI);
   elseif (strcmp(filetype,'dsge')==1)&(strcmp(lower(new_ext),'.dsge')==0);
      if nargout==2;
         file = '';
      end;
      txt = 'The file must be a DSGE model file since it will be read for input by YADA.';
      About(txt,'information','Invalid File Format',100,400,CurrINI);
   else;
      status = 1;
      if nargout==1;
         set(UIHandle,'String',[filepath filename]);
      else;
         file = [filepath filename];
      end;
   end;
else;
   if nargout==2;
      file = '';
   end;
end;

%
% end of FileBrowse.m
%
