function FrameHandles = AxesBox(PosVec,TitleStr,TitleWidth,BaseColors,VisibilityStr,CurrINI)
% AxesBox: Creates a frame on a figure.
%
% USAGE:
%
%       FrameHandles = AxesBox(PosVec,TitleStr,TitleWidth,BaseColors,VisibilityStr,CurrINI)
%
% REQUIRED INPUT:  PosVec (vector) of dimension 4 with the left, bottom, width, and height
%                         integer values (pixels) for the frame.
%
%                  TitleStr (string vector) with the title for the frame.
%
%                  TitleWidth (integer) with the width (in pixels) of the title. This width
%                         is later fixed by the function such that it properly matches the
%                         extent of the text object.
%
%                  BaseColors (vector) of diemsnion 2 or 4 with base colors for dark and light
%                         lines that make up the frame.
%
%                  VisibilityStr (string vector) that can take on the values "on" or "off".
%
%                  CurrINI (structure) with initialization information.
%
% OPTIONAL OUTPUT: FrameHandles (vector) of handles to the various objects making up the frame.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: July 21, 2006.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 20-5-2008:  Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% get the dimensions of the frame
%
left = PosVec(1);
bottom = PosVec(2);
width = PosVec(3);
height = PosVec(4);
%
DarkColor = BaseColors(1);
LightColor = BaseColors(2);
if size(BaseColors,2)==4;
   dark2 = BaseColors(3);
   light2 = BaseColors(4);
else;
   dark2 = DarkColor;
   light2 = LightColor;
end;
%
% create the different grey-scaled color vectors
%
lightvec = [LightColor LightColor LightColor];
darkvec = [DarkColor DarkColor DarkColor];
lightvec2 = [light2 light2 light2];
darkvec2 = [dark2 dark2 dark2];
%
if size(BaseColors,2)==2;
   if IsAlmostEqual(darkvec,CurrINI.frame_shade_color)==0;
      darkvec = CurrINI.frame_shade_color;
      darkvec2 = darkvec;
   end;
   if IsAlmostEqual(lightvec,CurrINI.frame_light_color)==0;
      lightvec = CurrINI.frame_light_color;
      lightvec2 = lightvec;
   end;
else;
   if IsAlmostEqual(darkvec,CurrINI.gui_shade_color)==0;
      darkvec = CurrINI.gui_shade_color;
   end;
   if IsAlmostEqual(lightvec,CurrINI.gui_dark_color)==0;
      lightvec = CurrINI.gui_dark_color;
   end;
   if IsAlmostEqual(lightvec2,CurrINI.gui_light_color)==0;
      lightvec2 = CurrINI.gui_light_color;
   end;
end;
%
fixed = 0;
if MatlabNumber>=6.0;
   fixed = 1;
end;
%
% setting up the axes
%
%
% the bottom x axis
%
h1 = axes('Units','pixels', ...
   'TickLength',[0 0], ...
   'CameraUpVector',[0 1 0], ...
   'Color',get(0,'defaultuicontrolbackgroundcolor'), ...
   'Position',[left+1 bottom width 1], ...
   'XTickLabelMode','manual', ...
   'YTickLabelMode','manual', ...
   'Tag','frame', ...
   'Visible',VisibilityStr, ...
   'XColor',lightvec2, ...
   'YColor',lightvec2, ...
   'ZColor',lightvec2);
%
h2 = axes('Units','pixels', ...
   'TickLength',[0 0], ...
   'CameraUpVector',[0 1 0], ...
   'Color',get(0,'defaultuicontrolbackgroundcolor'), ...
   'Position',[left bottom+1 width-1 1], ...
   'XTickLabelMode','manual', ...
   'YTickLabelMode','manual', ...
   'Tag','frame', ...
   'Visible',VisibilityStr, ...
   'XColor',darkvec2, ...
   'YColor',darkvec2, ...
   'ZColor',darkvec2);
%
% the left y axis
%
h3 = axes('Units','pixels', ...
   'TickLength',[0 0], ...
   'CameraUpVector',[0 1 0], ...
   'Color',get(0,'defaultuicontrolbackgroundcolor'), ...
   'Position',[left bottom+1 1 height-1], ...
   'XTickLabelMode','manual', ...
   'YTickLabelMode','manual', ...
   'Tag','frame', ...
   'Visible',VisibilityStr, ...
   'XColor',darkvec, ...
   'YColor',darkvec, ...
   'ZColor',darkvec);
%
h4 = axes('Units','pixels', ...
   'TickLength',[0 0], ...
   'CameraUpVector',[0 1 0], ...
   'Color',get(0,'defaultuicontrolbackgroundcolor'), ...
   'Position',[left+1 bottom+2 1 height-2], ...
   'XTickLabelMode','manual', ...
   'YTickLabelMode','manual', ...
   'Tag','frame', ...
   'Visible',VisibilityStr, ...
   'XColor',lightvec, ...
   'YColor',lightvec, ...
   'ZColor',lightvec);
%
% the top x axis
%
h5 = axes('Units','pixels', ...
   'TickLength',[0 0], ...
   'CameraUpVector',[0 1 0], ...
   'Color',get(0,'defaultuicontrolbackgroundcolor'), ...
   'Position',[left+1 bottom+height-1 width-2 1], ...
   'XTickLabelMode','manual', ...
   'YTickLabelMode','manual', ...
   'Tag','frame', ...
   'Visible',VisibilityStr, ...
   'XColor',lightvec, ...
   'YColor',lightvec, ...
   'ZColor',lightvec);
%
h6 = axes('Units','pixels', ...
   'TickLength',[0 0], ...
   'CameraUpVector',[0 1 0], ...
   'Color',get(0,'defaultuicontrolbackgroundcolor'), ...
   'Position',[left bottom+height width-1 1], ...
   'XTickLabelMode','manual', ...
   'YTickLabelMode','manual', ...
   'Tag','frame', ...
   'Visible',VisibilityStr, ...
   'XColor',darkvec, ...
   'YColor',darkvec, ...
   'ZColor',darkvec);
%
if MatlabNumber>=6.0;
   h9 = axes('Units','pixels', ...
      'TickLength',[0 0], ...
      'CameraUpVector',[0 1 0], ...
      'Color',get(0,'defaultuicontrolbackgroundcolor'), ...
      'Position',[left bottom+height+1 1 1], ...
      'XTickLabelMode','manual', ...
      'YTickLabelMode','manual', ...
      'Tag','frame', ...
      'Visible',VisibilityStr, ...
      'XColor',get(0,'defaultuicontrolbackgroundcolor'), ...
      'YColor',get(0,'defaultuicontrolbackgroundcolor'), ...
      'ZColor',get(0,'defaultuicontrolbackgroundcolor'));
else;
   h9 = [];
end;
%
% the right y axis
%
h7 = axes('Units','pixels', ...
   'TickLength',[0 0], ...
   'CameraUpVector',[0 1 0], ...
   'Color',get(0,'defaultuicontrolbackgroundcolor'), ...
   'Position',[left+width-1 bottom+1 1 height-fixed], ...
   'XTickLabelMode','manual', ...
   'YTickLabelMode','manual', ...
   'Tag','frame', ...
   'Visible',VisibilityStr, ...
   'XColor',darkvec2, ...
   'YColor',darkvec2, ...
   'ZColor',darkvec2);
%
h8 = axes('Units','pixels', ...
   'TickLength',[0 0], ...
   'CameraUpVector',[0 1 0], ...
   'Color',get(0,'defaultuicontrolbackgroundcolor'), ...
   'Position',[left+width bottom 1 height-fixed], ...
   'XTickLabelMode','manual', ...
   'YTickLabelMode','manual', ...
   'Tag','frame', ...
   'Visible',VisibilityStr, ...
   'XColor',lightvec2, ...
   'YColor',lightvec2, ...
   'ZColor',lightvec2);
%
if MatlabNumber>=6.0;
   h10 = axes('Units','pixels', ...
      'TickLength',[0 0], ...
      'CameraUpVector',[0 1 0], ...
      'Color',get(0,'defaultuicontrolbackgroundcolor'), ...
      'Position',[left+width+1 bottom-1 1 1], ...
      'XTickLabelMode','manual', ...
      'YTickLabelMode','manual', ...
      'Tag','frame', ...
      'Visible',VisibilityStr, ...
      'XColor',get(0,'defaultuicontrolbackgroundcolor'), ...
      'YColor',get(0,'defaultuicontrolbackgroundcolor'), ...
      'ZColor',get(0,'defaultuicontrolbackgroundcolor'));
else;
   h10 = [];
end;
%
if TitleWidth>0;
   h11 = uicontrol('Units','pixels', ...
      'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
      'Position',[left+10 bottom+height-8 TitleWidth 15], ...
      'ForegroundColor',CurrINI.frame_color, ...
      'FontSize',CurrINI.gui_fontsize, ...
      'FontName',CurrINI.gui_fontname, ...
      'FontWeight',CurrINI.gui_fontweight, ...
      'FontAngle',CurrINI.gui_fontangle, ...
      'Style','text', ...
      'UserData',bottom+height-8, ...
      'String',TitleStr, ...
      'Visible',VisibilityStr, ...
      'Tag','FrameText');
   %
   % fix based on extent property
   %
   txt_extent = get(h11,'Extent');
   set(h11,'Position',[left+10 bottom+height-8+((15-txt_extent(1,4))/2) txt_extent(1,3:4)]);
   if IsWinXP==1;
      set(h11,'Position',[left+10 bottom+height-8-(txt_extent(1,4)/4) txt_extent(1,3:4)],'UserData',bottom+height-15.5+(txt_extent(1,4)/4));
   end;
else;
   h11 = [];
end;
%
% provide the output if desired
%
if nargout==1;
   FrameHandles = [h1 h2 h3 h4 h5 h6 h7 h8 h9 h10 h11];
end;

%
% end of AxesBox.m
%
