function AutoCorrelationDistributionDLG(selector,SVEs,AutoCorrType,NameStr,BoxStr,DSGEModel,CurrINI)
% AutoCorrelationDistributionDLG: Displays plots of the autocorelation function based on the
%                                 DSGE model along with the results taken directly from the data.
%
% USAGE:
%
%       AutoCorrelationDistributionDLG(selector,SVEs,AutoCorrType,NameStr,BoxStr,DSGEModel,CurrINI)
%
% REQUIRED INPUT: selector (string vector) taking on the values "init", "showgraph", and "done".
%
% OPTIONAL INPUT: SVEs (structure) with fields "Data.Mean", "Data.Std", "Data.Corr", "Mean", "Std",
%                      "Corr", "AutoCovHorizon", and "Quantiles". Required if selector is "init".
%
%                 AutoCorrType (string vector) that is prepended to the tilte of the axes object.
%
%                 NameStr (string vector) with the text for the window name.
%
%                 BoxStr (string vector) with the text for axesbox.
%
%                 DSGEModel (structure) with fields containing all the information about a DSGE Model for YADA.
%
%                 CurrINI (structure) with initialization information.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: November 30, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 29-2-2008:  Added the input "AutoCorrType", "NameStr", and "BoxStr". Quantiles are now only plotted if they
%               exist.
%
% * 23-5-2008:  Updated the documentation.
%
% * 11-7-2008:  Changed the ordering of colors for confidence bands.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(selector,'init')==1;
   DSGEModel.SVEs = SVEs;
   DSGEModel.AutoCorrType = AutoCorrType;
else;
   AutoCorrelationDistributionGUI = findobj('Type','figure','Tag','AutoCorrelationDistributionDLG');
   controls = get(AutoCorrelationDistributionGUI,'UserData');
   CurrINI = get(controls.firstvariable,'UserData');
   DSGEModel = get(controls.firstvariabletext,'UserData');
end;
%
switch selector;
   case 'init';
      AutoCorrelationDistributionGUI = figure('Color',get(0,'defaultuicontrolbackgroundcolor'), ...
         'FileName','AutoCorrelationDistributionDLG.m', ...
         'MenuBar','none', ...
         'PaperUnits','points', ...
         'Units','pixels', ...
         'Position',[(CurrINI.scrsz(3)-525)/2 (CurrINI.scrsz(4)-120)/2 525 120], ...
         'Tag','AutoCorrelationDistributionDLG', ...
         'Visible','off', ...
         'Resize','off', ...
         'Name',NameStr, ...
         'NumberTitle','off', ...
         'CloseRequestFcn','AutoCorrelationDistributionDLG done', ...
         'ToolBar','none');
      %
      if MatlabNumber>=7.0;
         set(AutoCorrelationDistributionGUI,'DockControl','off');
      end;
      %
      % Outer axis
      %
      AxesBox([2 2 523 110],BoxStr,45,[0.5 1],'on',CurrINI);
      %
      %
      % Select variable 1
      %
      controls.firstvariabletext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 45 180 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'UserData',DSGEModel, ...
         'String','Select first variable:', ...
         'HorizontalAlignment','left', ...
         'Tag','ObsVar1Text');
      %
      controls.firstvariable = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[210 50 180 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',DSGEModel.VariableNames, ...
         'HorizontalAlignment','center', ...
         'Tag','ObsVar1Popup',...
         'Value',1);
      %
      % select variable 2
      %
      controls.secondvariabletext = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[24 16 180 20], ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'Style','text', ...
         'UserData',DSGEModel, ...
         'String','Select second variable:', ...
         'HorizontalAlignment','left', ...
         'Tag','ObsVar2Text');
      %
      controls.secondvariable = uicontrol('Units','pixels', ...
         'BackgroundColor',[1 1 1], ...
         'Position',[210 20 180 20], ...
         'Style','popupmenu', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'UserData',CurrINI, ...
         'String',DSGEModel.VariableNames, ...
         'HorizontalAlignment','center', ...
         'Tag','ObsVar2Popup',...
         'Value',1);
      %
      % Create graph button
      %
      controls.show = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[420 55 90 20], ...
         'String','Display', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','AutoCorrelationDistributionDLG showgraph', ...
         'Enable','on', ...
         'Tag','Done');
      %
      %  Done button
      %
      controls.done = uicontrol('Units','pixels', ...
         'BackgroundColor',get(0,'defaultuicontrolbackgroundcolor'), ...
         'Position',[420 20 90 20], ...
         'String','Done', ...
         'FontSize',CurrINI.gui_fontsize, ...
         'FontName',CurrINI.gui_fontname, ...
         'FontWeight',CurrINI.gui_fontweight, ...
         'FontAngle',CurrINI.gui_fontangle, ...
         'CallBack','AutoCorrelationDistributionDLG done', ...
         'Enable','on', ...
         'UserData','waiting', ...
         'Tag','Done');

      set(AutoCorrelationDistributionGUI,'UserData',controls,...
         'HandleVisibility','callback');
   case 'showgraph'
      FirstValue = get(controls.firstvariable,'Value');
      SecondValue = get(controls.secondvariable,'Value');
      VarStrMat = get(controls.firstvariable,'String');
      n = size(VarStrMat,1);
      FirstVariable = StringTrim(VarStrMat(FirstValue,:));
      SecondVariable = StringTrim(VarStrMat(SecondValue,:));
      if FirstValue<n;
         NewSecondValue = SecondValue;
         NewFirstValue = FirstValue+1;
      else;
         if SecondValue<n;
            NewFirstValue = SecondValue+1;
            NewSecondValue = SecondValue+1;
         else;
            NewFirstValue = 1;
            NewSecondValue = 1;
         end;
      end;
      set(controls.firstvariable,'Value',NewFirstValue);
      set(controls.secondvariable,'Value',NewSecondValue);
      %
      % determine which entry we should take
      %
      NumberMat = reshape(DuplicationMatrix(n)*(1:((n*(n+1))/2))',n,n);
      %
      % check if we have to flip the autocorrelations or not
      %
      if FirstValue>=SecondValue;
         AutoCovValue = NumberMat(FirstValue,SecondValue);
         DoFlip = 0;
      else;
         AutoCovValue = NumberMat(SecondValue,FirstValue);
         DoFlip = 1;
      end;
      %
      % create the figure
      %
      FigHandle = figure('Visible','off', ...
                     'Position',[(CurrINI.scrsz(3)-650)/2 (CurrINI.scrsz(4)-500)/2 650 500], ...
                     'Tag',[FirstVariable '-' SecondVariable], ...
                     'Name',['Autocorrelations - ' FirstVariable '(t-h) and ' SecondVariable '(t)']);
      %
      SetFigureProperties(FigHandle,CurrINI);
      %
      if isfield(DSGEModel.SVEs,'Quantiles')==1;
         %
         % deal with confidence bands for the update estimates
         %
         NumBounds = floor(length(DSGEModel.SVEs.Quantiles)/2);
         Handles = [];
         LegendStr = '';
         %
         for nb=1:NumBounds;
            if DoFlip==0;
               Lower = DSGEModel.SVEs.Quantiles(nb).Corr(AutoCovValue,:);
               Upper = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).Corr(AutoCovValue,:);
            else;
               Lower = fliplr(DSGEModel.SVEs.Quantiles(nb).Corr(AutoCovValue,:));
               Upper = fliplr(DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).Corr(AutoCovValue,:));
            end;
            PatchColor = ((1+(2*(nb-1)))/length(DSGEModel.Percentiles))*DSGEModel.ConfidenceBandBaseColor;
            ConfBound = DSGEModel.SVEs.Quantiles(length(DSGEModel.Percentiles)-nb+1).percent-DSGEModel.SVEs.Quantiles(nb).percent;
            ConfPatch = patch([DSGEModel.SVEs.AutoCovHorizon fliplr(DSGEModel.SVEs.AutoCovHorizon)],[Upper fliplr(Lower)],PatchColor);
            set(ConfPatch,'FaceColor',PatchColor,'EdgeColor',PatchColor);
            hold('on');
            Handles = [Handles ConfPatch];
            LegendStr = strvcat(LegendStr,[num2str(ConfBound,'%0.0f') ' confidence']);
         end;
         set(gca,'Layer','Top');
      else;
         Handles = [];
         LegendStr = '';
      end;
      %
      % plot a zero line
      %
      plot([min(DSGEModel.SVEs.AutoCovHorizon) max(DSGEModel.SVEs.AutoCovHorizon)],[0 0],'-','Color',[0 0 0]);
      hold('on');
      %
      % plot the mean autocorrelations
      %
      if DoFlip==0;
         postmean = plot(DSGEModel.SVEs.AutoCovHorizon,DSGEModel.SVEs.Corr(AutoCovValue,:),'--','Color',[0 0 1],'LineWidth',2);
      else;
         postmean = plot(DSGEModel.SVEs.AutoCovHorizon,fliplr(DSGEModel.SVEs.Corr(AutoCovValue,:)),'--','Color',[0 0 1],'LineWidth',2);
      end;
      hold('on');
      %
      % plot the data autocorrelations
      %
      if DoFlip==0;
         datamean = plot(DSGEModel.SVEs.AutoCovHorizon,DSGEModel.SVEs.Data.Corr(AutoCovValue,:),'.-','Color',[0.75 0 0],'LineWidth',2);
      else;
         datamean = plot(DSGEModel.SVEs.AutoCovHorizon,fliplr(DSGEModel.SVEs.Data.Corr(AutoCovValue,:)),'.-','Color',[0.75 0 0],'LineWidth',2);
      end;
      hold('on');
      %
      % fix the limits
      %
      set(gca,'XLim',[min(DSGEModel.SVEs.AutoCovHorizon) max(DSGEModel.SVEs.AutoCovHorizon)],'YLim',[-1 1]);
      LegendStr = strvcat('Data','Mean',LegendStr);
      Handles = [datamean postmean Handles];
      hold('off');
      %
      % set the title
      %
      SetAxesFonts(CurrINI,'Title',[DSGEModel.AutoCorrType ' autocorrelations for ' FirstVariable '(t-h) and ' SecondVariable '(t)'],'XLabel','horizon (h)');
      %
      legHandle = legend(Handles,LegendStr,2);
      SetAxesFonts(CurrINI,legHandle);
      %
      SetXTickIntegers(gca);
      %
      SetToolbarImages(FigHandle,CurrINI.images);
      set(FigHandle,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
      drawnow;
      pause(0.02);
   case 'done'
      set(controls.done,'UserData','done');
      delete(AutoCorrelationDistributionGUI);
      drawnow;
      pause(0.02);
end;
if strcmp(selector,'init')==1;
   set(AutoCorrelationDistributionGUI,'Visible','on');
   drawnow;
   drawnow;
   if MatlabNumber>=7;
      uicontrol(controls.done);
   end;
   waitfor(controls.done,'UserData','done');
end;

%
% end of AutoCorrelationDistributionDLG.m
%
