function AcceptanceRatioPlot(AcceptedDraws,NumBurnin,CurrChain,DSGEModel,CurrINI)
% AcceptanceRatioPlot: Plots the acceptance ratio from the random walk metropolis algorithm
%                      over the posterior draws.
%
% USAGE:
%
%       AcceptanceRatioPlot(AcceptedDraws,NumBurnin,CurrChain,DSGEModel,CurrINI)
%
% REQUIRED INPUT: AcceptedDraws (vector) of dimension (NumIterx1) with the number of accepted draws.
%
%                 NumBurnin (integer) with the number of burn-in draws.
%
%                 CurrChain (integer) with the number for the current MCMC chain.
%
%                 DSGEModel (structure)  whose fields contain all required information
%                                 about the model, such as input files, variable and sample
%                                 data. Actual observations are not stored in this file.
%
%                 CurrINI (structure) with initialization information.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: April 17, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 18-4-2007:  Updated the documentation.
%
% * 9-11-2007:  Updated the documentation.
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 10-3-2008:  Made sure that the "CurrChain" variable affects the filename for the figure.
%
% * 27-3-2008:  Made sure that the sequential grid has at least 2 values.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% create the acceptance ratio
%
NumIter = length(AcceptedDraws);
AcceptedDraws = AcceptedDraws' ./ (1:1:NumIter);
%
% locate the controls structure
%
maingui = findobj('Type','figure','Tag','YADA');
controls = get(maingui,'UserData');
%
% sequential marginal likelihood start value and step length
%
SeqStartStr = get(controls.dsge.sequentialstart,'String');
Origin = str2num(strrep(StringTrim(SeqStartStr(DSGEModel.SequentialStartIteration,:)),',',''));
SeqStepStr = get(controls.dsge.sequentialstep,'String');
StepLength = str2num(strrep(StringTrim(SeqStepStr(DSGEModel.SequentialStepLength,:)),',',''));
%
% make sure that Origin is less than NumIter
%
if Origin==NumIter;
   Origin = floor(NumIter/10);
end;
%
% Create the sequential grid over which we will compute the marginal likelihood
%
SeqGrid = (Origin:StepLength:NumIter);
if isempty(SeqGrid)==1;
   SeqGrid = NumIter;
elseif SeqGrid(length(SeqGrid))<NumIter;
   SeqGrid = [SeqGrid NumIter];
end;
%
% select the subset of AcceptedDraws
%
AcceptedDraws = AcceptedDraws(SeqGrid);
%
% prepare the figure
%
GraphDirectory = [DSGEModel.OutputDirectory '\graphs'];
%
FigHandle =figure('Visible','off', ...
       'Position',[(CurrINI.scrsz(3)-700)/2 (CurrINI.scrsz(4)-500)/2 700 500], ...
       'Tag','MargLikeCJ', ...
       'Name',['Sequential Acceptance Ratio - ' DSGEModel.NameOfModel ' Chain ' int2str(CurrChain)]);
%
SetFigureProperties(FigHandle,CurrINI);
%
plot(SeqGrid,AcceptedDraws,'LineStyle','-','Color',[0.75 0 0],'LineWidth',2,'Marker','.');
hold('on');
%
% check the YLim
%
ylim = get(gca,'YLim');
add_on = 2.0*(ylim(2)-ylim(1));
ylim = [ylim(1)-add_on ylim(2)+add_on];
set(gca,'XLim',[SeqGrid(1) NumIter],'YLim',[max(ylim(1),0) min(ylim(2),1)]);
%
% plot the number of burnin draws
%
plot(NumBurnin*ones(1,2),get(gca,'YLim'),'-','Color',[0 0 0],'LineWidth',1);
hold('off');
%
SetAxesFonts(CurrINI,'Title','Sequential acceptance ratio','XLabel',['RWM draw number (with ' num2str(NumBurnin,'%0.0f') ' burn-in draws)'],'YLabel','Ratio');
%
% save the grasph
%
if DirExist(GraphDirectory);
   file = [GraphDirectory '\SequentialAcceptanceRatio-' num2str(NumIter,'%0.0f') '-' int2str(CurrChain) '.fig'];
   saveas(FigHandle,file);
end;
%
% display the graph
%
SetToolbarImages(FigHandle,CurrINI.images);
set(FigHandle,'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
drawnow;

%
% end of AcceptanceRatioPlot.m
%
