function [S,L] = SchurDecomposition(Sigma)
% SchurDecomposition: Decomposes a covarince matrix based on the Schur decomposition.
%
% USAGE:
%
%       [S,L] = SchurDecomposition(Sigma)
%
% REQUIRED INPUT:  Sigma (nxn matrix) that should be positive semidefinite. The rank of Sigma
%                        is s<=n.
%
% REQUIRED OUTPUT: S (nxs matrix) containing the eigenvectors of Sigma such that Sigma = S*Lambda*S',
%                        where S'*S = eye(s), and Lambda is an sxs diagonal matrix with the non-zero
%                        eigenvalues of Sigma, ordered from the largest to the smallest.
%
% OPTIONAL OUTPUT: L (sxs matrix) containing the non-zero eigenvalues of Sigma, i.e., the matrix
%                        Lambda.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: October 17, 2006.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 8-11-2006:  Added the optional output L. Renamed the function from DecomposeStateMSE to SchurDecomposition.
%
% * 23-2-2007:  Made sure that L has zeros below and above the diagonal.
%
% * 5-11-2007:  Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

[S,Lambda] = schur(Sigma);
%
% make sure we sort the diagonal elements of Lambda from the largest to the smallest
% We then reorder the columns of S accordingly
%
[NewLambda,ind] = sort(diag(Lambda));
ind = flipud(ind);
Lambda = Lambda(ind,ind);
S = S(:,ind);
rstar = rank(Lambda);
S = S(:,1:rstar);
if nargout==2;
   L = diag(diag(Lambda(1:rstar,1:rstar)));
end;

%
% end of SchurDecomposition.m
%
