function SaveDSGEModel(ModelFile,DSGEModel)
% SaveDSGEModel: Save the DSGE model data to a text file that can be read by YADA.
%
% USAGE:
%
%       SaveDSGEModel(ModelFile,DSGEModel)
%
% REQUIRED INPUTS: ModelFile (string) with full path and filename
%
%                  DSGEModel (structure) whose fields contain all required information
%                            about the model, such as input files, variable and sample
%                            data. Actual observations are not stored in this file.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: August 1, 2006.
%                        This version: February 12, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 02-08-2006: Added path  and filename of AIMDataFile. Also, pwd is stripped from all paths. This path is prepended
%               in ReadDSGEModel function if first character is \ or entry is empty!
%
% * 09-08-2006: Added the fields "UseDoublingAlgorithm", "DAMaximumIterationsValue", "ShowProgress" and "ShowProgressClock"
%
% * 22-08-2006: Added the field "InitializeKalman".
%
% * 30-08-2006: Added the fields "OverwriteDraws", "MarginalLikelihoodValue", "SequentialStartIteration",
%               "SequentialStepLength", "SequentialML", "CoverageStart", "CoverageIncrement", and "CoverageEnd".
%
% * 31-10-2006: Added the fields "DAToleranceValue", "RiccatiMaxIterations", and "RiccatiToleranceValue".
%
% * 17-11-2006: Added the field "StateCovConst".
%
% * 20-11-2006: Added the fields "UseOwnInitialState" and "InitialStateValues".
%
% * 28-11-2006: Added the fields "PosteriorDrawsValue", "SampleBatchValue", "ParallelChainsValue", "BurnInValue",
%               "MHInitialScaleFactor", "MHScaleFactor", and "RandomWeightValue".
%
% * 04-12-2006: Added the fields "ShockGroupNames" and "ShockGroups".
%
% * 07-12-2006: Added the fields "OptMaxIterationsValue", "OptToleranceValue", "MaximizeAlgorithmValue", "StepLengthHessian",
%               "NumberOfGridPoints", "FiniteDifferenceHessian", "CheckOptimum", "CheckTransformedOptimum", "GridWidth",
%               "IRHorizon", "PriorKernel", "KernelDensityValue", and "KernelDensityEstimator".
%
% * 15-12-2006: Added the field "PosteriorDensityValue".
%
% * 18-12-2006: Added the field "RandomNumberValue".
%
% * 21-12-2006: Added the field "MaxForecastHorizon".
%
% * 22-12-2006: Added the field "FreqParameterReest".
%
% * 05-01-2007: Added the field "ShockGroupColors".
%
% * 11-01-2007: Added the fields "BVARLags", "HarmonicLagDecayValue", "CrossEqTightnessValue", "OverallTightnessValue",
%               "PriorType", "PriorLevelMeanValue", and "PriorDiffMeanValue".
%
% * 12-01-2007: Added the fields "OmegaPriorType", "WishartType", "VarianceTightnessValue", "WishartDFValue",
%               "SteadyStatePriorFile", "BVARY", and "BVARX".
%
% * 22-02-2007: Added the fields "NumPredPathsValue" and "PostDrawsUsageValue".
%
% * 26-02-2007: Added the field "RandomizeDraws".
%
% * 28-02-2007: Added the field "ConfidenceBandBaseColor".
%
% * 08-03-2007: Added the fields "ConditionalShocks" and "ConditionalVariables".
%
% * 03-04-2007: Added the field "PostDrawsPercentValue".
%
% * 10-04-2007: Added the field "InverseHessianEstimator".
%
% * 16-04-2007: Added the field "MaxCorrelationValue".
%
% * 25-04-2007: Added the fields "ScenarioShocks", and "ScenarioVariables".
%
% * 26-04-2007: Added the fields "AnnScenarioValue" and "ScenarioValue".
%
% * 02-05-2007: Added the field "ScenarioParameters".
%
% * 03-05-2007: Added the field "PredictionEvent".
%
% * 23-08-2007: Added the field "AdjustPredictionPaths".
%
% * 28-09-2007: Added the field "ObsVariableScenarioFile".
%
% * 25-10-2007: The field "FreqParameterReest" is no longer supported.
%
% * 26-10-2007: Added the field "ParameterScenario".
%
% * 06-11-2007: Added the field "OVSComparisonFile".
%
% * 17-12-2007: Added the field "ShockControlMethod".
%
% * 20-12-2007: Added the field "UnitRootStates".
%
% * 02-01-2008: Added the field "KalmanFirstObservation". The field "InitializeKalman" is no longer saved.
%
% * 22-01-2008: The field "OVSComparisonFile" is no longer supported.
%
% * 24-01-2008: Added the field "KsiUseCondData".
%
% * 31-01-2008: Added the fields "ShockColorStart" and "ShockColorEnd".
%
% * 18-02-2008: Added the field "ModifiedHessian".
%
% * 16-05-2008: Added the field "AIMTolerance".
%
% * 23-05-2008: Updated the documentation.
%
% * 02-07-2008: Added the field "AllowUnitRoot".
%
% * 28-07-2008: Added the field "ParameterScenarioValue".
%
% * 19-12-2008: Updated the documentation.
%
% * 29-01-2009: Added the field "InitializeHessian".
%
% * 12-02-2009: Added the field "ParameterCovMatrix".
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

f_id = fopen(ModelFile,'wt');
%
% input files, model name, and output directory
%
fprintf(f_id,'[Input Data]\n');
fprintf(f_id,'DataConstructionFile = %s\n',strrep(DSGEModel.DataConstructionFile,pwd,''));
fprintf(f_id,'MeasurementEquationFile = %s\n',strrep(DSGEModel.MeasurementEquationFile,pwd,''));
fprintf(f_id,'PriorFile = %s\n',strrep(DSGEModel.PriorFile,pwd,''));
fprintf(f_id,'PriorFileSheet = %s\n',DSGEModel.PriorFileSheet);
fprintf(f_id,'UpdateParameterFile = %s\n',strrep(DSGEModel.UpdateParameterFile,pwd,''));
fprintf(f_id,'InitializeParameterFile = %s\n',strrep(DSGEModel.InitializeParameterFile,pwd,''));
fprintf(f_id,'AIMFile = %s\n',strrep(DSGEModel.AIMFile,pwd,''));
fprintf(f_id,'AIMDataFile = %s\n',strrep(DSGEModel.AIMDataFile,pwd,''));
fprintf(f_id,'NameOfModel = %s\n',DSGEModel.NameOfModel);
fprintf(f_id,'OutputDirectory = %s\n',strrep(DSGEModel.OutputDirectory,pwd,''));
fprintf(f_id,'ObsVariableScenarioFile = %s\n',DSGEModel.ObsVariableScenarioFile);
%
% Variable data
%
fprintf(f_id,'\n[Variable Data]\n');
fprintf(f_id,'StateShockNames = %s\n',vecstr(DSGEModel.StateShockNames,'$'));
fprintf(f_id,['StateShockPositions = ' num2str(DSGEModel.StateShockPositions) '\n']);
fprintf(f_id,'StateVariableNames = %s\n',vecstr(DSGEModel.StateVariableNames,'$'));
fprintf(f_id,['StateVariablePositions = ' num2str(DSGEModel.StateVariablePositions) '\n']);
fprintf(f_id,['UnitRootStates = ' num2str(DSGEModel.UnitRootStates) '\n']);
fprintf(f_id,'ShockGroupNames = %s\n',vecstr(DSGEModel.ShockGroupNames,'$'));
fprintf(f_id,['ShockGroups = ' num2str(DSGEModel.ShockGroups) '\n']);
fprintf(f_id,'ShockGroupColors = %s\n',num2str(vec(DSGEModel.ShockGroupColors)'));
fprintf(f_id,'ShockColorStart = %s\n',num2str(vec(DSGEModel.ShockColorStart)'));
fprintf(f_id,'ShockColorEnd = %s\n',num2str(vec(DSGEModel.ShockColorEnd)'));
fprintf(f_id,['ConditionalShocks = ' num2str(DSGEModel.ConditionalShocks) '\n']);
fprintf(f_id,['ConditionalVariables = ' num2str(DSGEModel.ConditionalVariables) '\n']);
fprintf(f_id,['ScenarioShocks = ' num2str(DSGEModel.ScenarioShocks) '\n']);
fprintf(f_id,['ScenarioVariables = ' num2str(DSGEModel.ScenarioVariables) '\n']);
fprintf(f_id,['ScenarioParameters = ' num2str(DSGEModel.ScenarioParameters) '\n']);
if isempty(DSGEModel.PredictionEvent)==0;
   fprintf(f_id,'PredictionEvent = %s\n',num2str(vec(DSGEModel.PredictionEvent)'));
else;
   fprintf(f_id,'PredictionEvent = \n');
end;
fprintf(f_id,['ParameterScenario = ' num2str(DSGEModel.ParameterScenario) '\n']);
%
% sample data
%
fprintf(f_id,'\n[Sample Data]\n');
fprintf(f_id,'DataFrequency = %s\n',DSGEModel.DataFrequency);
fprintf(f_id,'BeginYear = %s\n',DSGEModel.BeginYear);
fprintf(f_id,'BeginPeriod = %s\n',DSGEModel.BeginPeriod);
fprintf(f_id,'EndYear = %s\n',DSGEModel.EndYear);
fprintf(f_id,'EndPeriod = %s\n',DSGEModel.EndPeriod);
fprintf(f_id,'SubBeginYear = %s\n',DSGEModel.SubBeginYear);
fprintf(f_id,'SubBeginPeriod = %s\n',DSGEModel.SubBeginPeriod);
fprintf(f_id,'SubEndYear = %s\n',DSGEModel.SubEndYear);
fprintf(f_id,'SubEndPeriod = %s\n',DSGEModel.SubEndPeriod);
%
% optimization data
%
fprintf(f_id,'\n[Optimization]\n');
fprintf(f_id,'MaximizeAlgorithmValue = %0.0f\n',DSGEModel.MaximizeAlgorithmValue);
fprintf(f_id,'OptToleranceValue = %0.0f\n',DSGEModel.OptToleranceValue);
fprintf(f_id,'OptMaxIterationsValue = %0.0f\n',DSGEModel.OptMaxIterationsValue);
fprintf(f_id,'InitializeHessian = %0.0f\n',DSGEModel.InitializeHessian);
fprintf(f_id,'FiniteDifferenceHessian = %0.0f\n',DSGEModel.FiniteDifferenceHessian);
fprintf(f_id,'StepLengthHessian = %0.0f\n',DSGEModel.StepLengthHessian);
fprintf(f_id,'CheckOptimum = %0.0f\n',DSGEModel.CheckOptimum);
fprintf(f_id,'CheckTransformedOptimum = %0.0f\n',DSGEModel.CheckTransformedOptimum);
fprintf(f_id,'ModifiedHessian = %0.0f\n',DSGEModel.ModifiedHessian);
fprintf(f_id,'NumberOfGridPoints = %0.0f\n',DSGEModel.NumberOfGridPoints);
fprintf(f_id,'GridWidth = %0.0f\n',DSGEModel.GridWidth);
%
% Posterior sampling data 
%
fprintf(f_id,'\n[Posterior Sampling Data]\n');
fprintf(f_id,'PosteriorDrawsValue = %0.0f\n',DSGEModel.PosteriorDrawsValue);
fprintf(f_id,'SampleBatchValue = %0.0f\n',DSGEModel.SampleBatchValue);
fprintf(f_id,'ParallelChainsValue = %0.0f\n',DSGEModel.ParallelChainsValue);
fprintf(f_id,'BurnInValue = %0.0f\n',DSGEModel.BurnInValue);
fprintf(f_id,'InverseHessianEstimator = %0.0f\n',DSGEModel.InverseHessianEstimator);
fprintf(f_id,'ParameterCovMatrix = %s\n',strrep(DSGEModel.ParameterCovMatrix,pwd,''));
fprintf(f_id,'MaxCorrelationValue = %0.0f\n',DSGEModel.MaxCorrelationValue);
fprintf(f_id,'MHInitialScaleFactor = %0.0f\n',DSGEModel.MHInitialScaleFactor);
fprintf(f_id,'MHScaleFactor = %0.0f\n',DSGEModel.MHScaleFactor);
fprintf(f_id,'RandomWeightValue = %0.0f\n',DSGEModel.RandomWeightValue);
fprintf(f_id,'RandomizeDraws = %0.0f\n',DSGEModel.RandomizeDraws);
fprintf(f_id,'PostDrawsPercentValue = %0.0f\n',DSGEModel.PostDrawsPercentValue);
fprintf(f_id,'PostDrawsUsageValue = %0.0f\n',DSGEModel.PostDrawsUsageValue);
fprintf(f_id,'MarginalLikelihoodValue = %0.0f\n',DSGEModel.MarginalLikelihoodValue);
fprintf(f_id,'SequentialStartIteration = %0.0f\n',DSGEModel.SequentialStartIteration);
fprintf(f_id,'SequentialStepLength = %0.0f\n',DSGEModel.SequentialStepLength);
fprintf(f_id,'CoverageStart = %0.0f\n',DSGEModel.CoverageStart);
fprintf(f_id,'CoverageIncrement = %0.0f\n',DSGEModel.CoverageIncrement);
fprintf(f_id,'CoverageEnd = %0.0f\n',DSGEModel.CoverageEnd);
fprintf(f_id,'SequentialML = %0.0f\n',DSGEModel.SequentialML);
fprintf(f_id,'OverwriteDraws = %0.0f\n',DSGEModel.OverwriteDraws);
%
% Tools data
%
fprintf(f_id,'\n[Tools Options]\n');
fprintf(f_id,'IRHorizon = %0.0f\n',DSGEModel.IRHorizon);
fprintf(f_id,'MaxForecastHorizon = %0.0f\n',DSGEModel.MaxForecastHorizon);
fprintf(f_id,'ShockControlMethod = %0.0f\n',DSGEModel.ShockControlMethod);
%fprintf(f_id,'FreqParameterReest = %0.0f\n',DSGEModel.FreqParameterReest);
fprintf(f_id,'NumPredPathsValue = %0.0f\n',DSGEModel.NumPredPathsValue);
fprintf(f_id,'AdjustPredictionPaths = %0.0f\n',DSGEModel.AdjustPredictionPaths);
fprintf(f_id,'KsiUseCondData = %0.0f\n',DSGEModel.KsiUseCondData);
fprintf(f_id,'ScenarioValue = %0.0f\n',DSGEModel.ScenarioValue);
fprintf(f_id,'AnnScenarioValue = %0.0f\n',DSGEModel.AnnScenarioValue);
fprintf(f_id,'ParameterScenarioValue = %0.0f\n',DSGEModel.ParameterScenarioValue);
%
% graphics data
%
fprintf(f_id,'\n[Graphics Options]\n');
fprintf(f_id,'PosteriorDensityValue = %0.0f\n',DSGEModel.PosteriorDensityValue);
fprintf(f_id,'PriorKernel = %0.0f\n',DSGEModel.PriorKernel);
fprintf(f_id,'KernelDensityValue = %0.0f\n',DSGEModel.KernelDensityValue);
fprintf(f_id,'KernelDensityEstimator = %s\n',DSGEModel.KernelDensityEstimator);
fprintf(f_id,'ConfidenceBandBaseColor = %s\n',num2str(vec(DSGEModel.ConfidenceBandBaseColor)'));
%
% Bayesian VAR data
%
fprintf(f_id,'\n[Bayesian VAR]\n');
fprintf(f_id,['BVARY = ' num2str(DSGEModel.BVARY) '\n']);
fprintf(f_id,['BVARX = ' num2str(DSGEModel.BVARX) '\n']);
fprintf(f_id,'SteadyStatePriorFile = %s\n',strrep(DSGEModel.SteadyStatePriorFile,pwd,''));
fprintf(f_id,'BVARLags = %0.0f\n',DSGEModel.BVARLags);
fprintf(f_id,'PriorType = %0.0f\n',DSGEModel.PriorType);
fprintf(f_id,'PriorLevelMeanValue = %0.0f\n',DSGEModel.PriorLevelMeanValue);
fprintf(f_id,'PriorDiffMeanValue = %0.0f\n',DSGEModel.PriorDiffMeanValue);
fprintf(f_id,'OverallTightnessValue = %0.0f\n',DSGEModel.OverallTightnessValue);
fprintf(f_id,'CrossEqTightnessValue = %0.0f\n',DSGEModel.CrossEqTightnessValue);
fprintf(f_id,'HarmonicLagDecayValue = %0.0f\n',DSGEModel.HarmonicLagDecayValue);
fprintf(f_id,'OmegaPriorType = %0.0f\n',DSGEModel.OmegaPriorType);
fprintf(f_id,'WishartType = %0.0f\n',DSGEModel.WishartType);
fprintf(f_id,'VarianceTightnessValue = %0.0f\n',DSGEModel.VarianceTightnessValue);
fprintf(f_id,'WishartDFValue = %0.0f\n',DSGEModel.WishartDFValue);
%
% run-time data
%
fprintf(f_id,'\n[Other Data]\n');
fprintf(f_id,'RunInitializeFirst = %0.0f\n',DSGEModel.RunInitializeFirst);
fprintf(f_id,'UseDoublingAlgorithm = %0.0f\n',DSGEModel.UseDoublingAlgorithm);
fprintf(f_id,'DAMaximumIterationsValue = %0.0f\n',DSGEModel.DAMaximumIterationsValue);
fprintf(f_id,'DAToleranceValue = %0.0f\n',DSGEModel.DAToleranceValue);
fprintf(f_id,'StateCovConst = %0.0f\n',DSGEModel.StateCovConst);
fprintf(f_id,'UseOwnInitialState = %0.0f\n',DSGEModel.UseOwnInitialState);
fprintf(f_id,'AllowUnitRoot = %0.0f\n',DSGEModel.AllowUnitRoot);
fprintf(f_id,'InitialStateValues = %s\n',num2str(DSGEModel.InitialStateValues'));
fprintf(f_id,'RiccatiMaxIterations = %0.0f\n',DSGEModel.RiccatiMaxIterations);
fprintf(f_id,'RiccatiToleranceValue = %0.0f\n',DSGEModel.RiccatiToleranceValue);
fprintf(f_id,'ShowProgress = %0.0f\n',DSGEModel.ShowProgress);
fprintf(f_id,'ShowProgressClock = %0.0f\n',DSGEModel.ShowProgressClock);
fprintf(f_id,'KalmanFirstObservation = %0.0f\n',DSGEModel.KalmanFirstObservation);
fprintf(f_id,'AIMTolerance = %0.0f\n',DSGEModel.AIMTolerance);
fprintf(f_id,'RandomNumberValue = %0.0f\n',DSGEModel.RandomNumberValue);
fclose(f_id);

%
% end of SaveDSGEModel.m
%
