function DSGEModel = ReadDSGEModel(ModelFile)
% ReadDSGEModel: Reads the saved DSGE model from a text file.
%
% USAGE:
%
%       DSGEModel = ReadDSGEModel(ModelFile,DSGEModel)
%
% REQUIRED INPUTS: ModelFile (string), with full path plus name of the DSGE model file to be read.
%
% REQUIRED OUTPUT: DSGEModel (structure), with fields containing the DSGE model information that
%                            is stored in the ModelFile
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: August 1, 2006.
%                        This version: February 12, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 02-08-2006: Added reading the AIMDataFile entry. Also, if first character of file and directory entries is equal
%               to \ or the entry is empty, then pwd is prepended to the string.
%
% * 09-08-2006: Added reading the "UseDoublingAlgorithm", "DAMaximumIterationsValue", "ShowProgress", and
%               "ShowProgressClock" entries.
%
% * 18-08-2006: Changed the StrTrim function to StringStrim.
%
% * 22-08-2006: Added the field "InitializeKalman".
%
% * 30-08-2006: Added the fields "OverwriteDraws", "MarginalLikelihoodValue", "SequentialStartIteration",
%               "SequentialStepLength", "SequentialML", "CoverageStart", "CoverageIncrement", and "CoverageEnd".
%
% * 31-10-2006: Added the fields "DAToleranceValue", "RiccatiMaxIterations", and "RiccatiToleranceValue".
%
% * 17-11-2006: Changed the default value of "UseDoublingAlgorithm" reflecting that the number of options have increased
%               when computing the initial state covariance matrix for the Kalman filter. The field "StateCovConst" was
%               also added.
%
% * 20-11-2006: Added the fields "UseOwnInitialState" and "InitialStateValues".
%
% * 28-11-2006: Added the fields "PosteriorDrawsValue", "SampleBatchValue", "ParallelChainsValue", "BurnInValue",
%               "MHInitialScaleFactor", "MHScaleFactor", and "RandomWeightValue".
%
% * 30-11-2006: Changed the default value of "MarginalLikelihoodValue" from 1 to 2. This means that by default only Geweke's
%               modified harmonic mean estimator is used.
%
% * 04-12-2006: Added the fields "ShockGroupNames" and "ShockGroups".
%
% * 07-12-2006: Added the fields "OptMaxIterationsValue", "OptToleranceValue", "MaximizeAlgorithmValue", "StepLengthHessian",
%               "NumberOfGridPoints", "FiniteDifferenceHessian", "CheckOptimum", "CheckTransformedOptimum", "GridWidth",
%               "IRHorizon", "PriorKernel", "KernelDensityValue", and "KernelDensityEstimator".
%
% * 15-12-2006: Added the field "PosteriorDensityValue".
%
% * 18-12-2006: Added the field "RandomNumberValue".
%
% * 21-12-2006: The ranges of "MHScaleFactor" and "MHInitialScaleFactor" were changed in YADAGUI.m. The default values of
%               the field entries therefore needed to be adjusted. The field "MaxForecastHorizon" was also added.
%
% * 22-12-2006: Added the field "FreqParameterReest".
%
% * 05-01-2007: Added the field "ShockGroupColors".
%
% * 11-01-2007: Added the fields "BVARLags", "HarmonicLagDecayValue", "CrossEqTightnessValue", "OverallTightnessValue",
%               "PriorType", "PriorLevelMeanValue", and "PriorDiffMeanValue".
%
% * 12-01-2007: Added the fields "OmegaPriorType", "WishartType", "VarianceTightnessValue", "WishartDFValue",
%               "SteadyStatePriorFile", "BVARY", and "BVARX".
%
% * 22-02-2007: Added the fields "NumPredPathsValue" and "PostDrawsUsageValue".
%
% * 26-02-2007: Added the field "RandomizeDraws".
%
% * 28-02-2007: Added the field "ConfidenceBandBaseColor".
%
% * 01-03-2007: Made sure that the "ConfidenceBandBaseColor" vector has dimension 1x3.
%
% * 08-03-2007: Added the fields "ConditionalShocks" and "ConditionalVariables".
%
% * 16-03-2007: Made sure that the fields "ConditionalShocks" and "ConditionalVariables" are row vectors.
%
% * 03-04-3007: Added the field "PostDrawsPercentValue".
%
% * 10-04-2007: Added the field "InverseHessianEstimator".
%
% * 16-04-2007: Added the field "MaxCorrelationValue".
%
% * 25-04-2007: Added the fields "ScenarioShocks", and "ScenarioVariables".
%
% * 26-04-2007: Added the fields "AnnScenarioValue" and "ScenarioValue".
%
% * 02-05-2007: Added the field "ScenarioParameters".
%
% * 03-05-2007: Added the field "PredictionEvent".
%
% * 04-05-2007: The field "PredictionEvent" should now hold a matrix with 3 columns.
%
% * 07-05-2007: Made sure that "PredictionEvent" has the correct dimension.
%
% * 23-08-2007: Added the field "AdjustPredictionPaths".
%
% * 28-09-2007: Added the field "ObsVariableScenarioFile".
%
% * 25-10-2007: The field "FreqParameterReest" is no longer supported.
%
% * 26-10-2007: Added the field "ParameterScenario".
%
% * 31-10-2007: Added the field "OVSComparisonFile".
%
% * 06-11-2007: Updated the documentation.
%
% * 17-12-2007: Added the field "ShockControlMethod".
%
% * 20-12-2007: Added the field "UnitRootStates".
%
% * 02-01-2008: Added the field "KalmanFirstObservation".
%
% * 22-01-2008: The field "OVSComparisonFile" is no longer supported.
%
% * 23-01-2008: The default value of "MaxForecastHorizon" is now 1.
%
% * 24-01-2008: Added the field "KsiUseCondData".
%
% * 31-01-2008: Added the fields "OrigScenarioVariables", "OrigScenarioShocks", "ShockColorStart", and "ShockColorEnd".
%
% * 18-02-2008: Added the field "ModifiedHessian".
%
% * 16-05-2008: Added the field "AIMTolerance".
%
% * 23-05-2008: Updated the documentation.
%
% * 26-05-2008: The "SequentialML" variable now has the default value 0. The "PostDrawsUsageValue" variabel now has
%               default value 1. The "MHInitialScaleFactor" variable now has default value 1.
%
% * 02-07-2008: Added the field "AllowUnitRoot".
%
% * 03-07-2008: Made sure that "SubBeginYear", "SubBeginPeriod", "SubEndYear", and "SubEndPeriod" by default are
%               equal to "BeginYear", "BeginPeriod", "EndYear", and "EndPeriod", respectively, when the information
%               is the dsge-file gives empty strings.
%
% * 28-07-2008: Added the field "ParameterScenarioValue".
%
% * 19-12-2008: Updated the documentation.
%
% * 29-01-2009: Added the field "InitializeHessian".
%
% * 12-02-2009: Added the field "ParameterCovMatrix".
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

ModTxt = ReadText(ModelFile);
ModTxt = char(ModTxt);
%
% locate all the necessary entries
%
pos = LocateStr(ModTxt,'DataConstructionFile = ');
if pos>0;
   DSGEModel.DataConstructionFile = StringTrim(strrep(ModTxt(pos,:),'DataConstructionFile = ',''));
   if (isempty(DSGEModel.DataConstructionFile)==1)|(strcmp(DSGEModel.DataConstructionFile(1,1),'\')==1);
      DSGEModel.DataConstructionFile = [pwd DSGEModel.DataConstructionFile];
   end;
else;
   DSGEModel.DataConstructionFile = pwd;
end;
%
pos = LocateStr(ModTxt,'MeasurementEquationFile = ');
if pos>0;
   DSGEModel.MeasurementEquationFile = StringTrim(strrep(ModTxt(pos,:),'MeasurementEquationFile = ',''));
   if (isempty(DSGEModel.MeasurementEquationFile)==1)|(strcmp(DSGEModel.MeasurementEquationFile(1,1),'\')==1);
      DSGEModel.MeasurementEquationFile = [pwd DSGEModel.MeasurementEquationFile];
   end;
else;
   DSGEModel.MeasurementEquationFile = pwd;
end;
%
pos = LocateStr(ModTxt,'PriorFile = ');
if pos>0;
   DSGEModel.PriorFile = StringTrim(strrep(ModTxt(pos,:),'PriorFile = ',''));
   if (isempty(DSGEModel.PriorFile)==1)|(strcmp(DSGEModel.PriorFile(1,1),'\')==1);
      DSGEModel.PriorFile = [pwd DSGEModel.PriorFile];
   end;
else;
   DSGEModel.PriorFile = pwd;
end;
%
pos = LocateStr(ModTxt,'PriorFileSheet = ');
if pos>0;
   DSGEModel.PriorFileSheet = StringTrim(strrep(ModTxt(pos,:),'PriorFileSheet = ',''));
else;
   DSGEModel.PriorFileSheet = 'None';
end;
%
pos = LocateStr(ModTxt,'UpdateParameterFile = ');
if pos>0;
   DSGEModel.UpdateParameterFile = StringTrim(strrep(ModTxt(pos,:),'UpdateParameterFile = ',''));
   if (isempty(DSGEModel.UpdateParameterFile)==1)|(strcmp(DSGEModel.UpdateParameterFile(1,1),'\')==1);
      DSGEModel.UpdateParameterFile = [pwd DSGEModel.UpdateParameterFile];
   end;
else;
   DSGEModel.UpdateParameterFile = pwd;
end;
%
pos = LocateStr(ModTxt,'InitializeParameterFile = ');
if pos>0;
   DSGEModel.InitializeParameterFile = StringTrim(strrep(ModTxt(pos,:),'InitializeParameterFile = ',''));
   if (isempty(DSGEModel.InitializeParameterFile)==1)|(strcmp(DSGEModel.InitializeParameterFile(1,1),'\')==1);
      DSGEModel.InitializeParameterFile = [pwd DSGEModel.InitializeParameterFile];
   end;
else;
   DSGEModel.InitializeParameterFile = pwd;
end;
%
pos = LocateStr(ModTxt,'AIMFile = ');
if pos>0;
   DSGEModel.AIMFile = StringTrim(strrep(ModTxt(pos,:),'AIMFile = ',''));
   if (isempty(DSGEModel.AIMFile)==1)|(strcmp(DSGEModel.AIMFile(1,1),'\')==1);
      DSGEModel.AIMFile = [pwd DSGEModel.AIMFile];
   end;
else;
   DSGEModel.AIMFile = pwd;
end;
pos = LocateStr(ModTxt,'AIMDataFile = ');
if pos>0;
   DSGEModel.AIMDataFile = StringTrim(strrep(ModTxt(pos,:),'AIMDataFile = ',''));
   if (isempty(DSGEModel.AIMDataFile)==1)|(strcmp(DSGEModel.AIMDataFile(1,1),'\')==1);
      DSGEModel.AIMDataFile = [pwd DSGEModel.AIMDataFile];
   end;
else;
   DSGEModel.AIMDataFile = pwd;
end;
%
pos = LocateStr(ModTxt,'NameOfModel = ');
if pos>0;
   DSGEModel.NameOfModel = StringTrim(strrep(ModTxt(pos,:),'NameOfModel = ',''));
else;
   DSGEModel.NameOfModel = '';
end;
%
pos = LocateStr(ModTxt,'OutputDirectory = ');
if pos>0;
   DSGEModel.OutputDirectory = StringTrim(strrep(ModTxt(pos,:),'OutputDirectory = ',''));
   if (isempty(DSGEModel.OutputDirectory)==1)|(strcmp(DSGEModel.OutputDirectory(1,1),'\')==1);
      DSGEModel.OutputDirectory = [pwd DSGEModel.OutputDirectory];
   end;
else;
   DSGEModel.OutputDirectory = pwd;
end;
%
pos = LocateStr(ModTxt,'StateShockNames = ');
if pos>0;
   StateShockNames = StringTrim(strrep(ModTxt(pos,:),'StateShockNames = ',''));
   DSGEModel.StateShockNames = Str2Matrix(StateShockNames,'$');
else;
   DSGEModel.StateShockNames = '';
end;
%
pos = LocateStr(ModTxt,'StateShockPositions = ');
if pos>0;
   DSGEModel.StateShockPositions = str2num(StringTrim(strrep(ModTxt(pos,:),'StateShockPositions = ','')));
else;
   DSGEModel.StateShockPositions = [];
end;
%
pos = LocateStr(ModTxt,'StateVariableNames = ');
if pos>0;
   StateVariableNames = StringTrim(strrep(ModTxt(pos,:),'StateVariableNames = ',''));
   DSGEModel.StateVariableNames = Str2Matrix(StateVariableNames,'$');
else;
   DSGEModel.StateVariableNames = '';
end;
%
pos = LocateStr(ModTxt,'StateVariablePositions = ');
if pos>0;
   DSGEModel.StateVariablePositions = str2num(StringTrim(strrep(ModTxt(pos,:),'StateVariablePositions = ','')));
else;
   DSGEModel.StateVariablePositions = [];
end;
%
pos = LocateStr(ModTxt,'DataFrequency = ');
if pos>0;
   DSGEModel.DataFrequency = StringTrim(strrep(ModTxt(pos,:),'DataFrequency = ',''));
else;
   DSGEModel.DataFrequency = '';
end;
%
pos = LocateStr(ModTxt,'BeginYear = ');
if pos>0;
   DSGEModel.BeginYear = StringTrim(strrep(ModTxt(pos,:),'BeginYear = ',''));
else;
   DSGEModel.BeginYear = '';
end;
%
pos = LocateStr(ModTxt,'BeginPeriod = ');
if pos>0;
   DSGEModel.BeginPeriod = StringTrim(strrep(ModTxt(pos,:),'BeginPeriod = ',''));
else;
   DSGEModel.BeginPeriod = '';
end;
%
pos = LocateStr(ModTxt,'EndYear = ');
if pos>0;
   DSGEModel.EndYear = StringTrim(strrep(ModTxt(pos,:),'EndYear = ',''));
else;
   DSGEModel.EndYear = '';
end;
%
pos = LocateStr(ModTxt,'EndPeriod = ');
if pos>0;
   DSGEModel.EndPeriod = StringTrim(strrep(ModTxt(pos,:),'EndPeriod = ',''));
else;
   DSGEModel.EndPeriod = '';
end;
%
pos = LocateStr(ModTxt,'SubBeginYear = ');
if pos>0;
   DSGEModel.SubBeginYear = StringTrim(strrep(ModTxt(pos,:),'SubBeginYear = ',''));
   if isempty(StringTrim(DSGEModel.SubBeginYear))==1;
      DSGEModel.SubBeginYear = DSGEModel.BeginYear;
   end;
else;
   DSGEModel.SubBeginYear = DSGEModel.BeginYear;
end;
%
pos = LocateStr(ModTxt,'SubBeginPeriod = ');
if pos>0;
   DSGEModel.SubBeginPeriod = StringTrim(strrep(ModTxt(pos,:),'SubBeginPeriod = ',''));
   if isempty(StringTrim(DSGEModel.SubBeginPeriod))==1;
      DSGEModel.SubBeginPeriod = DSGEModel.BeginPeriod;
   end;
else;
   DSGEModel.SubBeginPeriod = DSGEModel.BeginPeriod;
end;
%
pos = LocateStr(ModTxt,'SubEndYear = ');
if pos>0;
   DSGEModel.SubEndYear = StringTrim(strrep(ModTxt(pos,:),'SubEndYear = ',''));
   if isempty(StringTrim(DSGEModel.SubEndYear))==1;
      DSGEModel.SubEndYear = DSGEModel.EndYear;
   end;
else;
   DSGEModel.SubEndYear = DSGEModel.EndYear;
end;
%
pos = LocateStr(ModTxt,'SubEndPeriod = ');
if pos>0;
   DSGEModel.SubEndPeriod = StringTrim(strrep(ModTxt(pos,:),'SubEndPeriod = ',''));
   if isempty(StringTrim(DSGEModel.SubEndPeriod))==1;
      DSGEModel.SubEndPeriod = DSGEModel.EndPeriod;
   end;
else;
   DSGEModel.SubEndPeriod = DSGEModel.EndPeriod;
end;
%
pos = LocateStr(ModTxt,'RunInitializeFirst = ');
if pos>0;
   DSGEModel.RunInitializeFirst = str2num(StringTrim(strrep(ModTxt(pos,:),'RunInitializeFirst = ','')));
else;
   DSGEModel.RunInitializeFirst = 0;
end;
% UseDoublingAlgorithm
pos = LocateStr(ModTxt,'UseDoublingAlgorithm = ');
if pos>0;
   DSGEModel.UseDoublingAlgorithm = str2num(StringTrim(strrep(ModTxt(pos,:),'UseDoublingAlgorithm = ','')));
else;
   DSGEModel.UseDoublingAlgorithm = 2;
end;
% DAMaximumIterationsValue
pos = LocateStr(ModTxt,'DAMaximumIterationsValue = ');
if pos>0;
   DSGEModel.DAMaximumIterationsValue = str2num(StringTrim(strrep(ModTxt(pos,:),'DAMaximumIterationsValue = ','')));
else;
   DSGEModel.DAMaximumIterationsValue = 1;
end;
% ShowProgress
pos = LocateStr(ModTxt,'ShowProgress = ');
if pos>0;
   DSGEModel.ShowProgress = str2num(StringTrim(strrep(ModTxt(pos,:),'ShowProgress = ','')));
else;
   DSGEModel.ShowProgress = 1;
end;
% ShowProgressClock
pos = LocateStr(ModTxt,'ShowProgressClock = ');
if pos>0;
   DSGEModel.ShowProgressClock = str2num(StringTrim(strrep(ModTxt(pos,:),'ShowProgressClock = ','')));
else;
   DSGEModel.ShowProgressClock = 0;
end;
% InitializeKalman
pos = LocateStr(ModTxt,'InitializeKalman = ');
if pos>0;
   DSGEModel.InitializeKalman = str2num(StringTrim(strrep(ModTxt(pos,:),'InitializeKalman = ','')));
else;
   DSGEModel.InitializeKalman = 1;
end;
% KalmanFirstObservation
pos = LocateStr(ModTxt,'KalmanFirstObservation = ');
if pos>0;
   DSGEModel.KalmanFirstObservation = str2num(StringTrim(strrep(ModTxt(pos,:),'KalmanFirstObservation = ','')));
else;
   DSGEModel.KalmanFirstObservation = 1;
end;
% OverwriteDraws
pos = LocateStr(ModTxt,'OverwriteDraws = ');
if pos>0;
   DSGEModel.OverwriteDraws = str2num(StringTrim(strrep(ModTxt(pos,:),'OverwriteDraws = ','')));
else;
   DSGEModel.OverwriteDraws = 0;
end;
% MarginalLikelihoodValue
pos = LocateStr(ModTxt,'MarginalLikelihoodValue = ');
if pos>0;
   DSGEModel.MarginalLikelihoodValue = str2num(StringTrim(strrep(ModTxt(pos,:),'MarginalLikelihoodValue = ','')));
else;
   DSGEModel.MarginalLikelihoodValue = 2;
end;
% SequentialStartIteration
pos = LocateStr(ModTxt,'SequentialStartIteration = ');
if pos>0;
   DSGEModel.SequentialStartIteration = str2num(StringTrim(strrep(ModTxt(pos,:),'SequentialStartIteration = ','')));
else;
   DSGEModel.SequentialStartIteration = 4;
end;
% SequentialStepLength
pos = LocateStr(ModTxt,'SequentialStepLength = ');
if pos>0;
   DSGEModel.SequentialStepLength = str2num(StringTrim(strrep(ModTxt(pos,:),'SequentialStepLength = ','')));
else;
   DSGEModel.SequentialStepLength = 4;
end;
% SequentialML
pos = LocateStr(ModTxt,'SequentialML = ');
if pos>0;
   DSGEModel.SequentialML = str2num(StringTrim(strrep(ModTxt(pos,:),'SequentialML = ','')));
else;
   DSGEModel.SequentialML = 0;
end;
% CoverageStart
pos = LocateStr(ModTxt,'CoverageStart = ');
if pos>0;
   DSGEModel.CoverageStart = str2num(StringTrim(strrep(ModTxt(pos,:),'CoverageStart = ','')));
else;
   DSGEModel.CoverageStart = 2;
end;
% CoverageIncrement
pos = LocateStr(ModTxt,'CoverageIncrement = ');
if pos>0;
   DSGEModel.CoverageIncrement = str2num(StringTrim(strrep(ModTxt(pos,:),'CoverageIncrement = ','')));
else;
   DSGEModel.CoverageIncrement = 2;
end;
% CoverageEnd
pos = LocateStr(ModTxt,'CoverageEnd = ');
if pos>0;
   DSGEModel.CoverageEnd = str2num(StringTrim(strrep(ModTxt(pos,:),'CoverageEnd = ','')));
else;
   DSGEModel.CoverageEnd = 9;
end;
% DAToleranceValue
pos = LocateStr(ModTxt,'DAToleranceValue = ');
if pos>0;
   DSGEModel.DAToleranceValue = str2num(StringTrim(strrep(ModTxt(pos,:),'DAToleranceValue = ','')));
else;
   DSGEModel.DAToleranceValue = 7;
end;
% RiccatiMaxIterations
pos = LocateStr(ModTxt,'RiccatiMaxIterations = ');
if pos>0;
   DSGEModel.RiccatiMaxIterations = str2num(StringTrim(strrep(ModTxt(pos,:),'RiccatiMaxIterations = ','')));
else;
   DSGEModel.RiccatiMaxIterations = 10;
end;
% RiccatiToleranceValue
pos = LocateStr(ModTxt,'RiccatiToleranceValue = ');
if pos>0;
   DSGEModel.RiccatiToleranceValue = str2num(StringTrim(strrep(ModTxt(pos,:),'RiccatiToleranceValue = ','')));
else;
   DSGEModel.RiccatiToleranceValue = 7;
end;
% StateCovConst
pos = LocateStr(ModTxt,'StateCovConst = ');
if pos>0;
   DSGEModel.StateCovConst = str2num(StringTrim(strrep(ModTxt(pos,:),'StateCovConst = ','')));
else;
   DSGEModel.StateCovConst = 19;
end;
% UseOwnInitialState
pos = LocateStr(ModTxt,'UseOwnInitialState = ');
if pos>0;
   DSGEModel.UseOwnInitialState = str2num(StringTrim(strrep(ModTxt(pos,:),'UseOwnInitialState = ','')));
else;
   DSGEModel.UseOwnInitialState = 0;
end;
% InitialStateValues
pos = LocateStr(ModTxt,'InitialStateValues = ');
if pos>0;
   DSGEModel.InitialStateValues = str2num(StringTrim(strrep(ModTxt(pos,:),'InitialStateValues = ','')))';
else;
   DSGEModel.InitialStateValues = [];
end;
% PosteriorDrawsValue
pos = LocateStr(ModTxt,'PosteriorDrawsValue = ');
if pos>0;
   DSGEModel.PosteriorDrawsValue = str2num(StringTrim(strrep(ModTxt(pos,:),'PosteriorDrawsValue = ','')))';
else;
   DSGEModel.PosteriorDrawsValue = 1;
end;
% SampleBatchValue
pos = LocateStr(ModTxt,'SampleBatchValue = ');
if pos>0;
   DSGEModel.SampleBatchValue = str2num(StringTrim(strrep(ModTxt(pos,:),'SampleBatchValue = ','')))';
else;
   DSGEModel.SampleBatchValue = 1;
end;
% ParallelChainsValue
pos = LocateStr(ModTxt,'ParallelChainsValue = ');
if pos>0;
   DSGEModel.ParallelChainsValue = str2num(StringTrim(strrep(ModTxt(pos,:),'ParallelChainsValue = ','')))';
else;
   DSGEModel.ParallelChainsValue = 1;
end;
% BurnInValue
pos = LocateStr(ModTxt,'BurnInValue = ');
if pos>0;
   DSGEModel.BurnInValue = str2num(StringTrim(strrep(ModTxt(pos,:),'BurnInValue = ','')))';
else;
   DSGEModel.BurnInValue = 1;
end;
% MHInitialScaleFactor
pos = LocateStr(ModTxt,'MHInitialScaleFactor = ');
if pos>0;
   DSGEModel.MHInitialScaleFactor = str2num(StringTrim(strrep(ModTxt(pos,:),'MHInitialScaleFactor = ','')))';
else;
   DSGEModel.MHInitialScaleFactor = 1;
end;
% MHScaleFactor
pos = LocateStr(ModTxt,'MHScaleFactor = ');
if pos>0;
   DSGEModel.MHScaleFactor = str2num(StringTrim(strrep(ModTxt(pos,:),'MHScaleFactor = ','')))';
else;
   DSGEModel.MHScaleFactor = 26;
end;
% RandomWeightValue
pos = LocateStr(ModTxt,'RandomWeightValue = ');
if pos>0;
   DSGEModel.RandomWeightValue = str2num(StringTrim(strrep(ModTxt(pos,:),'RandomWeightValue = ','')))';
else;
   DSGEModel.RandomWeightValue = 6;
end;
% ShockGroupNames
pos = LocateStr(ModTxt,'ShockGroupNames = ');
if pos>0;
   ShockGroupNames = StringTrim(strrep(ModTxt(pos,:),'ShockGroupNames = ',''));
   DSGEModel.ShockGroupNames = Str2Matrix(ShockGroupNames,'$');
else;
   DSGEModel.ShockGroupNames = '';
   if isempty(DSGEModel.StateShockNames)==0;
      DSGEModel.ShockGroupNames = DSGEModel.StateShockNames;
   end;
end;
% ShockGroups
pos = LocateStr(ModTxt,'ShockGroups = ');
if pos>0;
   DSGEModel.ShockGroups = str2num(StringTrim(strrep(ModTxt(pos,:),'ShockGroups = ','')));
else;
   DSGEModel.ShockGroups = [];
   if isempty(DSGEModel.StateShockPositions)==0;
      DSGEModel.ShockGroups = (1:length(DSGEModel.StateShockPositions));
   end;
end;
% OptMaxIterationsValue
pos = LocateStr(ModTxt,'OptMaxIterationsValue = ');
if pos>0;
   DSGEModel.OptMaxIterationsValue = str2num(StringTrim(strrep(ModTxt(pos,:),'OptMaxIterationsValue = ','')))';
else;
   DSGEModel.OptMaxIterationsValue = 1;
end;
% OptToleranceValue
pos = LocateStr(ModTxt,'OptToleranceValue = ');
if pos>0;
   DSGEModel.OptToleranceValue = str2num(StringTrim(strrep(ModTxt(pos,:),'OptToleranceValue = ','')))';
else;
   DSGEModel.OptToleranceValue = 4;
end;
% MaximizeAlgorithmValue
pos = LocateStr(ModTxt,'MaximizeAlgorithmValue = ');
if pos>0;
   DSGEModel.MaximizeAlgorithmValue = str2num(StringTrim(strrep(ModTxt(pos,:),'MaximizeAlgorithmValue = ','')))';
else;
   DSGEModel.MaximizeAlgorithmValue = 1;
end;
% StepLengthHessian
pos = LocateStr(ModTxt,'StepLengthHessian = ');
if pos>0;
   DSGEModel.StepLengthHessian = str2num(StringTrim(strrep(ModTxt(pos,:),'StepLengthHessian = ','')))';
else;
   DSGEModel.StepLengthHessian = 12;
end;
% NumberOfGridPoints
pos = LocateStr(ModTxt,'NumberOfGridPoints = ');
if pos>0;
   DSGEModel.NumberOfGridPoints = str2num(StringTrim(strrep(ModTxt(pos,:),'NumberOfGridPoints = ','')))';
else;
   DSGEModel.NumberOfGridPoints = 3;
end;
% FiniteDifferenceHessian
pos = LocateStr(ModTxt,'FiniteDifferenceHessian = ');
if pos>0;
   DSGEModel.FiniteDifferenceHessian = str2num(StringTrim(strrep(ModTxt(pos,:),'FiniteDifferenceHessian = ','')))';
else;
   DSGEModel.FiniteDifferenceHessian = 0;
end;
% CheckOptimum
pos = LocateStr(ModTxt,'CheckOptimum = ');
if pos>0;
   DSGEModel.CheckOptimum = str2num(StringTrim(strrep(ModTxt(pos,:),'CheckOptimum = ','')))';
else;
   DSGEModel.CheckOptimum = 1;
end;
% CheckTransformedOptimum
pos = LocateStr(ModTxt,'CheckTransformedOptimum = ');
if pos>0;
   DSGEModel.CheckTransformedOptimum = str2num(StringTrim(strrep(ModTxt(pos,:),'CheckTransformedOptimum = ','')))';
else;
   DSGEModel.CheckTransformedOptimum = 1;
end;
% GridWidth
pos = LocateStr(ModTxt,'GridWidth = ');
if pos>0;
   DSGEModel.GridWidth = str2num(StringTrim(strrep(ModTxt(pos,:),'GridWidth = ','')))';
else;
   DSGEModel.GridWidth = 4;
end;
% IRHorizon
pos = LocateStr(ModTxt,'IRHorizon = ');
if pos>0;
   DSGEModel.IRHorizon = str2num(StringTrim(strrep(ModTxt(pos,:),'IRHorizon = ','')))';
else;
   DSGEModel.IRHorizon = 10;
end;
% PriorKernel
pos = LocateStr(ModTxt,'PriorKernel = ');
if pos>0;
   DSGEModel.PriorKernel = str2num(StringTrim(strrep(ModTxt(pos,:),'PriorKernel = ','')))';
else;
   DSGEModel.PriorKernel = 0;
end;
% KernelDensityValue
pos = LocateStr(ModTxt,'KernelDensityValue = ');
if pos>0;
   DSGEModel.KernelDensityValue = str2num(StringTrim(strrep(ModTxt(pos,:),'KernelDensityValue = ','')))';
else;
   DSGEModel.KernelDensityValue = 2;
end;
% KernelDensityEstimator
pos = LocateStr(ModTxt,'KernelDensityEstimator = ');
if pos>0;
   DSGEModel.KernelDensityEstimator = StringTrim(strrep(ModTxt(pos,:),'KernelDensityEstimator = ',''));
else;
   DSGEModel.KernelDensityEstimator = 'kepan';
end;
% PosteriorDensityValue
pos = LocateStr(ModTxt,'PosteriorDensityValue = ');
if pos>0;
   DSGEModel.PosteriorDensityValue = str2num(StringTrim(strrep(ModTxt(pos,:),'PosteriorDensityValue = ','')))';
else;
   DSGEModel.PosteriorDensityValue = 1;
end;
% RandomNumberValue
pos = LocateStr(ModTxt,'RandomNumberValue = ');
if pos>0;
   DSGEModel.RandomNumberValue = str2num(StringTrim(strrep(ModTxt(pos,:),'RandomNumberValue = ','')))';
else;
   DSGEModel.RandomNumberValue = 0;
end;
% MaxForecastHorizon
pos = LocateStr(ModTxt,'MaxForecastHorizon = ');
if pos>0;
   DSGEModel.MaxForecastHorizon = str2num(StringTrim(strrep(ModTxt(pos,:),'MaxForecastHorizon = ','')))';
else;
   DSGEModel.MaxForecastHorizon = 1;
end;
% FreqParameterReest
%pos = LocateStr(ModTxt,'FreqParameterReest = ');
%if pos>0;
%   DSGEModel.FreqParameterReest = str2num(StringTrim(strrep(ModTxt(pos,:),'FreqParameterReest = ','')))';
%else;
%   DSGEModel.FreqParameterReest = 1;
%end;
% ShockGroupColors
pos = LocateStr(ModTxt,'ShockGroupColors = ');
if pos>0;
   DSGEModel.ShockGroupColors = str2num(StringTrim(strrep(ModTxt(pos,:),'ShockGroupColors = ','')));
   %
   % transform this into a matrix if it is not empty
   %
   if isempty(DSGEModel.ShockGroupColors)==0;
      DSGEModel.ShockGroupColors = reshape(DSGEModel.ShockGroupColors,length(DSGEModel.ShockGroupColors)/3,3);
   end;
else;
   DSGEModel.ShockGroupColors = [];
end;
% ShockColorStart
pos = LocateStr(ModTxt,'ShockColorStart = ');
if pos>0;
   DSGEModel.ShockColorStart = str2num(StringTrim(strrep(ModTxt(pos,:),'ShockColorStart = ','')));
   %
   % transform this into a matrix if it is not empty
   %
   if isempty(DSGEModel.ShockColorStart)==0;
      DSGEModel.ShockColorStart = reshape(DSGEModel.ShockColorStart,length(DSGEModel.ShockColorStart)/3,3);
   end;
else;
   DSGEModel.ShockColorStart = [];
end;
if isempty(DSGEModel.ShockColorStart)==1;
   DSGEModel.ShockColorStart = DSGEModel.ShockGroupColors;
end;
% ShockColorEnd
pos = LocateStr(ModTxt,'ShockColorEnd = ');
if pos>0;
   DSGEModel.ShockColorEnd = str2num(StringTrim(strrep(ModTxt(pos,:),'ShockColorEnd = ','')));
   %
   % transform this into a matrix if it is not empty
   %
   if isempty(DSGEModel.ShockColorEnd)==0;
      DSGEModel.ShockColorEnd = reshape(DSGEModel.ShockColorEnd,length(DSGEModel.ShockColorEnd)/3,3);
   end;
else;
   DSGEModel.ShockColorEnd = [];
end;
if isempty(DSGEModel.ShockColorEnd)==1;
   DSGEModel.ShockColorEnd = DSGEModel.ShockGroupColors;
end;
% BVARLags
pos = LocateStr(ModTxt,'BVARLags = ');
if pos>0;
   DSGEModel.BVARLags = str2num(StringTrim(strrep(ModTxt(pos,:),'BVARLags = ','')))';
else;
   DSGEModel.BVARLags = 4;
end;
% HarmonicLagDecayValue
pos = LocateStr(ModTxt,'HarmonicLagDecayValue = ');
if pos>0;
   DSGEModel.HarmonicLagDecayValue = str2num(StringTrim(strrep(ModTxt(pos,:),'HarmonicLagDecayValue = ','')))';
else;
   DSGEModel.HarmonicLagDecayValue = 2;
end;
% CrossEqTightnessValue
pos = LocateStr(ModTxt,'CrossEqTightnessValue = ');
if pos>0;
   DSGEModel.CrossEqTightnessValue = str2num(StringTrim(strrep(ModTxt(pos,:),'CrossEqTightnessValue = ','')))';
else;
   DSGEModel.CrossEqTightnessValue = 10;
end;
% OverallTightnessValue
pos = LocateStr(ModTxt,'OverallTightnessValue = ');
if pos>0;
   DSGEModel.OverallTightnessValue = str2num(StringTrim(strrep(ModTxt(pos,:),'OverallTightnessValue = ','')))';
else;
   DSGEModel.OverallTightnessValue = 4;
end;
% PriorType
pos = LocateStr(ModTxt,'PriorType = ');
if pos>0;
   DSGEModel.PriorType = str2num(StringTrim(strrep(ModTxt(pos,:),'PriorType = ','')))';
else;
   DSGEModel.PriorType = 1;
end;
% PriorLevelMeanValue
pos = LocateStr(ModTxt,'PriorLevelMeanValue = ');
if pos>0;
   DSGEModel.PriorLevelMeanValue = str2num(StringTrim(strrep(ModTxt(pos,:),'PriorLevelMeanValue = ','')))';
else;
   DSGEModel.PriorLevelMeanValue = 1;
end;
% PriorDiffMeanValue
pos = LocateStr(ModTxt,'PriorDiffMeanValue = ');
if pos>0;
   DSGEModel.PriorDiffMeanValue = str2num(StringTrim(strrep(ModTxt(pos,:),'PriorDiffMeanValue = ','')))';
else;
   DSGEModel.PriorDiffMeanValue = 1;
end;
% OmegaPriorType
pos = LocateStr(ModTxt,'OmegaPriorType = ');
if pos>0;
   DSGEModel.OmegaPriorType = str2num(StringTrim(strrep(ModTxt(pos,:),'OmegaPriorType = ','')))';
else;
   DSGEModel.OmegaPriorType = 1;
end;
% WishartType
pos = LocateStr(ModTxt,'WishartType = ');
if pos>0;
   DSGEModel.WishartType = str2num(StringTrim(strrep(ModTxt(pos,:),'WishartType = ','')))';
else;
   DSGEModel.WishartType = 1;
end;
% VarianceTightnessValue
pos = LocateStr(ModTxt,'VarianceTightnessValue = ');
if pos>0;
   DSGEModel.VarianceTightnessValue = str2num(StringTrim(strrep(ModTxt(pos,:),'VarianceTightnessValue = ','')))';
else;
   DSGEModel.VarianceTightnessValue = 20;
end;
% WishartDFValue
pos = LocateStr(ModTxt,'WishartDFValue = ');
if pos>0;
   DSGEModel.WishartDFValue = str2num(StringTrim(strrep(ModTxt(pos,:),'WishartDFValue = ','')))';
else;
   DSGEModel.WishartDFValue = 1;
end;
% SteadyStatePriorFile
pos = LocateStr(ModTxt,'SteadyStatePriorFile = ');
if pos>0;
   DSGEModel.SteadyStatePriorFile = StringTrim(strrep(ModTxt(pos,:),'SteadyStatePriorFile = ',''));
   if (isempty(DSGEModel.SteadyStatePriorFile)==1)|(strcmp(DSGEModel.SteadyStatePriorFile(1,1),'\')==1);
      DSGEModel.SteadyStatePriorFile = [pwd DSGEModel.SteadyStatePriorFile];
   end;
else;
   DSGEModel.SteadyStatePriorFile = pwd;
end;
% BVARY
pos = LocateStr(ModTxt,'BVARY = ');
if pos>0;
   DSGEModel.BVARY = str2num(StringTrim(strrep(ModTxt(pos,:),'BVARY = ','')))';
else;
   DSGEModel.BVARY = [];
end;
% BVARX
pos = LocateStr(ModTxt,'BVARX = ');
if pos>0;
   DSGEModel.BVARX = str2num(StringTrim(strrep(ModTxt(pos,:),'BVARX = ','')))';
else;
   DSGEModel.BVARX = [];
end;
% NumPredPathsValue
pos = LocateStr(ModTxt,'NumPredPathsValue = ');
if pos>0;
   DSGEModel.NumPredPathsValue = str2num(StringTrim(strrep(ModTxt(pos,:),'NumPredPathsValue = ','')))';
else;
   DSGEModel.NumPredPathsValue = 1;
end;
% PostDrawsUsageValue
pos = LocateStr(ModTxt,'PostDrawsUsageValue = ');
if pos>0;
   DSGEModel.PostDrawsUsageValue = str2num(StringTrim(strrep(ModTxt(pos,:),'PostDrawsUsageValue = ','')))';
else;
   DSGEModel.PostDrawsUsageValue = 1;
end;
% RandomizeDraws
pos = LocateStr(ModTxt,'RandomizeDraws = ');
if pos>0;
   DSGEModel.RandomizeDraws = str2num(StringTrim(strrep(ModTxt(pos,:),'RandomizeDraws = ','')))';
else;
   DSGEModel.RandomizeDraws = 0;
end;
% ConfidenceBandBaseColor
pos = LocateStr(ModTxt,'ConfidenceBandBaseColor = ');
if pos>0;
   DSGEModel.ConfidenceBandBaseColor = str2num(StringTrim(strrep(ModTxt(pos,:),'ConfidenceBandBaseColor = ','')))';
   if isempty(DSGEModel.ConfidenceBandBaseColor)==0;
      DSGEModel.ConfidenceBandBaseColor = reshape(DSGEModel.ConfidenceBandBaseColor,1,3);
   end;
else;
   DSGEModel.ConfidenceBandBaseColor = [1 1 1];
end;
% ConditionalShocks
pos = LocateStr(ModTxt,'ConditionalShocks = ');
if pos>0;
   DSGEModel.ConditionalShocks = str2num(StringTrim(strrep(ModTxt(pos,:),'ConditionalShocks = ','')));
else;
   DSGEModel.ConditionalShocks = [];
end;
% ConditionalVariables
pos = LocateStr(ModTxt,'ConditionalVariables = ');
if pos>0;
   DSGEModel.ConditionalVariables = str2num(StringTrim(strrep(ModTxt(pos,:),'ConditionalVariables = ','')));
else;
   DSGEModel.ConditionalVariables = [];
end;
% PostDrawsPercentValue
pos = LocateStr(ModTxt,'PostDrawsPercentValue = ');
if pos>0;
   DSGEModel.PostDrawsPercentValue = str2num(StringTrim(strrep(ModTxt(pos,:),'PostDrawsPercentValue = ','')));
else;
   DSGEModel.PostDrawsPercentValue = 109;
end;
% InverseHessianEstimator
pos = LocateStr(ModTxt,'InverseHessianEstimator = ');
if pos>0;
   DSGEModel.InverseHessianEstimator = str2num(StringTrim(strrep(ModTxt(pos,:),'InverseHessianEstimator = ','')));
else;
   DSGEModel.InverseHessianEstimator = 1;
end;
% MaxCorrelationValue
pos = LocateStr(ModTxt,'MaxCorrelationValue = ');
if pos>0;
   DSGEModel.MaxCorrelationValue = str2num(StringTrim(strrep(ModTxt(pos,:),'MaxCorrelationValue = ','')));
else;
   DSGEModel.MaxCorrelationValue = 1;
end;
% ScenarioShocks
pos = LocateStr(ModTxt,'ScenarioShocks = ');
if pos>0;
   DSGEModel.ScenarioShocks = str2num(StringTrim(strrep(ModTxt(pos,:),'ScenarioShocks = ','')));
else;
   DSGEModel.ScenarioShocks = [];
end;
% ScenarioVariables
pos = LocateStr(ModTxt,'ScenarioVariables = ');
if pos>0;
   DSGEModel.ScenarioVariables = str2num(StringTrim(strrep(ModTxt(pos,:),'ScenarioVariables = ','')));
else;
   DSGEModel.ScenarioVariables = [];
end;
%
% intializing "OrigScenarioVariables" and "OrigScenarioShocks"
%
DSGEModel.OrigScenarioVariables = DSGEModel.ScenarioVariables;
DSGEModel.OrigScenarioShocks = DSGEModel.ScenarioShocks;
% AnnScenarioValue
pos = LocateStr(ModTxt,'AnnScenarioValue = ');
if pos>0;
   DSGEModel.AnnScenarioValue = str2num(StringTrim(strrep(ModTxt(pos,:),'AnnScenarioValue = ','')));
else;
   DSGEModel.AnnScenarioValue = 1;
end;
% ScenarioValue
pos = LocateStr(ModTxt,'ScenarioValue = ');
if pos>0;
   DSGEModel.ScenarioValue = str2num(StringTrim(strrep(ModTxt(pos,:),'ScenarioValue = ','')));
else;
   DSGEModel.ScenarioValue = 1;
end;
% ScenarioParameters
pos = LocateStr(ModTxt,'ScenarioParameters = ');
if pos>0;
   DSGEModel.ScenarioParameters = str2num(StringTrim(strrep(ModTxt(pos,:),'ScenarioParameters = ','')));
else;
   DSGEModel.ScenarioParameters = [];
end;
% PredictionEvent
pos = LocateStr(ModTxt,'PredictionEvent = ');
if pos>0;
   DSGEModel.PredictionEvent = str2num(StringTrim(strrep(ModTxt(pos,:),'PredictionEvent = ','')));
   %
   % transform this into a matrix if it is not empty
   %
   if (isempty(DSGEModel.PredictionEvent)==0)&(IsInteger(length(DSGEModel.PredictionEvent)/3)==1);
      DSGEModel.PredictionEvent = reshape(DSGEModel.PredictionEvent,floor(length(DSGEModel.PredictionEvent)/3),3);
   else;
      DSGEModel.PredictionEvent = [];
   end;
else;
   DSGEModel.PredictionEvent = [];
end;
% AdjustPredictionPaths
pos = LocateStr(ModTxt,'AdjustPredictionPaths = ');
if pos>0;
   DSGEModel.AdjustPredictionPaths = str2num(StringTrim(strrep(ModTxt(pos,:),'AdjustPredictionPaths = ','')));
else;
   DSGEModel.AdjustPredictionPaths = 1;
end;
% KsiUseCondData
pos = LocateStr(ModTxt,'KsiUseCondData = ');
if pos>0;
   DSGEModel.KsiUseCondData = str2num(StringTrim(strrep(ModTxt(pos,:),'KsiUseCondData = ','')));
else;
   DSGEModel.KsiUseCondData = 0;
end;
% ObsVariableScenarioFile
pos = LocateStr(ModTxt,'ObsVariableScenarioFile = ');
if pos>0;
   DSGEModel.ObsVariableScenarioFile = StringTrim(strrep(ModTxt(pos,:),'ObsVariableScenarioFile = ',''));
else;
   DSGEModel.ObsVariableScenarioFile = '';
end;
% ParameterScenario
pos = LocateStr(ModTxt,'ParameterScenario = ');
if pos>0;
   DSGEModel.ParameterScenario = str2num(StringTrim(strrep(ModTxt(pos,:),'ParameterScenario = ','')));
else;
   DSGEModel.ParameterScenario = [];
end;
% ShockControlMethod
pos = LocateStr(ModTxt,'ShockControlMethod = ');
if pos>0;
   DSGEModel.ShockControlMethod = str2num(StringTrim(strrep(ModTxt(pos,:),'ShockControlMethod = ','')));
else;
   DSGEModel.ShockControlMethod = 1;
end;
% UnitRootStates
pos = LocateStr(ModTxt,'UnitRootStates = ');
if pos>0;
   DSGEModel.UnitRootStates = str2num(StringTrim(strrep(ModTxt(pos,:),'UnitRootStates = ','')));
else;
   DSGEModel.UnitRootStates = [];
end;
% ModifiedHessian
pos = LocateStr(ModTxt,'ModifiedHessian = ');
if pos>0;
   DSGEModel.ModifiedHessian = str2num(StringTrim(strrep(ModTxt(pos,:),'ModifiedHessian = ','')));
else;
   DSGEModel.ModifiedHessian = 0;
end;
% AIMTolerance
pos = LocateStr(ModTxt,'AIMTolerance = ');
if pos>0;
   DSGEModel.AIMTolerance = str2num(StringTrim(strrep(ModTxt(pos,:),'AIMTolerance = ','')));
else;
   DSGEModel.AIMTolerance = 6;
end;
% AllowUnitRoot
pos = LocateStr(ModTxt,'AllowUnitRoot = ');
if pos>0;
   DSGEModel.AllowUnitRoot = str2num(StringTrim(strrep(ModTxt(pos,:),'AllowUnitRoot = ','')));
else;
   DSGEModel.AllowUnitRoot = 0;
end;
% ParameterScenarioValue
pos = LocateStr(ModTxt,'ParameterScenarioValue = ');
if pos>0;
   DSGEModel.ParameterScenarioValue = str2num(StringTrim(strrep(ModTxt(pos,:),'ParameterScenarioValue = ','')));
else;
   DSGEModel.ParameterScenarioValue = 1;
end;
% InitializeHessian
pos = LocateStr(ModTxt,'InitializeHessian = ');
if pos>0;
   DSGEModel.InitializeHessian = str2num(StringTrim(strrep(ModTxt(pos,:),'InitializeHessian = ','')));
else;
   DSGEModel.InitializeHessian = 5;
end;
% ParameterCovMatrix
pos = LocateStr(ModTxt,'ParameterCovMatrix = ');
if pos>0;
   DSGEModel.ParameterCovMatrix = StringTrim(strrep(ModTxt(pos,:),'ParameterCovMatrix = ',''));
   if (strcmp(DSGEModel.ParameterCovMatrix(1,1),'\')==1);
      DSGEModel.ParameterCovMatrix = [pwd DSGEModel.ParameterCovMatrix];
   end;
   if FileExist(DSGEModel.ParameterCovMatrix)==0;
      DSGEModel.ParameterCovMatrix = 'none';
   end;
else;
   DSGEModel.ParameterCovMatrix = 'none';
end;

%
% end of ReadDSGEModel.m
%
