function [A,H,R] = MeasurementEqFileRegimeShift(ModelParameters,StateVarNames,VariableNames,XVariableNames,n,r,k)
% MeasurementEqFileRegimeShift: Shows how to set up the measurement equations for the An-Schorfheide model.
%
% USAGE:
%
%       [A,H,R] = MeasurementEqFileRegimeShift(ModelParameters,StateVarNames,VariableNames,XVariableNames,n,r,k)
%
%
% REQUIRED INPUTS:  ModelParameters (structure) with fields given by all the model parameters
%                                   and the fields taking scalar values. All model parameters
%                                   includes the parameters to be estimated, the parameters that
%                                   are calibrated, and the parameters that are functions of
%                                   the above.
%
%                   StateVarNames (string matrix) with rows being the names of the variables
%                                   as they have been assigned in the AiM model file. This will
%                                   typically include all state variables, as well as the shocks
%                                   to the state variables (and possibly a constant).
%
%                   VariableNames (string matrix) where each row contains a name of an observed
%                                   variables.
%
%                   XVariableNames (string matrix) where each row contains a name of an exogenous
%                                   variable.
%
%                   n (integer) with the number of observed variables.
%
%                   r (integer) with the number of state variables.
%
%                   k (integer) with the number of exogenous variables.
%
% REQUIRED OUTPUTS: A (kxn) matrix with parameters on the exogenous variables. Notice that k=
%                                   number of exogenous variables, n=number of observed variables
%                                   in the measurement equation.
%
%                   H (rxn) matrix with parameters on the state variables. Here, r=number of state
%                                   variables.
%
%                   R (nxn) matrix with covariances for the measurement errors. This matrix has
%                                   to be positive semi-definite!
%
%
% NOTICE: The measurement equations are expressed as:
%
%             y(t) = A'*x(t) + H'*xi(t) + w(t)
%
%         where y (nx1) is the vector of observed variable, x (kx1) is the vector of exogenous
%         variables, xi (rx1) the vector of state variables, and w (nx1) the vector of measurement
%         errors, with covariance matrix R.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: July 31, 2006.
%                        This version: January 23, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 03-08-2006: Fixed som bugs in the H matrix.
%
% * 17-08-2006: Added the two new inputs "VariableNames" and "XVariableNames". These are now used with the loc function to
%               locate the correct positions of the observed and the exogenous variables in the A and H matrices.
%
%               NOTE: For small models it probably doesn't matter much for computation time of the measurement equation
%                     file if the loc command is used or the exact position of a variable is speicified directly. For
%                     large models with many calls to the measurement equation file, using loc can presumably increase
%                     computation time considerably.
%
% * 14-09-2006: Added the inputs n, r, and k to allow for easy access to the number of observed variables, the number
%               of state variables, and the number of exogenous variables. Among these inputs, only the number of state
%               variables can't be deduced from the other inputs.
%
% * 07-11-2007: Updated the documentation.
%
% * 09-01-2008: Updated the documentation.
%
% * 14-07-2008: Made H 3D.
%
% * 23-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% The constant (exogenous)
%
A = zeros(k,n);
A(loc(XVariableNames,'const'),loc(VariableNames,'YGR')) = ModelParameters.gammaQ;
A(loc(XVariableNames,'const'),loc(VariableNames,'INFL')) = ModelParameters.piA;
A(loc(XVariableNames,'const'),loc(VariableNames,'INT')) = ModelParameters.piA + ModelParameters.rA + (4*ModelParameters.gammaQ);
%
% The parameters on the state variables.
%
% NOTICE: There are no actual parameters from the ModelParameters structure being used here.
%         However, the StateVarNames string along with the loc functions is used to locate
%         the positions of the state variables. These positions correspond to the relevant
%         rows in the H matrix.
%
H1 = zeros(r,n);
H1(loc(StateVarNames,'yhat'),loc(VariableNames,'YGR')) = 100;
H1(loc(StateVarNames,'yhatlag'),loc(VariableNames,'YGR')) = -100;
H1(loc(StateVarNames,'zhat'),loc(VariableNames,'YGR')) = 100;
H1(loc(StateVarNames,'pihat'),loc(VariableNames,'INFL')) = 400;
H1(loc(StateVarNames,'rhat'),loc(VariableNames,'INT')) = 400;
H2 = zeros(r,n);
H2(loc(StateVarNames,'yhat2'),loc(VariableNames,'YGR')) = 100;
H2(loc(StateVarNames,'yhatlag2'),loc(VariableNames,'YGR')) = -100;
H2(loc(StateVarNames,'zhat2'),loc(VariableNames,'YGR')) = 100;
H2(loc(StateVarNames,'pihat2'),loc(VariableNames,'INFL')) = 400;
H2(loc(StateVarNames,'rhat2'),loc(VariableNames,'INT')) = 400;
H = zeros(r,n,120);
for t=1:50;
   H(:,:,t) = H1;
end;
for t=51:120;
   H(:,:,t) = H2;
end;
%
% The covariance matrix for the measurement errors. Here, we have no measurement errors!
%
R = zeros(n,n);

%
% end of MeasurementEqFile.m
%
