function StructureForData = DataConstFile
% DataConstFile: Shows how the data construction output should be provided for YADA. This example is based
%                on the An-Schorfheide model
%
% USAGE:
%
%       StructureForData = DataConstFile
%
% where StructureForData is a proper structure in matlab. The file is run in matlab via the eval
% function.
%
% INPUTS:          None!
%
% REQUIRED OUTPUT: StructureForData (structure). Any name for this structure will do since YADA renames it internally.
%                  REQUIRED FIELDS:  Y.data              (nxT) data-matrix of the n observable variables that should be evaluated by the
%                                                        Kalman filter
%                                    X.data              (kxT) data-matrix with exogenous data, may be an empty matrix, but must be stated
%                                    Y.names             (n dimensional cellstring) with variable names
%                                    X.names             (k dimensional cellstring) with names for exogenous variables (may be empty)
%                                    sample              (1x4 matrix) containing [start-year start-period end-yer end-period]. Must match
%                                                        the number of observations T.
%                                    frequency           (string) data frequency. Valid entries are: a(nnual), q(uarterly), m(onthly)
%                                                        if invalid frequency is reported, it is set to annual with sample being set
%                                                        to [1 1 T 1]
%                  OPTIONAL FIELDS:  Y.levels            (1xn) vector, where 0 indicates a first differenced variable and 1 a levels
%                                                        variable
%                                    Y.annual            (1xn) vector, where 1 indicates that the variable should be annualized based
%                                                        on the data frequency, while 0 means that it should not be annualized (i.e.,
%                                                        can already be regarded as annualized).
%                                    Y.comparisondata    (nxT) matrix with comparison data for the observed variables.
%                                    Y.annualscale       (1xn) vector with scaling coefficient to multiply the annualized variables
%                                                        with.
%                                    Y.BVAR              (1xn1) vector where n1<=n that gives the indices for the variables in Y that
%                                                        should be included in a Bayesian VAR.
%                                    X.BVAR              (1xk1) vector where k1<=k that gives the indices for the variables in X that
%                                                        should be included in a Bayesian VAR.
%                                    X.extradata         (kxT1) data matrix with out-of-sample exogenous data.
%                                    Y.transformation    structure with subfields having the same names as the observed variables. For each
%                                                        such subfield, the fields "fcn", "partial", "annualizefcn", "annualizepartial",
%                                                        "initial", "x", "invertfcn", "invertinitial", "invertx", "exportfcn", "exportinitial",
%                                                        and "exportc" are available. The "fcn" and "annualizefcn" fields contain inline
%                                                        function strings determining how to transform the data for the variable, denoted
%                                                        e.g. by y. The fields "partial" and "annualizepartial" are string vectors with the
%                                                        partial derivative of the function in the "fcn" and "annualizefcn" fields, respectively,
%                                                        with respect the first variable in the function times the first variable in the
%                                                        function. The field "initial" contains a vector with all the initial values that
%                                                        should be taken into account. These are prepended to y before transforming the
%                                                        variable. The field "x" is a matrix of dimension dxTx with any additional variables
%                                                        that are required by the transformation. This can, for example, include the data for
%                                                        a linear trend. The field "invertfcn" holds a string with the function needed to
%                                                        invert the "fcn" field function. The fields "invertinitial" and "invertx" serve the
%                                                        same purpose as "initial" and "x", but are instead used by "invertfcn". Similarly,
%                                                        the field "exportfcn" holds the function that is used when exporting data from the menu
%                                                        functions for exporting. The fields "exportinitial" and "exportx" are used by this
%                                                        function like in the case of "initial" and "x" for the "fcn" function. Furthermore,
%                                                        the fields "exporttitle" and "exportname" are supported. They are only used when
%                                                        exporting the data to a spreadsheet where they replace the variable name.
%                                    Z.data              (mxTz) data matrix with conditioning information.
%                                    Z.names             (m dimensional cell string) with names of the variables in the conditioning
%                                                        information set.
%                                    Z.K1                (nxm) matrix used to map Y(t) into Z(t).
%                                    Z.K2                (nxm) matrix used to map Y(t-i) into Z(t).
%                                    U.data              (mxTu) matrix with data for initial conditions for conditional forecasting.
%                                    percentiles         (1xs) vector with positive intergers greater than 0 and less than 100. This
%                                                        is used when plotting percentiles from distributions.
%                                    C.data              (cxTc) matrix with data for observed variable scenarios.
%                                    C.comparisondata    (cxTc) matrix with comparison data for observed variable scenarios.
%                                    C.names             (c dimensional cell string) with names of the variables for observed variables
%                                                        scenarios.
%                                    C.transformation    structure with the same construction as Y.transformation.
%                                    C.C1                (nxc) natrix used to map Y(t) into C(t).
%                                    C.C2                (nxc) matrix used to mape Y(t-i) into C(t).
%                                    CBar.data           (cxTc) matrix with data for initial conditions for the observed variable
%                                                        scenarios.
%                                    CBar.comparisondata (cxTc) matrix with data for initial conditions for the comparison data for
%                                                        the observed variable scenarios.
%                                    C.exportprepend     (cxTd) matrix with data to prepend after transforming the export data.
%                                    C.export            structure with two subfields "data" and "names". The field "data" can hold a matrix
%                                                        of dimension mx(Tc+Td), while "names" is a cellarray of dimension 2xm. These subfields
%                                                        will be written to the export-file is present.
%                                    C.exportvariables   structure with fields "names" and "titles", where the former gives the name of
%                                                        the variable be be created and exported, while the latter is used for the title
%                                                        in the export file. Both fields hold cell arrays of string, where the number
%                                                        of entries should be equal to the number of variables to create. For each string
%                                                        in the names cell srray there should be a field with the same name. For each
%                                                        such variable field, there should be 5 sub-fields: fcn, x, y, subsample, and
%                                                        priordata. The fcn field is a string that defines the function to apply. The
%                                                        x field is a matrix with all the exogneous variables for the function, while
%                                                        the y-field is a cell array of strings with the names of the scenario variables
%                                                        to apply in the function. Since the variables will be exported it should be
%                                                        kept in mind that the scenario data variables are taken after they have been
%                                                        transformed by the export transform functions. The field subsample gives the
%                                                        positions in the output vector from the function that should be used, while
%                                                        priordata is a vector with fixed values to prepend to the used subsample of the
%                                                        created variable.
%                                    C.createdata        structure with field "names", a cell array of strings that hold the names of
%                                                        the variables to create from the scenario data and exogenous variables. For
%                                                        each string in "names" the structure should have fields with the same name.
%                                                        Each such field should have at least 5 sub-fields. Namely, "createbeforefcn",
%                                                        "createafterfcn", "createx", "createcompx", "createy", and "before". The
%                                                        "createbeforefcn" and "createafterfcn" fields hold string vectors with the
%                                                        functions to use for creating the variable before and after any input have been
%                                                        transformed. The field "createx" is a matrix with exogenous variables that the
%                                                        functions needs. For the comparison data the field "createcompx" holds the valid
%                                                        data for the exogenous variables. The number of observations should be equal to
%                                                        the number of observations on the scenario data (Tc). The field "createy" is a cell
%                                                        array with names for the scenario data variables to use. The last required variable
%                                                        field is "before", a boolean variable that is 1 if creation functions should only
%                                                        be applied before any transformation functions have been applied to the scenario
%                                                        variables, and 0 if the creation functions should be applied also after the
%                                                        transformation functions have been run on the scenario variable. In this case
%                                                        "createafterfcn" is used. If the field "before" is 1, then each variable must have
%                                                        8 additional fields. Those additional fields are: "fcn", "partial", "annualizefcn",
%                                                        "annualizepartial", "initial", "x", "compinitial", and "compx". The fields hold
%                                                        the data for applying a transformation function to the created variable.
%                                                        YADA assumes that this is only necessary if the inputs to the created variables
%                                                        have not been transformed yet. The data in these fields should be of the same
%                                                        type as the transformation functions in, e.g., C.transformation. When the comparison
%                                                        data is used instead of the scenario data, then "x" and "initial" are swapped with
%                                                        "compx" and "compinitial".
%                                    C.summaryorder      vector with c integers between 1 and c. This vector can be used to change the
%                                                        order of the secnario variables when displaying summary information.
%                                    C.startperiod       1x2 vector with the start-year start-period of the observed variable scenario.
%
%
% NOTICE:
%         * Any valid matlab functions on the path can be used by the function. How you transform any data you read in from
%           a spreadsheet, text file, of matlab mat-file is up to you.
%         * Any file format supported by matlab can be used for your data.
%
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: July 31, 2006.
%                        This version: February 26, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 08-12-2006: The subfield "levels" was added to the field "Y". This vector indicates that variable is in levels
%               if 1, and in first differences if 0.
%
% * 12-01-2007: Added the subfield "BVAR" to the field "Y" and the field "X". These vectors indicate which variables
%               should be included in a Bayesian VAR estimation.
%
% * 22-01-2007: Added the subfield "annual" to the field "Y". This vector indicates how to annualize variables in
%               first differences. The value "4" means that the data are in quarerly changes terms and, hence, that
%               we should add "4" consequetive observations to obtain the annual growth rate. The value "1" means that
%               the variable is already annualized. If the variable is considered a levels variable, then the entry
%               in this vector is ignored.
%
% * 23-01-2007: The vector "StructureForData.Y.annual" has been changed such that 1 means that the corresponding
%               variable should be annualized based on the data frequency. The value 0 means that teh variable in
%               question should not be annualized. For quarterlu data, the value 1 means that the current and previous
%               3 observation are summed to get the annualized observation.
%
% * 25-01-2007: The subfield "annualscale" has been added to the field "Y". This vector holds constants to multiply
%               the annualized variables with. These constants must be non-zero to be valid. Any zeros that YADA locates
%               will be replaced with unit values. For the An and Schorfheide model, inflation has been 'annualized' at
%               the quarterly frequency by mutliplying the variable by 4. Hence, for a proper annualized inflation we
%               should multiply this variable by 1/4 and then add 4 consecutive quarters. This is indicated by 0.25
%               in the vector "StructureForData.Y.annualscale" below.
%
% * 16-02-2007: Added the subfield "extradata" to the field "X". This data is necessary to compute predictions outside
%               the available sample. The subfield is treated as optional.
%
% * 20-02-2007: Added the optional field "Percentiles". This data is used for settings up confidence bands in plots of
%               various distributions. The default is [5 15 85 95].
%
% * 21-02-2007: Changed the data in the "percentiles" field. Also, this field should use lower case characters.
%
% * 08-03-2007: Added the output necessary to conduct conditional forecasts.
%
% * 09-03-2007: Changed the length of the conditioning sample from 16 to 8 periods.
%
% * 26-04-2007: Added the output necessary for computing the observed variables scenario.
%
% * 04-09-2007: Edited the data for the observed variable scenario.
%
% * 10-09-2007: Allow for NaN's in the "C.data" matrix.
%
% * 18-10-2007: Increased the size of the set of conditional variables.
%
% * 22-10-2007: Added the sub-field "transformation" to "Y" and "C".
%
% * 23-10-2007: Added the sub-field "annualizefcn" as an additional transformation function string.
%
% * 30-10-2007: Added the sub-fields "inversefcn", "inverseinitial", and "inversex" as additional
%               transformation function data.
%
% * 07-11-2007: Updated the documentation.
%
% * 09-11-2007: The fields "exportfcn", "exportinitial", and "exportx" are added as additional
%               transformation functions.
%
% * 14-11-2007: The fields "x", "invertx", and "exportx" now has two sub-fields "data" and "numinitialobs".
%
% * 20-11-2007: removed the subfields "data" and "numinitialobs" for "x", "invertx", and "exportx".
%
% * 21-11-2007: Changed the observed variable scenario data slightly.
%
% * 09-01-2008: The "comparisondata" field was added to the field "C" for observed variable scenarios.
%
% * 10-01-2008: Updated the documentation.
%
% * 11-01-2008: Added the "comparisondata" field to the "Y" field for the observed variables.
%
% * 25-01-2008: Added the field "exportname" as a sub-sub-field to the transformation field.
%
% * 28-03-2008: Added the field "exporttitle" as a sub-sub-field the the transformation field. The field
%               export with subfields "data" and "names" were added to the "C" field. Moreover, the
%               field "summaryorder" was added to the "C" field.
%
% * 02-04-2008: Added the field "startperiod" to the "C" field.
%
% * 07-04-2008: Added the field "exportprepend" to the "C" field.
%
% * 24-07-2008: Made a change to the C.data matrix.
%
% * 31-07-2008: Added the partial field to the transformation structure.
%
% * 01-08-2008: Added partial function for the conditioning variables.
%
% * 08-08-2008: The last term of the partial function is now assumed to be treated as
%               as x-variable. The setting of the "x" field is not performed in the data
%               construction file, but internally by YADA. The reason is that the partial
%               function is used for decompositions of the variables and hence, the partial
%               derivative is evaluated at the sum of the components, while the level of the
%               variable is evaluated for the different components separately.
%
% * 12-08-2008: Added the annualizepartial field.
%
% * 20-08-2008: Added the exportvariables field.
%
% * 28-08-2008: Added the createdata field.
%
% * 23-09-2008: Added the "createafterpartial" field to the createdata structure.
%
% * 23-01-2009: Updated the documentation.
%
% * 26-02-2009: Changed the ZData matrix values.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% Read data from a file. Make sure that the full path is given since this file will be moved
% into the sub-driectory tmp of pwd (pwd = working directory, i.e. the directory where YADA.m
% is located)
%
% Here we read in simulated data from the An Schorfheide model
%
StructureForData.Y.data = wk1read([pwd '\example\AnSchorfheide\data\AnSchorfheideData.wk1']);
%
% transpose the data since YADA prefers it that way
%
StructureForData.Y.data = StructureForData.Y.data';
StructureForData.Y.names = {'YGR' 'INFL' 'INT'};
%
% the comparison data is here with an constant interest rate for the last 7 years.
%
StructureForData.Y.comparisondata = [StructureForData.Y.data(1:2,:);[StructureForData.Y.data(3,1:72) StructureForData.Y.data(3,72)*ones(1,28)]];
%
% These fields make it possible to transform the data into desired formats.
%
StructureForData.Y.transformation.YGR.fcn = '100*(exp(YGR/100)-1)';
StructureForData.Y.transformation.YGR.partial = 'exp(YGR/100).*PartYGR';
StructureForData.Y.transformation.YGR.annualizefcn = '100*(exp((1/100)*(YGR(4:length(YGR))+YGR(3:length(YGR)-1)+YGR(2:length(YGR)-2)+YGR(1:length(YGR)-3)))-1)';
StructureForData.Y.transformation.YGR.annualizepartial = 'exp((1/100)*(YGR(4:length(YGR))+YGR(3:length(YGR)-1)+YGR(2:length(YGR)-2)+YGR(1:length(YGR)-3))).*(PartYGR(4:length(PartYGR))+PartYGR(3:length(PartYGR)-1)+PartYGR(2:length(PartYGR)-2)+PartYGR(1:length(PartYGR)-3))';
StructureForData.Y.transformation.YGR.initial = [];
StructureForData.Y.transformation.YGR.x = [];
StructureForData.Y.transformation.YGR.invertfcn = '100*log(1+(YGR/100))';
StructureForData.Y.transformation.YGR.invertinitial = [];
StructureForData.Y.transformation.YGR.invertx = [];
StructureForData.Y.transformation.YGR.exportfcn = '100*exp(cumsum(YGR)/100)';
StructureForData.Y.transformation.YGR.exportinitial = 0;
StructureForData.Y.transformation.YGR.exportx = [];
StructureForData.Y.transformation.YGR.exporttitle = 'Real GDP';
StructureForData.Y.transformation.YGR.exportname = 'DY';
StructureForData.Y.transformation.INFL.fcn = '400*(exp(INFL/400)-1)';
StructureForData.Y.transformation.INFL.partial = 'exp(INFL/400).*PartINFL';
StructureForData.Y.transformation.INFL.annualizefcn = '100*(exp((1/400)*(INFL(4:length(INFL))+INFL(3:length(INFL)-1)+INFL(2:length(INFL)-2)+INFL(1:length(INFL)-3)))-1)';
StructureForData.Y.transformation.INFL.annualizepartial = '(1/4)*exp((1/400)*(INFL(4:length(INFL))+INFL(3:length(INFL)-1)+INFL(2:length(INFL)-2)+INFL(1:length(INFL)-3))).*(PartINFL(4:length(PartINFL))+PartINFL(3:length(PartINFL)-1)+PartINFL(2:length(PartINFL)-2)+PartINFL(1:length(PartINFL)-3))';
StructureForData.Y.transformation.INFL.initial = [];
StructureForData.Y.transformation.INFL.x = [];
StructureForData.Y.transformation.INFL.invertfcn = '400*log(1+(INFL/400))';
StructureForData.Y.transformation.INFL.invertinitial = [];
StructureForData.Y.transformation.INFL.invertx = [];
StructureForData.Y.transformation.INFL.exportfcn = '400*(exp(INFL/400)-1)';
StructureForData.Y.transformation.INFL.exportinitial = [];
StructureForData.Y.transformation.INFL.exportx = [];
StructureForData.Y.transformation.INFL.exporttitle = 'Inflation';
StructureForData.Y.transformation.INFL.exportname = 'DP';
StructureForData.Y.transformation.INT.fcn = 'INT';
StructureForData.Y.transformation.INT.partial = '(1-(0*INT)).*PartINT';
StructureForData.Y.transformation.INT.annualizefcn = 'INT';
StructureForData.Y.transformation.INT.annualizepartial = '(1-(0*INT)).*PartINT';
StructureForData.Y.transformation.INT.initial = [];
StructureForData.Y.transformation.INT.x = [];
StructureForData.Y.transformation.INT.invertfcn = 'INT';
StructureForData.Y.transformation.INT.invertinitial = [];
StructureForData.Y.transformation.INT.invertx = [];
StructureForData.Y.transformation.INT.exportfcn = 'INT';
StructureForData.Y.transformation.INT.exportinitial = [];
StructureForData.Y.transformation.INT.exportx = [];
StructureForData.Y.transformation.INT.exporttitle = 'Interest rate';
StructureForData.Y.transformation.INT.exportname = 'INT';
%
StructureForData.Y.levels = [0 0 1];
StructureForData.Y.annual = [1 1 0];
StructureForData.Y.annualscale = [1 0.25 1];
StructureForData.Y.BVAR = [1 2 3];
StructureForData.X.data = ones(1,100);
StructureForData.X.extradata = ones(1,20);
StructureForData.X.names = {'const'};
StructureForData.X.BVAR = [1];
StructureForData.sample = [1980 1 2004 4];
StructureForData.frequency = 'q';
StructureForData.percentiles = [2.5 5 15 25 75 85 95 97.5];
StructureForData.Z.data = [StructureForData.Y.data(1,1:92);StructureForData.Y.data(3,1:84) 5.6*ones(1,8)];
StructureForData.Z.names = {'YGR' 'INT'};
StructureForData.Z.K1 = [1 0;0 0;0 1];
StructureForData.Z.K2 = zeros(3,2);
StructureForData.U.data = [];
StructureForData.Z.transformation.YGR.fcn = '100*(exp(YGR/100)-1)';
StructureForData.Z.transformation.YGR.partial = 'exp(YGR/100).*PartYGR';
StructureForData.Z.transformation.YGR.annualizefcn = '100*(exp((1/100)*(YGR(4:length(YGR))+YGR(3:length(YGR)-1)+YGR(2:length(YGR)-2)+YGR(1:length(YGR)-3)))-1)';
StructureForData.Z.transformation.YGR.annualizepartial = 'exp((1/100)*(YGR(4:length(YGR))+YGR(3:length(YGR)-1)+YGR(2:length(YGR)-2)+YGR(1:length(YGR)-3))).*(PartYGR(4:length(PartYGR))+PartYGR(3:length(PartYGR)-1)+PartYGR(2:length(PartYGR)-2)+PartYGR(1:length(PartYGR)-3))';
StructureForData.Z.transformation.YGR.initial = [];
StructureForData.Z.transformation.YGR.x = [];
StructureForData.Z.transformation.YGR.invertfcn = '100*log(1+(YGR/100))';
StructureForData.Z.transformation.YGR.invertinitial = [];
StructureForData.Z.transformation.YGR.invertx = [];
StructureForData.Z.transformation.YGR.exportfcn = '100*exp(cumsum(YGR)/100)';
StructureForData.Z.transformation.YGR.exportinitial = 0;
StructureForData.Z.transformation.YGR.exportx = [];
StructureForData.Z.transformation.YGR.exportname = 'DY';
StructureForData.Z.transformation.YGR.exporttitle = 'Real GDP';
StructureForData.Z.transformation.INT.fcn = 'INT';
StructureForData.Z.transformation.INT.partial = '(1-(0*INT)).*PartINT';
StructureForData.Z.transformation.INT.annualizefcn = 'INT';
StructureForData.Z.transformation.INT.annualizepartial = '(1-(0*INT)).*PartINT';
StructureForData.Z.transformation.INT.initial = [];
StructureForData.Z.transformation.INT.x = [];
StructureForData.Z.transformation.INT.invertfcn = 'INT';
StructureForData.Z.transformation.INT.invertinitial = [];
StructureForData.Z.transformation.INT.invertx = [];
StructureForData.Z.transformation.INT.exportfcn = 'INT';
StructureForData.Z.transformation.INT.exportinitial = [];
StructureForData.Z.transformation.INT.exportx = [];
StructureForData.Z.transformation.INT.exporttitle = 'Interest rate';
StructureForData.Z.transformation.INT.exportname = 'INT';
%
%StructureForData.C.data = [StructureForData.Y.data(1,1:72) ones(1,31)*NaN;StructureForData.Y.data(2,1:68) NaN*ones(1,35);StructureForData.Y.data(3,1:60) StructureForData.Y.data(3,61)*ones(1,43)];
StructureForData.C.data = [StructureForData.Y.data(1,1:72) ones(1,28)*StructureForData.Y.data(1,73);StructureForData.Y.data(2,:);StructureForData.Y.data(3,:)];
%StructureForData.C.comparisondata = [StructureForData.Y.data(1,1:80) ones(1,23)*NaN;StructureForData.Y.data(2,1:92) NaN*ones(1,11);StructureForData.Y.data(3,1:80) StructureForData.Y.data(3,81)*ones(1,23)];
StructureForData.C.comparisondata = StructureForData.Y.data;
StructureForData.C.names = {'YGR' 'INFL' 'INT'};
StructureForData.C.transformation = StructureForData.Y.transformation;
StructureForData.C.exportprepend = [100*(exp(StructureForData.C.data(1,1)/100)-1)+100;400*(exp(StructureForData.C.data(2,1)/400)-1);StructureForData.C.data(3,1)];
StructureForData.C.export.data = [StructureForData.C.data(3,1)-StructureForData.C.data(2,1) StructureForData.C.data(3,:)-StructureForData.C.data(2,:)];
StructureForData.C.export.names = {'Real interest rate';'R'};
StructureForData.C.exportvariables.names = {'DY' 'DP'};
StructureForData.C.exportvariables.titles = {'Twice YGR' 'INFL plus constant adjusted for trend'};
StructureForData.C.exportvariables.DY.fcn = '2*YGR';
StructureForData.C.exportvariables.DY.x = [];
StructureForData.C.exportvariables.DY.y = {'YGR'};
StructureForData.C.exportvariables.DY.subsample = [2 101];
StructureForData.C.exportvariables.DY.priordata = 2*StructureForData.C.exportprepend(1,1);
StructureForData.C.exportvariables.DP.fcn = 'x1+(INFL/4)-x2';
StructureForData.C.exportvariables.DP.x = [ones(101,1) (0:0.1:10)'];
StructureForData.C.exportvariables.DP.y = {'INFL'};
StructureForData.C.exportvariables.DP.subsample = [1 101];
StructureForData.C.exportvariables.DP.priordata = [];
%
StructureForData.C.createdata.names = {'RR'};
StructureForData.C.createdata.RR.createbeforefcn = 'INT-INFL';
StructureForData.C.createdata.RR.createafterfcn = 'INT-INFL';
StructureForData.C.createdata.RR.createafterpartial = {'(1-(0*INT)-(0*INFL)).*PartINT' '-(1+(0*INT)+(0*INFL)).*PartINFL'};
StructureForData.C.createdata.RR.createx = [];
StructureForData.C.createdata.RR.createcompx = [];
StructureForData.C.createdata.RR.createy = {'INT' 'INFL'};
StructureForData.C.createdata.RR.before = 0;
StructureForData.C.createdata.RR.fcn = 'RR';
StructureForData.C.createdata.RR.partial = '(1-(0*RR)).*PartRR';
StructureForData.C.createdata.RR.annualizefcn = 'RR';
StructureForData.C.createdata.RR.annualizepartial = '(1-(0*RR)).*PartRR';
StructureForData.C.createdata.RR.initial = [];
StructureForData.C.createdata.RR.x = [];
StructureForData.C.createdata.RR.compinitial = [];
StructureForData.C.createdata.RR.compx = [];
%
StructureForData.C.summaryorder = [3 2 1];
StructureForData.C.startperiod = [1997 2];
StructureForData.C.C1 = eye(3);
StructureForData.C.C2 = zeros(3,3);
StructureForData.CBar.data = [];
StructureForData.CBar.comparisondata = [];

%
% end of DataConstFile.m
%
