function lnjac = logJacobian(phi,thetaIndex,UniformBounds)
% logJacobian: Calculates the log of the Jacobian in the transformation from theta to phi, where
%              theta has bounded support and phi has unbounded support.
%
% USAGE: lnjac = logJacobian(phi,thetaIndex,UniformBounds)
%
% REQUIRED INPUT:  phi (vector) with parameters whose support is unbounded.
%
%                  thetaIndex (vector) indicating the type of distribution the parameters in theta are drawn from.
%                             If 1, then the distribution is gamma, inverse gamma, or left truncated normal.
%                             If 2, then the distribution is standardized beta, and if 3 then it is uniform. If 0,
%                             then the distribution is normal and we can ignore it since log(1) = 0.
%
%                  UniformBounds (matrix) with the same number of rows as parameters and 2 columns. The columns are
%                             0 1 except for the uniform and the beta distribution, where they represent the upper
%                             and lower bounds.
%
% REQUIRED OUTPUT: lnjac (scalar) being the value of the log of the Jacobian of the function theta = f(phi).
%
% NOTE: The transformation function used for the gamma, inverse gamma, and left truncated normal is:
%
%       (1)     theta = exp(phi)+LowerBound
%
%       while for the beta and the uniform distribution, the transformation function is:
%
%       (2)     theta = [a+b*exp(phi)] ./ (1+exp(phi))
%
%       For (1) we therefore have that:
%
%               log(d theta/d phi) = phi
%
%       and for (2) we have that
%
%               log(d theta/d phi) = log(a+b)+phi - 2*log(1-exp(phi))
%
%       Since the log of the (joint) prior for theta is the sum of the logs of the individual prior
%       densities, we need to sum individual log Jacobians.
%         
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: August 7, 2006.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 25-8-2006:  Added the input argument UniformBounds needed to handle parameters with a uniform prior.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 8-7-2008:   Fixed a bug where the exponential function is applied to 0 (yielding 1) needs to be multiplied by 0
%               to give the correct value.
%
% * 23-7-2008:  Fixed a bug where the columns of UniformBounds were summed, instead of the second minus the first.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

lnjac = sum(phi .* ((thetaIndex==1)+(thetaIndex==2)+(thetaIndex==3)))+sum(log(UniformBounds(:,2)-UniformBounds(:,1)))-(2*sum(log(1+exp(phi .* ((thetaIndex==2)+(thetaIndex==3))) .* ((thetaIndex==2)+(thetaIndex==3)) )));

%
% end of logJacobian.m
%
