function DSGEModel = VerifyConditioningInformation(DSGEModel,DataStructure)
% VerifyConditioningInformation: Checks if there is valid information for running conditional
%                                forecasts in the DataStructure structure.
%
% USAGE:
%
%       DSGEModel = VerifyConditioningInformation(DSGEModel,DataStructure)
%
% REQUIRED INPUT:  DSGEModel (structure) whose fields contain all required information
%                            about the model, such as input files, variable and sample
%                            data. Actual observations are not stored in this file.
%
%                  DataStrcuture (structure) where this function looks at the fields Z.data
%                            Z.names, Z.K1, Z.K2, and U.data.
%
% REQUIRED OUTPUT: DSGEModel (structure) whose fields contain all required information
%                            about the model, such as input files, variable and sample
%                            data. Actual observations are not stored in this file.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: March 8, 2007.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 26-3-2007:  Fixed a bug when initialising "ZVariableNames".
%
% * 25-4-2007:  Updated the documnetation.
%
% * 23-10-2007: The fields "ZTransformation" and "ZTransformationOriginal" are now taken into account.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
DSGEModel.Z = [];
DSGEModel.ZVariableNames = '';
DSGEModel.K1 = [];
DSGEModel.K2 = [];
DSGEModel.U = [];
DSGEModel.ZTransformation = [];
%
DSGEModel.ZTransformationOriginal = [];
%
if isfield(DSGEModel,'ConditionalShocks')==0;
   DSGEModel.ConditionalShocks = [];
end;
if isfield(DSGEModel,'ConditionalVariables')==0;
   DSGEModel.ConditionalVariables = [];
end;
if isfield(DataStructure,'Z')==0;
   DSGEModel.ConditionalShocks = [];
   DSGEModel.ConditionalVariables = [];
   return;
end;
if isfield(DataStructure.Z,'data')==0;
   DSGEModel.ConditionalShocks = [];
   DSGEModel.ConditionalVariables = [];
   return;
end;
if isfield(DataStructure.Z,'names')==0;
   DSGEModel.ConditionalShocks = [];
   DSGEModel.ConditionalVariables = [];
   return;
end;
if isfield(DataStructure.Z,'K1')==0;
   DSGEModel.ConditionalShocks = [];
   DSGEModel.ConditionalVariables = [];
   return;
end;
if isfield(DataStructure.Z,'K2')==0;
   DSGEModel.ConditionalShocks = [];
   DSGEModel.ConditionalVariables = [];
   return;
end;
if isfield(DataStructure,'U')==0;
   DSGEModel.ConditionalShocks = [];
   DSGEModel.ConditionalVariables = [];
   return;
end;
if isfield(DataStructure.U,'data')==0;
   DSGEModel.ConditionalShocks = [];
   DSGEModel.ConditionalVariables = [];
   return;
end;
%
% all fields are available.
%
% 1. check if dimensions match for Z, ZVariableNames, and K1
%
K1 = DataStructure.Z.K1;
Z = DataStructure.Z.data;
ZVariableNames = DataStructure.Z.names;
K2 = DataStructure.Z.K2;
U = DataStructure.U.data;
[n m] = size(K1);
if size(Z,1)>size(Z,2);
   Z = Z';
end;
Tz = size(Z,2);
if (Tz==0)|(m==0)|(m~=size(Z,1))|(m~=length(ZVariableNames))|(m~=rank(K1));
   DSGEModel.ConditionalShocks = [];
   DSGEModel.ConditionalVariables = [];
   return;
end;
%
% allow for empty inputs for K2 and U
%
if isempty(K2)==1;
   K2 = zeros(n,m);
end;
if isempty(U)==1;
   U = zeros(m,Tz);
end;
%
% make sure that K1 and K2 have same dimensions
%
if (size(K2,1)~=n)|(size(K2,2)~=m);
   DSGEModel.ConditionalShocks = [];
   DSGEModel.ConditionalVariables = [];
   return;
end;
if size(U,1)>size(U,2);
   U = U';
end;
Tu = size(U,2);
%
% make sure that U has the m rows
%
if (m~=size(U,1))|(Tu==0);
   DSGEModel.ConditionalShocks = [];
   DSGEModel.ConditionalVariables = [];
   return;
end;
%
% Everything looks OK. we can now let the DSGEModel structure hold
% these data
%
DSGEModel.Z = Z;
DSGEModel.ZVariableNames = char(ZVariableNames);
DSGEModel.K1 = K1;
DSGEModel.K2 = K2;
DSGEModel.U = U;
if isfield(DataStructure.Z,'transformation')==1;
   DSGEModel.ZTransformation = DataStructure.Z.transformation;
end;
DSGEModel.ZTransformationOriginal = DSGEModel.ZTransformation;
%
if (isempty(DSGEModel.ConditionalVariables)==1)|(max(DSGEModel.ConditionalVariables)>m);
   DSGEModel.ConditionalVariables = (1:m);
end;
if (isempty(DSGEModel.ConditionalShocks)==1)|(max(DSGEModel.ConditionalShocks)>length(DSGEModel.StateShockPositions));
   DSGEModel.ConditionalShocks = (1:m);
end;

%
% end of VerifyConditioningInformation.m
%
