function [z,Status,ErrorStr,HeaderStr,FcnStr] = TransformData(y,Transformation,CurrVariable,AnnualizeData)
% TransformData: Transforms the vector input data in y using the transformation information provided
%                through the structure Transformation looking at the field name provided by CurrVariable
%
% USAGE:
%
%       [z,Status,ErrorStr,HeaderStr,FcnStr] = TransformData(y,Transformation,CurrVariable,AnnualizeData)
%
% REQUIRED INPUT:  y (vector) of dimension Tx1 with the main data to use for the transformation.
%
%                  Transformation (structure) with field names. The field name to use is determined
%                                 by CurrVariable. This field name has subfields "fcn", "annualizefcn",
%                                 "initial", and "x" that are used for the transformation of y.
%
%                  CurrVariable (string vector) with the name of the variable to transform. This
%                                 name should correspond to the field name in the structure Transformation
%                                 that will be used.
%
%                  AnnualizeData (boolean) that takes on the value 1 if the annualizefcn function should
%                                 be run and 0 if the fcn function should be used.
%
% REQUIRED OUTPUT: z (vector) of dimension T1x1 with the transformed data of the input y
%
%                  Status (boolean) that takes the value 0 if the transformation was performed without
%                                 error and 1 otherwise.
%
%                  ErrorStr (string vector) with error messages (if any).
%
%                  HeaderStr (string vector) with a suitable header message (if any).
%
%                  FcnStr (string vector) with the function string sent to the inline function.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: October 23, 2007.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 24-10-2007: Fixed a bug where "int2str" has now replaced 3 calls to "intstr".
%
% * 26-10-2007: Updated the documentation.
%
% * 30-10-2007: Updated the documentation.
%
% * 14-11-2007: Added the new input variables "FirstObs" and "LastObs" and took the new fields "x.data"
%               and "x.numinitialobs" into account.
%
% * 20-11-2007: The fields "data" and "numinitialobs" are no longer used. The inputs "FirstObs" and "LastObs"
%               have also been removed.
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%

z = y;
Status = 0;
ErrorStr = '';
HeaderStr = '';
FcnStr = '';
%
if isfield(Transformation,CurrVariable)==1;
   CurrField = getfield(Transformation,CurrVariable);
   if (isfield(CurrField,'fcn')==1)&(isfield(CurrField,'annualizefcn')==1)&(isfield(CurrField,'initial')==1)&(isfield(CurrField,'x')==1);
      %
      % select the function string
      %
      if AnnualizeData==0;
         FcnStr = CurrField.fcn;
      else;
         FcnStr = CurrField.annualizefcn;
      end;
      %
      % remove any NaN's at the beginning of y
      %
      StartPos = NaN;
      for i=1:length(y);
         if isnan(y(i))==0;
            StartPos = i;
            break;
         end;
      end;
      if isnan(StartPos)==1;
         ErrorStr = ['The data for ' CurrVariable ' has only NaN''s. YADA is therefore unable to perform the transformation.'];
         HeaderStr = 'Bad Export Function';
         Status = 1;
         return;
      end;
      y = y(StartPos:length(y));
      %
      % check how many input arguments the inline function
      %
      NumInputs = length(symvar(FcnStr));
      %
      % make sure that time dimension is given by the rows
      %
      if size(CurrField.x,2)>size(CurrField.x,1);
         CurrField.x = CurrField.x';
      end;
      if size(CurrField.x,2)~=NumInputs-1;
         CurrField.x = [];
      end;
      %
      % deal with initial values for the variable to be transformed
      %
      if isempty(CurrField.initial)==0;
         y = [vec(CurrField.initial);y];
      end;
      RunArg = 'y';
      %
      % check if we should create the additional variables for the command line
      %
      data = CurrField.x;
      for i=1:size(data,2);
         eval(['x' int2str(i) ' = RemovePreNaNs(data(:,' int2str(i) '));']);
         RunArg = [RunArg ',x' int2str(i)];
      end;
      %
      % now we try to create the inline function
      %
      ErrorStr = '';
      try;
         if isempty(data)==1;
            YADAf = inline(FcnStr);
         else;
            YADAf = YADAInline(FcnStr);
         end;
      catch;
         ErrorStr = ['The was an error creating the inline function for ' CurrVariable '. The following message was caught: ' lasterr];
         HeaderStr = 'Bad Transformation Function';
         Status = 1;
         z = y;
      end;
      %
      % check if the inline function executes correctly
      %
      if isempty(ErrorStr)==1;
         try;
            eval(['z = YADAf(' RunArg ');']);
         catch;
            ErrorStr = ['YADA caught the following error message for the transformation function for the variable ' CurrVariable ': ' lasterr];
            HeaderStr = 'Bad Transformation Function';
            Status = 1;
            z = y;
         end;
      end;
   else;
      ErrorStr = ['The transformation data for the variable "' CurrVariable '" is not accurate. At least one of ' CurrVariable '''s the subfields "fcn", "annualizefcn", "initial", or "x" is missing. YADA is therefore unable to perform the transformation.'];
      HeaderStr = 'Bad Transformation Information';
      Status = 1;
   end;
else;
   ErrorStr = ['The transformation data for the variable "' CurrVariable '" is not accurate. The field "'  CurrVariable '" is missing.'];
   HeaderStr = 'Bad Transformation Information';
   Status = 1;
end;

%
% end of TransformData.m
%
