function phi = ThetaToPhi(theta,thetaIndex,UniformBounds,LowerBound)
% ThetaToPhi: Transforms parameters theta with support that is bounded to parameters phi
%             whose support is unbounded.
%
% USAGE:
%
%       phi = ThetaToPhi(theta,thetaIndex,UniformBounds,LowerBound)
%
% REQUIRED INPUT:  theta (vector) of dimension NumParamx1 with parameters whose support is bounded
%
%                  thetaIndex (vector) indicating the type of distribution the parameters in theta are drawn from.
%                             If 1, then the distribution is gamma, inverse gamma, or left truncated normal.
%                             If 2, then the distribution is standardized beta. If 0, then the distribution
%                             is normal. Finally, if 3, then it is uniform.
%
%                  UniformBounds (matrix) with the same number of rows as parameters and 2 columns. The columns
%                             are 0 1 except for the uniform and the beta distribution, where they represent
%                             the upper and lower bounds.
%
%                  LowerBound (vector) with the lower bounds for the distributions.
%
% REQUIRED OUTPUT: phi (vector) of same length as theta, but with the transformed parameters.
%
%
% NOTE: The transformation function used for the gamma, inverse gamma, and left truncated normal is:
%
%               phi = log(theta-LowerBound)
%
%       while for the beta and uniform distribution, the transformation function is:
%
%               phi = log((theta - a) ./ (b-theta))
%
%       where a<b represent the upper and lower bounds respectively.
%         
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: August 7, 2006.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 25-8-2006:  Added the input argument UniformBounds needed to handle parameters with a uniform prior.
%
% * 15-9-2006:  Took care of the fact that Inf*0 = NaN.
%
% * 13-11-2007: Updated the documentation.
%
% * 7-12-2007:  Fixed a bug when "log(theta-LowerBound)" yields a NaN, while thetaIndex==1 is 0 for those
%               elements.
%
% * 11-12-2007: Changed the NaN fixes to vector solutions.
%
% * 23-5-2008:  Updated the documentation.
%
% * 21-7-2008:  Updated the documentation.
%
% * 23-7-2008:  Took into account that the beta distribution also support lower and upper bounds different
%               from 0 and 1.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% parameters that should be left alone
%
phi = (theta .* (thetaIndex==0));
%
% parameters that should be log transformed
%
phi0 = (log(theta-LowerBound) .* (thetaIndex==1));
if sum(isnan(phi0))>0;
   ind = isnan(phi0).*(1:length(phi0))';
   ind = ind(ind>0);
   phi0(ind) = 0;
end;
phi = phi+phi0;
%
% transformation for parameters with upper and lower bounds
%
phi1 = (log((theta-UniformBounds(:,1))./(UniformBounds(:,2)-theta)) .* ((thetaIndex==2)+(thetaIndex==3)));
if sum(isnan(phi1))>0;
   ind = isnan(phi1).*(1:length(phi1))';
   ind = ind(ind>0);
   phi1(ind) = 0;
end;
phi = phi+phi1;

%
% end of ThetaToPhi.m
%
