function ParameterNames = ReadAdditionalParameterNames(ParameterNames,ModelParameters,OriginalModelParameters,AdditionalPositions)
% ReadAdditionalParameterNames: Reads any names of DSGE model parameters that have been added through selected
%                               functions for initializing certain parameters and parameters that should be
%                               updated.
%
% USAGE:
%
%       ParameterNames = ReadAdditionalParameterNames(ParameterNames,ModelParameters,OriginalModelParameters)
%
% REQUIRED INPUT:  ParameterNames (structure) with fields "calibrated", "beta", "gamma", "normal", "invgamma", "truncnormal",
%                             "all" and "estimated". Each field returns a string matrix with the parameter names.
%
%                  ModelParameters (structure) with fields having the names of the inidividual parameters as they are
%                             specified under the "model parameter" header in the file with prior distribution data.
%                             In addition, fields located in the "update parameters file" and the "initialize
%                             parameters file" have been added to the structure.
%
%                  OriginalModelParameters (structure), same as ModelParameters except that the fields from running the
%                             "update parameters file" and the "initialize parameters file" are missing.
%
%                  AdditionalPositions (1x4 vector) with the entries:
%
%                                [firstUpdatePosition lastUpdatePosition firstInitializePosition lastInitializePosition]
%
%                             where a 0 value means that the Updated or Initialized parameters are not available.
%
% REQUIRED OUTPUT: ParameterNames (structure) with fields "calibrated", "beta", "gamma", "normal", "invgamma", "truncnormal",
%                             "all", "estimated", and "additional". Each field returns a string matrix with the parameter names.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: August 14, 2006.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 28-8-2006:  Added the additional parameter names to the ParameterNames.all field.
%
% * 16-10-2006: Made sure the function add parameter names from the initialized parameters to the calibrated
%               instead of the additional field. The new input "AdditionalPositions" contains the necessary information.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 21-20-2008: Made sure that the field "UserVariables" is skipped.
%
% * 29-10-2008: Made sure that ParameterNames.all is updated along with ParameterNames.additional and
%               ParameterNames.calibrated.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

ParameterNames.additional = '';
allnames = fieldnames(ModelParameters);
orignames = fieldnames(OriginalModelParameters);
if length(allnames)>length(orignames);
   if AdditionalPositions(1)>0;
      %
      % we have additional parameters that are updated
      %
      for i=AdditionalPositions(1):AdditionalPositions(2);
         if strcmp(allnames{i},'UserVariables')==0;
            ParameterNames.additional = strvcat(ParameterNames.additional,allnames{i});
            ParameterNames.all = strvcat(ParameterNames.all,allnames{i});
         end;
      end;
   end;
   if AdditionalPositions(3)>0;
      %
      % we have calibrated parameters from the initialization function
      %
      for i=AdditionalPositions(3):AdditionalPositions(4);
         if strcmp(allnames{i},'UserVariables')==0;
            ParameterNames.calibrated = strvcat(ParameterNames.calibrated,allnames{i});
            ParameterNames.all = strvcat(ParameterNames.all,allnames{i});
         end;
      end;
   end;
%   np = length(allnames)-length(orignames);
%   for i=1:np;
%      ParameterNames.additional = strvcat(ParameterNames.additional,allnames{length(orignames)+i});
%   end;
%   ParameterNames.all = strvcat(ParameterNames.all,ParameterNames.additional);
end;

%
% end of ReadAdditionalParameterNames.m
%
