function InitialInvHessMat = PriorHessianMatrix(thetaDist,thetaIndex,PriorDist,UniformBounds,LowerBound)
% PriorHessianMatrix: Computes a diagonal covariance matrix for the transformed parameters using draws from
%                     the prior distribution.
%
% USAGE:
%
%       InitialInvHessMat = PriorHessianMatrix(thetaDist,thetaIndex,PriorDist,UniformBounds,LowerBound)
%
% REQUIRED INPUT:  thetaDist (vector), with integers 1 if gamma, 2 if beta, 3 if normal, 4 if invgamma, 5 if
%                            truncnormal, and 6 if uniform.
%
%                  thetaIndex (vector) indicating the type of distribution the parameters in theta are drawn from.
%                            If 1, then the distribution is gamma, inverse gamma, or left truncated normal.
%                            If 2, then the distribution is standardized beta. If 0, then the distribution
%                            is normal. Finally, if 3, then it is uniform.
%
%                  PriorDist (structure), with fields "beta", "gamma", "normal", "invgamma", "truncnormal", and "uniform" where
%                            the values are matrices with the parameter values for the distributions. In the case of
%                            the normal, beta, and gamma these parameter values are the mean and the standard deviation.
%                            For the invgamma it is the s and q parameters, where q is a positive integer (degrees of
%                            freedom). Finally, for the left truncated normal there's a third parameter, the lower bound.
%
%                  UniformBounds (matrix) with the same number of rows as parameters and 2 columns. The columns
%                             are 0 1 except for the uniform and the beta distribution, where they represent
%                             the upper and lower bounds.
%
%                  LowerBound (vector) with the lower bounds for the distributions.
%
% REQUIRED OUTPUT: InitialInvHessMat (matrix) whose diagonal elements are the estimated variances of the transformed
%                             parameters using draws from the prior distribution.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: January 29, 2009.
%                        This version: January 30, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 30-01-2009: Finished the function.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
InitialInvHessMat = [];
%
% draw from the prior distribution
%
n_beta = 0;
n_gamma = 0;
n_normal = 0;
n_invgamma = 0;
n_truncnormal = 0;
n_uniform = 0;
thetaDraws = zeros(length(thetaDist),5000);
for i=1:length(thetaDist);
   if thetaDist(i)==1;
      %
      % we have a gamma prior
      %
      n_gamma = n_gamma+1;
      thetaDraws(i,:) = LowerBound(i)+GammaRndFcn(PriorDist.gamma_ab(n_gamma,1),PriorDist.gamma_ab(n_gamma,2),5000);
   elseif thetaDist(i)==2;
      %
      % we have a beta prior
      %
      n_beta = n_beta+1;
      if size(PriorDist.beta,2)==4;
         %
         % get the lower and upper bounds (if available from PriorDist.)
         %
         c = PriorDist.beta(n_beta,3);
         d = PriorDist.beta(n_beta,4);
      else;
         c = 0;
         d = 1;
      end;
      thetaDraws(i,:) = BetaRndFcn(PriorDist.beta_ab(n_beta,1),PriorDist.beta_ab(n_beta,2),c,d,5000);
   elseif thetaDist(i)==3;
      %
      % we have a normal prior
      %
      n_normal = n_normal+1;
      thetaDraws(i,:) = NormalRndFcn(PriorDist.normal(n_normal,1),PriorDist.normal(n_normal,2),5000);
   elseif thetaDist(i)==4;
      %
      % we have an inverse gamma prior
      %
      n_invgamma = n_invgamma+1;
      p_par1 = PriorDist.invgamma(n_invgamma,1);
      p_par2 = PriorDist.invgamma(n_invgamma,2);
      thetaDraws(i,:) = LowerBound(i)+InvGammaRndFcn(p_par1-LowerBound(i),p_par2,5000);
   elseif thetaDist(i)==5;
      %
      % we have a left truncated normal prior
      %
      n_truncnormal = n_truncnormal+1;
      thetaDraws(i,:) = LTNormalRndFcn(PriorDist.truncnormal(n_truncnormal,1),PriorDist.truncnormal(n_truncnormal,2),PriorDist.truncnormal(n_truncnormal,3),5000);
   else;
      %
      % we have a uniform prior
      %
      n_uniform = n_uniform+1;
      thetaDraws(i,:) = UniformRndFcn(PriorDist.uniform(n_uniform,1),PriorDist.uniform(n_uniform,2),5000);
   end;
end;
%
% transform all parameters to the real line
%
phiDraws = zeros(size(thetaDraws));
for i=1:5000;
   phiDraws(:,i) = ThetaToPhi(thetaDraws(:,i),thetaIndex,UniformBounds,LowerBound);
end;
%
% compute the average
%
phiAverage = (1/5000)*sum(phiDraws,2)*ones(1,5000);
phiDraws = phiDraws-phiAverage;
%
% compute the variances and make it a diagonal matrix
%
InitialInvHessMat = (1/5000)*sum((phiDraws.*phiDraws),2);
%
% check to make sure that elements less than 0.0001 are set to this minimum value
%
CheckVec = (InitialInvHessMat<10^(-6)).*(1:length(InitialInvHessMat))';
CheckVec = CheckVec(CheckVec>0);
if isempty(CheckVec)==0;
   InitialInvHessMat(CheckVec) = 10^(-6);
end;
InitialInvHessMat = diag(InitialInvHessMat);

%
% end of PriorHessianMatrix.m
%
