function theta = PhiToTheta(phi,thetaIndex,UniformBounds,LowerBound)
% PhiToTheta: Transforms parameters phi whose support in unbounded to parameters theta
%             whose support is bounded.
%
% USAGE:
%
%       theta = PhiToTheta(phi,thetaIndex,UniformBounds,LowerBound)
%
% REQUIRED INPUT:  phi (vector) with parameters whose support is unbounded.
%
%                  thetaIndex (vector) indicating the type of distribution the parameters in theta are drawn from.
%                             If 1, then the distribution is gamma, inverse gamma, or left truncated normal.
%                             If 2, then the distribution is standardized beta. If 0, then the distribution
%                             is normal.
%
%                  UniformBounds (matrix) with the same number of rows as parameters and 2 columns. The columns
%                             are 0 1 except for the uniform and the beta distribution, where they represent
%                             the upper and lower bounds.
%
%                  LowerBound (vector) with the lower bounds for the distributions.
%
% REQUIRED OUTPUT: theta (vector) of same length as phi, but with the transformed parameters.
%
%
% NOTE: The transformation function used for the gamma, inverse gamma, and left truncated normal is:
%
%               theta = exp(phi)+LowerBound
%
%       while for the beta distribution, the transformation function is:
%
%               theta = exp(phi) ./ (1+exp(phi))
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European central Bank.
%
%                       First version: August 7, 2006.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 25-8-2006:  Added the input argument UniformBounds needed to handle parameters with a uniform prior.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 23-7-2008:  Took into account that the beta distribution also support lower and upper bounds different
%               from 0 and 1.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

theta = (phi .* (thetaIndex==0))+((exp(phi)+LowerBound) .* (thetaIndex==1))+(((UniformBounds(:,1)+(UniformBounds(:,2).*exp(phi)))./(1+exp(phi))) .* ((thetaIndex==2)+(thetaIndex==3)));

%
% end of PhiToTheta.m
%
