function [PredEventData,PredictionEvent,ShockMean,ShockNames,StateCov,ShockCov,MeasureCov,ParameterCov] = LoadDSGEPredictionUncertainty(OutputDirectory,CondStr,ShockMethod,NameOfModel,MaxPH,NumDraws,NumPredPaths,AnnualizeData,TransData,CurrChain)
% LoadDSGEPredictionUncertainty: Loads the results for the prediction uncertainty decomposition for a DSGE model from disk. It
%                                also loads any data it finds about prediction events.
%
% USAGE:
%
%       [PredEventData,PredictionEvent,ShockMean,ShockNames,StateCov,ShockCov,MeasureCov,ParameterCov] = LoadDSGEPredictionUncertainty(OutputDirectory,CondStr,ShockMethod,NameOfModel,MaxPH,NumDraws,NumPredPaths,AnnualizeData,TransData,CurrChain)
%
% REQUIRED INPUT:  OutputDirectory (string vector) with the full path to the base output directory.
%
%                  CondStr (string vector) that is either "Cond" or "".
%
%                  ShockMethod (string vector) that is either "Direct-", "WaggonerZha-", or "".
%
%                  NameOfModel (string vector) with the name of the model.
%
%                  MaxPH (integer) with the length of the prediction horizon.
%
%                  NumDraws (integer) with the total number of parameter draws.
%
%                  NumPredPaths (integer) with tht total number of prediction paths per parameter draw.
%
%                  AnnualizeData (boolean) that take the value 1 if annualized results should be checked and 0 otherwise.
%
%                  TransData (boolean) that is 1 if the prediction paths should be transformed and 0 otherwise.
%
%                  CurrChain (integer) with the number of the currently used Markov chain.
%
% REQUIRED OUTPUT: PredEventData (n x 3) matrix where the first column holds the number of times
%                                the PredPaths data satisfies the event, the second hold the number
%                                of times the PredPaths data is greater than the event, while the
%                                third column holds the number of times the event has been tested.
%
%                  PredictionEvent (n x 3) matrix with the prediction event information. The first
%                                column holds the upper bound, the second the lower bound for the event,
%                                while the third holds the event length in periods.
%
%                  ShockMean (q x MaxPH) matrix with the mean of the shocks under conditioning.
%
%                  StateCov (n x n x MaxPH) matrix with the estimated state uncertainty.
%
%                  ShockNames (q x qstr) string matrix with the names of the economic shocks.
%
%                  ShockCov (n x n x MaxPH) matrix with the estimated shock uncertainty.
%
%                  MeasureCov (n x n) matrix the the estimated measurement error uncertainty.
%
%                  ParameterCov (n x n x MaxPH) matrix with the estimated parameter uncertainty.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: February 23, 2007.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 12-3-2007:  Added the input "CondStr" to handle conditional predictive distributions.
%
% * 4-5-2007:   Added the outputs "PredEventData" and "PredictionEvent".
%
% * 10-5-2007:  Removed two unnecessary lines of code.
%
% * 6-6-2007:   The PredEventData matrix now has three columns.
%
% * 13-11-2007: Updated the documentation.
%
% * 18-12-2007: Added the outputs "ShockMean", "ShockNames" and the input "ShockMethod".
%
% * 19-12-2007: The input "ShockMethod" can now be an empty string.
%
% * 5-5-2008:   Added the input variable "TransData".
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize the output
%
PredEventData = [];
PredictionEvent = [];
StateCov = [];
ShockCov = [];
MeasureCov = [];
ParameterCov = [];
ShockMean = [];
ShockNames = '';
if AnnualizeData==0;
   if TransData==0;
      file = [OutputDirectory '\predictions\DSGE-' CondStr 'Prediction-Original-' ShockMethod NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '-' num2str(NumDraws,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.mat'];;
   else;
      file = [OutputDirectory '\predictions\DSGE-' CondStr 'Prediction-Transformed-' ShockMethod NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '-' num2str(NumDraws,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.mat'];;
   end;
else;
   file = [OutputDirectory '\predictions\DSGE-' CondStr 'Prediction-Annual-' ShockMethod NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '-' num2str(NumDraws,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.mat'];;
end;
if FileExist(file)==1;
   PredData = load(file);
   if isfield(PredData,'StateCov')==1;
      StateCov = PredData.StateCov;
      ShockCov = PredData.ShockCov;
      MeasureCov = PredData.MeasureCov;
      ParameterCov = PredData.ParameterCov;
   end;
   if isfield(PredData,'PredEventData')==1;
      PredEventData = PredData.PredEventData;
   end;
   if isfield(PredData,'PredictionEvent')==1;
      PredictionEvent = PredData.PredictionEvent;
   end;
   if isfield(PredData,'ShockMean')==1;
      ShockMean = PredData.ShockMean;
   end;
   if isfield(PredData,'ShockNames')==1;
      ShockNames = PredData.ShockNames;
   end;
end;

%
% end of LoadDSGEPredictionUncertainty.m
%
