function PredEventData = LoadDSGEPredictionPaths(PredictionEvent,OutputDirectory,CondStr,ShockMethod,NameOfModel,MaxPH,NumDraws,NumPredPaths,AnnualizeData,TransData,CurrChain,CurrINI)
% LoadDSGEPredictionPaths: Loads prediction paths for the DSGE model from disk and computes the prediction event data.
%
% USAGE:
%
%       PredEventData = LoadDSGEPredictionPaths(PredictionEvent,OutputDirectory,CondStr,ShockMethod,NameOfModel,MaxPH,NumDraws,NumPredPaths,AnnualizeData,TransData,CurrChain,CurrINI)
%
% REQUIRED INPUT:  PredictionEvent (n x 3) matrix with the prediction event information. The first
%                                  column holds the upper bound, the second the lower bound for the event,
%                                  while the third holds the event length in periods.
%
%                  OutputDirectory (string vector) with the full path to the base output directory.
%
%                  CondStr (string vector) that is either "Cond" or "".
%
%                  ShockMethod (string vector) that is either "Direct-", "WaggonerZha-", or "".
%
%                  NameOfModel (string vector) with the name of the model.
%
%                  MaxPH (integer) with the length of the prediction horizon.
%
%                  NumDraws (integer) with the total number of parameter draws.
%
%                  NumPredPaths (integer) with tht total number of prediction paths per parameter draw.
%
%                  AnnualizeData (boolean) that take the value 1 if annualized results should be checked and 0 otherwise.
%
%                  TransData (boolean) that is 1 if the prediction paths should be transformed and 0 otherwise.
%
%                  CurrChain (integer) with the number of the currently used Markov chain.
%
%                  CurrINI (structure) with initialization information.
%
% REQUIRED OUTPUT: PredEventData (n x 3) matrix where the first column holds the number of times
%                                  the PredPaths data satisfies the event, the second hold the number
%                                  of times the PredPaths data is greater than the event, while the
%                                  third column holds the number of times the event has been tested.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: May 11, 2007.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 6-6-2007:   The PredEventData matrix now has three columns.
%
% * 13-11-2007: Updated the documentation.
%
% * 19-12-2007: Added the input "ShockMethod".
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 4-4-2008:   Added time information to the wait dialog.
%
% * 7-4-2008:   The new "PredEventData" is now stored in the last mat-file.
%
% * 5-5-2008:   Added the input variable "TransData".
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
n = size(PredictionEvent,1);
PredEventData = zeros(n,3);
%
% check if we have already calculated the PredEventData for the PredictionEvent
%
if AnnualizeData==0;
   if TransData==0;
      file = [OutputDirectory '\predictions\DSGE-' CondStr 'Prediction-Original-' ShockMethod NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '-' num2str(NumDraws,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.mat'];;
   else;
      file = [OutputDirectory '\predictions\DSGE-' CondStr 'Prediction-Transformed-' ShockMethod NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '-' num2str(NumDraws,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.mat'];;
   end;
else;
   file = [OutputDirectory '\predictions\DSGE-' CondStr 'Prediction-Annual-' ShockMethod NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '-' num2str(NumDraws,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.mat'];;
end;
%
% load the data
%
PredData = load(file);
if (isfield(PredData,'PredictionEvent')==1)&(isfield(PredData,'PredEventData')==1)&(isempty(PredData.PredEventData)==0)&(isequal(PredictionEvent,PredData.PredictionEvent)==1);
   %
   % the PredEventData has already been calculated.
   %
   PredEventData = PredData.PredEventData;
   return;
end;
%
%
if isempty(CondStr)==1;
   HeadStr = 'Predictive Distributions';
else;
   HeadStr = 'Conditional Predictive Distributions';
end;
%
% set up a wait dialog
%
txt = ['Please wait while YADA computes the prediction event probabilities for the observed variables. The data are loaded sequentially from disk and it may therefore take some time. Calculation started at: ' StringTrim(datestr(now,14)) '.'];
WaitHandle = WaitDLG(txt,'information',HeadStr,500,150,CurrINI,0);
WaitControls = get(WaitHandle,'UserData');
%
for ParamIndex=1:NumDraws;
   if AnnualizeData==0;
      if TransData==0;
         file = [OutputDirectory '\predictions\DSGE-' CondStr 'Prediction-Original-' ShockMethod NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '-' num2str(ParamIndex,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.mat'];;
      else;
         file = [OutputDirectory '\predictions\DSGE-' CondStr 'Prediction-Transformed-' ShockMethod NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '-' num2str(ParamIndex,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.mat'];;
      end;
   else;
      file = [OutputDirectory '\predictions\DSGE-' CondStr 'Prediction-Annual-' ShockMethod NameOfModel '-' num2str(CurrChain,'%0.0f') '-' num2str(MaxPH,'%0.0f') '-' num2str(ParamIndex,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPredPaths,'%0.0f') '.mat'];;
   end;
   %
   % load the data
   %
   PredData = load(file);
   PredPaths = PredData.PredPaths;
   %
   % compute the prediction event data
   %
   PredEventData = PredEventData+CalculatePredictionEvents(PredPaths,PredictionEvent);
   %
   if ParamIndex==NumDraws;
      %
      % append PredictionEvent and PredEventData to file.
      %
      save(file,'PredEventData','PredictionEvent','-APPEND');
   end;
   clear('PredData');
   drawnow;
end;
%
% close the wait dialog
%
set(WaitControls.text,'UserData','done');
delete(WaitHandle);
drawnow;
pause(0.02);

%
% end of LoadDSGEPredictionPaths.m
%
