function [PostSample,thetaPostSample,LogPost,AcceptedDraws,NumBurnin] = LoadDSGEPosteriorDraws(DSGEModel,CurrINI,CurrentChain)
% LoadDSGEPosteriorDraws: Loads the data from posterior estimation.
%
% USAGE:
%
%       [PostSample,thetaPostSample,LogPost,AcceptedDraws,NumBurnin] = LoadDSGEPosteriorDraws(DSGEModel,CurrINI,CurrentChain)
%
% REQUIRED INPUT:  DSGEModel (structure) whose fields contain all required information
%                            about the model, such as input files, variable and sample
%                            data. Actual observations are not stored in this file.
%
%                  CurrINI (structure) with initialization information.
%
%                  CurrentChain (integer) the value of the parallel chain to load data from.
%
% REQUIRED OUTPUT: PostSample (matrix) of dimension (NumIterxNumParam) with all the non-discarded
%                            posterior draws of phi (the transformed parameters)
%
%                  thetaPostSample (matrix) of dimension (NumIterxNumParam) with all the non-discarded
%                            posterior draws of theta (the original parameters)
%
%                  LogPost (vector) of dimension (NumIterx1) with all the log posterior values for the
%                            non-discarded posterior draws. These posterior values are calculated
%                            using the "logPosteriorPhiDSGE" function.
%
%                  AcceptedDraws (vector) of dimension (NumIterx1) with the number of accepted draws.
%
%                  NumBurnin (integer) with the number of brun-in draws.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: December 12, 2006.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 28-2-2007:  The function now takes into account that the control "controls.posterior.burnin" displays the number
%               of burn-in draws rather than the percentage relative to the full posterior sample.
%
% * 10-4-2007:  The function now taken into account that the variable "InverseHessianEstimator" affects the name of
%               the PostSampleFile.
%
% * 11-4-2007:  Changed the AcceptedDraws vector to the number of accepted draws instead of the acceptance ratio.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% locate the controls structure
%
maingui = findobj('Type','figure','Tag','YADA');
controls = get(maingui,'UserData');
%
% Get the base values number of draws, etc.
%
% 1. Number of posterior draws
%
DrawsStr = get(controls.posterior.draws,'String');
NumPostDraws = str2num(StringTrim(strrep(DrawsStr(DSGEModel.PosteriorDrawsValue,:),',','')));
%
% 2. Number of posterior sample batches to save per chain
%
BatchStr = get(controls.posterior.batch,'String');
NumBatches = str2num(StringTrim(strrep(BatchStr(DSGEModel.SampleBatchValue,:),',','')));
if NumBatches>NumPostDraws;
   NumBatches = NumPostDraws;
end;
NumBatchDraws = NumPostDraws/NumBatches;
BatchIndex = (1:NumBatchDraws);
%
% 3. Number of parallel chains
%
ChainsStr = get(controls.posterior.chains,'String');
NumChains = str2num(StringTrim(ChainsStr(DSGEModel.ParallelChainsValue,:)));
%
% 4. Number of burnin draws per chain
%
BurninStr = get(controls.posterior.burnin,'String');
NumBurnin = str2num(strrep(StringTrim(BurninStr(DSGEModel.BurnInValue,:)),',',''));
if NumBurnin>=NumPostDraws;
   txt = ['The number of burn-in draws (' StringTrim(BurninStr(DSGEModel.BurnInValue,:)) ') leaves no posterior draws (' StringTrim(DrawsStr(DSGEModel.PosteriorDrawsValue,:)) ') for inference. YADA will therefore use 0 burn-in draws.'];
   About(txt,'information','YADA - Burnin Draws Problem',180,500,CurrINI);
   NumBurnin = 0;
end;
%
% Now we turn to batch no 1. since we need to determine how many parameters there are in phi and theta.
%
PostSampleFile = [DSGEModel.OutputDirectory '\rwm\RWM-' DSGEModel.NameOfModel '-' num2str(DSGEModel.InverseHessianEstimator,'%0.0f') '-' num2str(CurrentChain,'%0.0f') '.' num2str(NumChains,'%0.0f') '-' num2str(1,'%0.0f') '.' num2str(NumBatches,'%0.0f') '.' num2str(NumPostDraws,'%0.0f') '.mat'];
RWMData = load(PostSampleFile);
NumParam = size(RWMData.BatchPostSample,2);
%
% initialize the data matrices
%
PostSample = zeros(NumPostDraws,NumParam);
thetaPostSample = PostSample;
LogPost = zeros(NumPostDraws,1);
AcceptedDraws = LogPost;
%
% load the first batch of data
%
PostSample(BatchIndex,:) = RWMData.BatchPostSample;
thetaPostSample(BatchIndex,:) = RWMData.BatchthetaPostSample;
LogPost(BatchIndex,1) = RWMData.BatchLogPost;
AcceptedDraws(BatchIndex,1) = RWMData.BatchAcceptedDraws;
%
% now we loop over all draws
%
for NBatch=2:NumBatches;
   BatchIndex = BatchIndex+NumBatchDraws;
   PostSampleFile = [DSGEModel.OutputDirectory '\rwm\RWM-' DSGEModel.NameOfModel '-' num2str(DSGEModel.InverseHessianEstimator,'%0.0f') '-' num2str(CurrentChain,'%0.0f') '.' num2str(NumChains,'%0.0f') '-' num2str(NBatch,'%0.0f') '.' num2str(NumBatches,'%0.0f') '.' num2str(NumPostDraws,'%0.0f') '.mat'];
   RWMData = load(PostSampleFile);
   PostSample(BatchIndex,:) = RWMData.BatchPostSample;
   thetaPostSample(BatchIndex,:) = RWMData.BatchthetaPostSample;
   LogPost(BatchIndex,1) = RWMData.BatchLogPost;
   AcceptedDraws(BatchIndex,1) = RWMData.BatchAcceptedDraws;
end;

%
% end of LoadDSGEPosteriorDraws.m
%
