function UseDraws = KeepPosteriorDraws(DSGEModel,DrawsHandle,NumDraws)
% KeepPosteriorDraws: Determines which of the NumDraws to use.
%
% USAGE:
%
%       UseDraws = KeepPosteriorDraws(DSGEModel,DrawsHandle,NumDraws)
%
% REQUIRED INPUT:  DSGEModel (structure) whose fields contain all required information
%                            about the model, such as input files, variable and sample
%                            data. Actual observations are not stored in this file.
%
%                  DrawsHandle (handle) to the control on the YADA GUI that determines
%                            the percentage rate of the draws to use.
%
%                  NumDraws (integer) with the number of posterior draws that are available.
%
% REQUIRED OUTPUT: UseDraws (vector) with integer values providing the draws to use.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: April 3, 2007.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initializing
%
UseDraws = (1:NumDraws);
if DSGEModel.PostDrawsPercentValue==109;
   return;
end;
DrawsStr = get(DrawsHandle,'String');
DrawsPercent = str2num(strrep(StringTrim(DrawsStr(DSGEModel.PostDrawsPercentValue,:)),'%',''));
DrawsNum = ceil(DrawsPercent*NumDraws/100);
if DSGEModel.RandomizeDraws==0;
   %
   % maximum distance between draws
   %
   DrawFreq = round(NumDraws/DrawsNum);
   UseDraws = (1:DrawFreq:NumDraws);
   if length(UseDraws)+1==DrawsNum;
      UseDraws = [UseDraws NumDraws];
   elseif length(UseDraws)>DrawsNum;
      UseDraws = UseDraws(1:DrawsNum);
   end;
else;
   %
   % initialize the random number generator
   %
   if DSGEModel.RandomNumberValue==1;
      randn('state',0);
      rand('state',0);
   else;
      randn('state',sum(100*clock));
      rand('state',sum(100*clock));
   end;
   %
   % randomly selected draws
   %
   UseDraws = ceil(rand(1,DrawsNum)*NumDraws);
end;

%
% end of KeepPosteriorDraws.m
%
