function [y,Status,ErrorStr,HeaderStr] = InvertTransformData(z,Transformation,CurrVariable)
% InvertTransformData: applies the inversion transformation function to a variable.
%
% USAGE:
%
%       [y,Status,ErrorStr,HeaderStr] = InvertTransformData(z,Transformation,CurrVariable)
%
% REQUIRED iNPUT:  z (vector) with T observations.
%
%                  Transformation (structure) with field names. The field name to use is determined
%                                 by CurrVariable. This field name has subfields "invertfcn",
%                                 "invertinitial", and "invertx" that are used for the inversion of z into y.
%
%                  CurrVariable (string vector) with the name of the variable to transform. This
%                                 name should correspond to the field name in the structure Transformation
%                                 that will be used.
%
% REQUIRED OUTPUT: y (vector) with T1 observations.
%
%                  Status (boolean) that takes the value 0 if the inversion was performed without
%                                 error and 1 otherwise.
%
%                  ErrorStr (string vector) with error messages (if any).
%
%                  HeaderStr (string vector) with a suitable header message (if any).
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: October 30, 2007.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 9-11-2007:  Fixed an error when setting up the command line arguments. The field "CurrField.x" was used
%               for the loop rather than the "CurrField.invertx".
%
% * 14-11-2007: Added the input variables "FirstObs" and "LastObs" and took the new fields "invertx.data"
%               and "invertx.numinitialobs" into account.
%
% * 20-11-2007: The fields "data" and "numinitialobs" are no longer used. The inputs "FirstObs" and "LastObs"
%               have also been removed.
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
y = [];
Status = 0;
ErrorStr = '';
HeaderStr = '';
%
% get the structure with the required fields
%
CurrField = getfield(Transformation,CurrVariable);
%
% remove any NaN's at the beginning of z
%
StartPos = NaN;
for i=1:length(z);
   if isnan(z(i))==0;
      StartPos = i;
      break;
   end;
end;
if isnan(StartPos)==1;
   ErrorStr = ['The data for ' CurrVariable ' has only NaN''s. YADA is therefore unable to perform the inversion.'];
   HeaderStr = 'Bad Inversion Function';
   Status = 1;
   return;
end;
z = z(StartPos:length(z));
%
% check how many input arguments the inline function
%
NumInputs = length(symvar(CurrField.invertfcn));
%
% make sure that time dimension is given by the rows
%
if size(CurrField.invertx,2)>size(CurrField.invertx,1);
   CurrField.invertx = CurrField.invertx';
end;
if size(CurrField.invertx,2)~=NumInputs-1;
   CurrField.invertx = [];
end;
%
% deal with initial values for the variable to be inverted
%
if isempty(CurrField.invertinitial)==0;
   z = [vec(CurrField.invertinitial);z];
end;
RunArg = 'z';
%
% check if we should create the additional variables for the command line
%
data = CurrField.invertx;
for i=1:size(data,2);
   eval(['x' int2str(i) ' = RemovePreNaNs(data(:,' int2str(i) '));']);
   RunArg = [RunArg ',x' int2str(i)];
end;
%
% now we try to create the inline function
%
try;
   if isempty(data)==1;
      YADAf = inline(CurrField.invertfcn);
   else;
      YADAf = YADAInline(CurrField.invertfcn);
   end;
catch;
   ErrorStr = ['The was an error creating the inline function for ' CurrVariable '. The following message was caught: ' lasterr];
   HeaderStr = 'Bad Inversion Function';
   Status = 1;
   y = z;
   return;
end;
%
% check if the inline function executes correctly
%
ErrorStr = '';
try;
   eval(['y = YADAf(' RunArg ');']);
catch;
   ErrorStr = ['YADA caught the following error message for the inversion function for the variable ' CurrVariable ': ' lasterr];
   HeaderStr = 'Bad Inversion Function';
   Status = 1;
   y = z;
end;

%
% end of InvertTransformData.m
%
