function [UpdatePostSample,UpdateMode,UpdateNames] = GetUpdatePosteriorDraws(thetaPostSample,NumBurnin,CurrentChain,NumChains,CheckForData,DSGEModel,CurrINI)
% GetUpdatePosteriorDraws: Computes the posterior draw values for the parameters that are functions of
%                          estimated parameters.
%
% USAGE:
%
%       [UpdatePostSample,UpdateNames] = GetUpdatePosteriorDraws(thetaPostSample,NumBurnin,CurrentChain,NumChains,CheckForData,DSGEModel,CurrINI)
%
% REQUIRED INPUT:  thetaPostSample (matrix) of dimension (NumIterxNumParam) with all the non-discarded
%                                  posterior draws of theta (the original parameters)
%
%                  NumBurnin (integer) with the number of burn-in draws.
%
%                  CurrentChain (integer) with the number of the currently used MCMC draws.
%
%                  NumChains (integer) with the total number of MCMC chains that we can choose from.
%
%                  CheckForData (boolean) that takes the value 1 if the function should check if the
%                                  calculations have already been performed and stored on disk, and
%                                  0 if they should be calculated anyway.
%
%                  DSGEModel (structure) whose fields contain all required information about the model,
%                                  such as input files, variable and sample data.
%
%                  CurrINI (structure) with initialization information.
%
% REQUIRED OUTPUT: UpdatePostSample (matrix) of dimension (NumIter-NumBurninxNumAdditional) with the post
%                                  burnin sample values of the parameters that are functions of the
%                                  estimated parameters.
%
%                  UpdateMode (vector) of dimension (NumAdditional x 1) with the posterior mode estimates
%                                  of the parameters that are functions of the estimated parameters.
%
%                  UpdateNames (string matrix) of dimension (NumAdditional x N1) with the names of the
%                                  additional parameters.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: February 25, 2009.
%                        This version: March 9, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 26-02-2009: Added the input variable "CheckForData".
%
% * 09-03-2009: Added quotation marks around paths for the dos copy command.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
UpdatePostSample = [];
UpdateMode = [];
UpdateNames = '';
%
[NumIter,NumParam] = size(thetaPostSample);
%
% check if we can load the posterior mode data
%
ModeFile = [DSGEModel.OutputDirectory '\mode\PosteriorMode-' DSGEModel.NameOfModel '.mat'];
try;
   ModeData = load(ModeFile);
catch;
   txt = ['YADA can''t provide posterior densities of parameters that are a function of the estimated parameters. There was a problem trying to load the file with posterior mode data ("' GetFilename(ModeFile) '") in the directory "' GetPath(ModeFile) '". YADA has received the following error message: ' lasterr '.'];
   About(txt,'information','Posterior Densities - Additional Parameters',300,500,CurrINI);
   drawnow;
   pause(0.02);
   return;
end;
%
% copy update parameter file to tmp folder
%
[stat,msg] = dos(['copy /Y "' DSGEModel.UpdateParameterFile '" "' pwd '\tmp"']);
%
% get the names of the parameters to update
%
UpdateNames = ModeData.ParameterNames.additional;
thetaPositions = ModeData.thetaPositions;
ModelParameters = ModeData.ModelParameters;
%
NumAdditional = size(UpdateNames,1);
UpdatePostSample = zeros(NumIter-NumBurnin,NumAdditional);
%
% compute the posterior mode of the updated parameters
%
UpdateMode = zeros(NumAdditional,1);
%
% 1. set the ModelParameters structure to posterior mode values
%
ModelParameters = ThetaToModelParameters(ModelParameters,ModeData.thetaMode,thetaPositions);
%
% 2. update the additional parameters at posterior mode
%
eval(['ModelParameters = ' GetName(DSGEModel.UpdateParameterFile) '(ModelParameters);']);
for j=1:NumAdditional;
   UpdateMode(j) = getfield(ModelParameters,StringTrim(UpdateNames(j,:)));
end;
%
% check if previously calculated values are stored on disk
%
AddFile = [DSGEModel.OutputDirectory '\rwm\RWM-Additional-' DSGEModel.NameOfModel '-' num2str(DSGEModel.InverseHessianEstimator,'%0.0f') '-' num2str(CurrentChain,'%0.0f') '.' num2str(NumChains,'%0.0f') '-' num2str(NumBurnin,'%0.0f') '.' num2str(NumIter,'%0.0f') '.mat'];
if (CheckForData==1)&(FileExist(AddFile)==1);
   AddData = load(AddFile);
   %
   % check if we have the same posterior mode and the same parameter names
   %
   if (isfield(AddData,'UpdateNames')==1)&(isfield(AddData,'UpdateMode')==1)&(isfield(AddData,'UpdatePostSample')==1)&(isequal(UpdateMode,AddData.UpdateMode)==1)&(isequal(UpdateNames,AddData.UpdateNames)==1);
      txt = ['YADA has located previously computed values for the additional parameters using ' int2str(NumIter-NumBurnin) ' post burn-in draws of the estimated parameters. Would you like to use these values?'];
      answer = Query(txt,'information',200,'Posterior Densities - Additional Parameters',500,'yes',CurrINI);
      drawnow;
      pause(0.02);
      if strcmp(lower(answer),'yes')==1;
         UpdatePostSample = AddData.UpdatePostSample;
         return;
      end;
   end;
end;
%
if (CheckForData==1)&(NumIter-NumBurnin>=10000);
   sStr = '';
   RStr = 'is';
   if NumAdditional>1;
      sStr = 's';
      RStr = 'are';
   end;
   txt = ['Please wait while YADA prepares the ' int2str(NumIter-NumBurnin) ' post burn-in sample posterior draws of the ' int2str(NumAdditional) ' parameter' sStr ' that ' RStr ' a function of the estimated parameters. Computations started at: ' StringTrim(datestr(now,14)) '.'];
   WaitHandle = WaitDLG(txt,'information','Posterior Densities - Additional Parameters',500,200,CurrINI,0);
   WaitControls = get(WaitHandle,'UserData');
end;
%
for i=NumBurnin+1:NumIter;
   %
   % make sure the ModelParameters structure is properly updated
   %
   ModelParameters = ThetaToModelParameters(ModelParameters,thetaPostSample(i,:)',thetaPositions);
   %
   % now run the update parameter function
   %
   eval(['ModelParameters = ' GetName(DSGEModel.UpdateParameterFile) '(ModelParameters);']);
   %
   % now we update the vector of additional parameters
   %
   for j=1:NumAdditional;
      UpdatePostSample(i-NumBurnin,j) = getfield(ModelParameters,StringTrim(UpdateNames(j,:)));
   end;
   drawnow;
end;
%
% close the wait dialog
%
if (CheckForData==1)&(NumIter-NumBurnin>=10000);
   set(WaitControls.text,'UserData','done');
   delete(WaitHandle);
   drawnow;
   pause(0.02);
end;
%
% save the data to file
%
save(AddFile,'UpdatePostSample','UpdateNames','UpdateMode');

%
% end of GetUpdatePosteriorDraws.m
%
