function [FirstPeriod,LastPeriod,SubSampleStrMat] = CreateSubSample(DSGEModel)
% CreateSubSample: Sets up a sub-sample relatrive to the full sample.
%
% USAGE:
%
%       [FirstPeriod,LastPeriod,SubSampleStrMat] = CreateSubSample(DSGEModel)
%
% REQUIRED INPUT:  DSGEModel (structure) with fields containing all the information about a DSGE Model for YADA.
%
% REQUIRED OUTPUT: FirstPeriod (positive integer) indicating the position in the time dimension of a matrix with data
%                              where the sub-sample begins.
%
%                  LastPeriod (positive integer) indicating the position in the time dimension of a matrix with data
%                              where the sub-sample ends.
%
% OPTIONAL OUTPUT: SubSampleStrMat (string matrix) with rows specifying the year and the period for the sub-sample
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: August 15, 2006.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 9-2-2007:   Removed function test data.
%
% * 13-11-2007: Updated the documentation.
%
% * 2-1-2008:   Made sure that "SubSampleStrMat" is non-empty when FirstPeriod is zero.
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% collect information
%
BeginYear = str2num(DSGEModel.BeginYear);
BeginPeriod = str2num(DSGEModel.BeginPeriod);
EndYear = str2num(DSGEModel.EndYear);
EndPeriod = str2num(DSGEModel.EndPeriod);
if strcmp(DSGEModel.DataFrequency,'q')==1;
   SampleFreq = 4;
elseif strcmp(DSGEModel.DataFrequency,'m')==1;
   SampleFreq = 12;
elseif strcmp(DSGEModel.DataFrequency,'a')==1;
   SampleFreq = 1;
end;
%
% create a full sample string matrix
%
FullSampleStrMat = '';
for i=BeginYear:EndYear;
   for j=1:SampleFreq;
      if i==BeginYear;
         if j>=BeginPeriod;
            FullSampleStrMat = strvcat(FullSampleStrMat,[num2str(i,'%0.0f') ':' num2str(j,'%0.0f')]);
         end;
      elseif i==EndYear;
         if j<=EndPeriod;
            FullSampleStrMat = strvcat(FullSampleStrMat,[num2str(i,'%0.0f') ':' num2str(j,'%0.0f')]);
         end;
      else;
         FullSampleStrMat = strvcat(FullSampleStrMat,[num2str(i,'%0.0f') ':' num2str(j,'%0.0f')]);
      end;
   end;
end;
%
% Locate the positions of the sub-sample start point and end point.
%
BeginStr = [DSGEModel.SubBeginYear ':' DSGEModel.SubBeginPeriod];
EndStr = [DSGEModel.SubEndYear ':' DSGEModel.SubEndPeriod];
FirstPeriod = LocateStr(FullSampleStrMat,BeginStr);
LastPeriod = LocateStr(FullSampleStrMat,EndStr);
if nargout==3;
   if FirstPeriod>0;
      SubSampleStrMat = FullSampleStrMat(FirstPeriod:LastPeriod,:);
   else;
      SubSampleStrMat = ' ';
   end;
end;

%
% end of CreateSubSample.m
%
