function [PredStartYear,PredStartPeriod,PredEndYear,PredEndPeriod,PredDateVector,SampleStrMat] = CreatePredictionSample(OrigEndYear,OrigEndPeriod,Frequency,NumPredObs)
% CreatePredictionSample: Sets up the start period and end period of a prediction sample.
%
% USAGE:
%
%       [PredStartYear,PredStartPeriod,PredEndYear,PredEndPeriod,PredDateVector,SampleStrMat] = CreatePredictionSample(OrigEndYear,OrigEndPeriod,Frequency,SampleLength)
%
% REQUIRED INPUT:  OrigEndYear (string vector) with the end year of the original sample.
%
%                  OrigEndPeriod (string vector) with the end period of the original sample.
%
%                  Frequency (string vector) with the frequency of the data ('q', 'm', or 'a')
%
%                  NumPredObs (integer) with the number of observations in the prediction sample.
%
% REQUIRED OUTPUT: PredStartYear (string vector) with the start year of the prediction sample.
%
%                  PredStartPeriod (string vector) with the start period of the prediction sample.
%
%                  PredEndYear (string vector) with the end year of the prediction sample.
%
%                  PredEndPeriod (string vector) with the end period of the prediction sample.
%
% OPTIONAL OUTPUT: PredDateVector (1xT) vector with date numbers. Notice that center points
%                                 are used . Hence, 1980Q1 = 1980+(1/8); 1980m1 = 1980+(1/24).
%
%                  SampleStrMat (string matrix) of dimension TxT1 with date strings for the
%                                 prediction sample.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: February 21, 2007.
%                        This version: March 3, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 13-11-2007: Updated the documentation.
%
% * 23-05-2008: Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
% * 03-03-2009: Added the optional output variable "SampleStrMat".
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% turn input string into integers
%
EndYear = str2num(OrigEndYear);
EndPeriod = str2num(OrigEndPeriod);
if strcmp(Frequency,'q')==1;
   SampleFreq = 4;
elseif strcmp(Frequency,'m')==1;
   SampleFreq = 12;
else;
   SampleFreq = 1;
end;
NumYears = ceil(NumPredObs/SampleFreq);
%
% create a full sample string matrix
%
FullSampleStrMat = '';
for y=EndYear:EndYear+NumYears+1;
   for p=1:SampleFreq;
      FullSampleStrMat = strvcat(FullSampleStrMat,[num2str(y,'%0.0f') ':' num2str(p,'%0.0f')]);
   end;
end;
%
% locate the beginning of the prediction sample
%
OrigEnd = [OrigEndYear ':' OrigEndPeriod];
for i=1:size(FullSampleStrMat,1);
   if strcmp(StringTrim(FullSampleStrMat(i,:)),OrigEnd)==1;
      StartPos = i+1;
   end;
end;
EndPos = StartPos+NumPredObs-1;
PredStart = StringTrim(FullSampleStrMat(StartPos,:));
PredEnd = StringTrim(FullSampleStrMat(EndPos,:));
%
% determine the predtiction sample start year and period
%
FindStart = findstr(PredStart,':');
PredStartYear = PredStart(1,1:FindStart-1);
PredStartPeriod = PredStart(1,FindStart+1:length(PredStart));
%
% determine the prediction sample end year and period
%
FindEnd = findstr(PredEnd,':');
PredEndYear = PredEnd(1,1:FindEnd-1);
PredEndPeriod = PredEnd(1,FindEnd+1:length(PredEnd));
%
% if ask to, create the PredDateVector
%
if nargout>=5;
   PredDateVector = MakeDateVector(PredStartYear,PredStartPeriod,PredEndYear,PredEndPeriod,Frequency);
end;
if nargout==6;
   SampleStrMat = FullSampleStrMat(StartPos:EndPos,:);
end;

%
% end of CreatePredictionSample.m
%
