function [SubBeginYearValue,SubBeginPeriodValue,SubEndYearValue,SubEndPeriodValue,YearStrMatrix,PeriodStrMatrix] = CreateEstimationSampleValues(DSGEModel)
% CreateEstimationSampleValues: Creates the values for the estimation sample as integers
%                               reflecting their positions in string matrices.
%
% USAGE:
%
%       [SubBeginYearValue,SubBeginPeriodValue,SubEndYearValue,SubEndPeriodValue,YearStrMatrix,PeriodStrMatrix] = CreateEstimationSampleValues(DSGEModel)
%
% REQUIRED INPUT:  DSGEModel (structure)  whose fields contain all required information
%                            about the model, such as input files, variable and sample
%                            data. Actual observations are not stored in this file.
%
% REQUIRED OUTPUT: SubBeginYearValue (integer) reflecting the estimation start year as the row
%                            value in a string matrix with all the possible starting years.
%
%                  SubBeginPeriodValue (integer) reflecting the estimation start period as the
%                            row value in a string matrix with all the possible starting
%                            periods.
%
%                  SubEndYearValue (integer) reflecting the estimation end year as the row
%                            value in a string matrix with all the possible ending years.
%
%                  SubEndPeriodValue (integer) reflecting the estimation end period as the
%                            row value in a string matrix with all the possible ending
%                            periods.
%
%                  YearStrMatrix (string matrix) with all the possible years to consider for
%                            estimation start and end years.
%
%                  PeriodStrMatrix (string matrix) with all the possible periods to consider
%                            for estimation start and end periods.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: August 31, 2007.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 13-11-2007: Updated the documentation.
%
% * 2-1-2008:   Fixed a bug where no sample values lead to bad output.
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
SubBeginYearValue = 1;
SubBeginPeriodValue = str2num(DSGEModel.SubBeginPeriod);
SubEndYearValue = 1;
SubEndPeriodValue = str2num(DSGEModel.SubEndPeriod);
%
BeginYear = str2num(DSGEModel.BeginYear);
EndYear = str2num(DSGEModel.EndYear);
for i=BeginYear:EndYear;
   if i==str2num(DSGEModel.SubBeginYear);
      SubBeginYearValue = i-BeginYear+1;
   end;
   if i==str2num(DSGEModel.SubEndYear);
      SubEndYearValue = i-BeginYear+1;
      break;
   end;
end;
%
% provide the string matrices as additional output
%
YearStrMatrix = '';
PeriodStrMatrix = '1';
if strcmp(lower(DSGEModel.DataFrequency),'q')==1;
   PeriodStrMatrix = 'Quarter 1|Quarter 2|Quarter 3|Quarter 4';
elseif strcmp(lower(DSGEModel.DataFrequency),'m')==1;
   PeriodStrMatrix = 'January|February|March|April|May|June|July|August|September|October|November|December';
end;
for i=BeginYear:EndYear;
   YearStrMatrix = strvcat(YearStrMatrix,num2str(i,'%0.0f'));
end;
if isempty(YearStrMatrix)==1;
   YearStrMatrix = ' ';
end;

%
% end of CreateEstimationSampleValues.m
%
