function ParameterBounds = CreateBounds(thetaDist,LowerBound,UniformBounds)
% CreateBounds: Creates a matrix with lower and upper bounds for the parameters along with an indicator if
%               the lower and/or upper bound is an equality bound or not.
%
% USAGE:
%
%       ParameterBounds = CreateBounds(thetaDist,LowerBound,UniformBounds)
%
% REQUIRED INPUT:  thetaDist (vector)  with integers 1 if gamma, 2 if beta, 3 if normal, 4 if invgamma, 5 if
%                            truncnormal, and 6 if uniform.
%
%                  LowerBound (vector) with lower bound values for the parameter vector. Used by the
%                            left truncated normal, the gamma, and the inverse gamma distribution.
%
%                  UniformBounds (matrix) with the same number of rows as parameters and 2 columns. The
%                            columns are 0 1 except for the uniform and the beta distribution, where
%                            they represent the upper and lower bounds.
%
% REQUIRED OUTPUT: ParameterBounds (matrix) with lower and upper bounds for the parameters in the first
%                            and in the second column, respectively, while the third column provides
%                            and indicator it 0, 1, or 2 bounds are binding. If 1, then only the lower
%                            bound is binding.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: October 26, 2007.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 30-10-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 21-7-2008:  Added support for non-zero lower bound for the gamma and inverse gamma priors.
%
% * 23-7-2008:  Added support for beta prior with lower bound different from 0 and upper bound
%               different from 1.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
ParameterBounds = zeros(length(thetaDist),3);
for i=1:length(thetaDist);
   if thetaDist(i)==1;
      %
      % gamma prior
      %
      ParameterBounds(i,1) = LowerBound(i);
      ParameterBounds(i,2) = Inf;
   elseif thetaDist(i)==2;
      %
      % beta prior
      %
      ParameterBounds(i,1) = UniformBounds(i,1);
      ParameterBounds(i,2) = UniformBounds(i,2);
      
   elseif thetaDist(i)==3;
      %
      % normal prior
      %
      ParameterBounds(i,1) = -Inf;
      ParameterBounds(i,2) = Inf;
   elseif thetaDist(i)==4;
      %
      % inverse gamma prior
      %
      ParameterBounds(i,1) = LowerBound(i);
      ParameterBounds(i,2) = Inf;
   elseif thetaDist(i)==5;
      %
      % left truncated normal prior
      %
      ParameterBounds(i,1) = LowerBound(i);
      ParameterBounds(i,2) = Inf;
      ParameterBounds(i,3) = 1;
   elseif thetaDist(i)==6;
      %
      % uniform prior
      %
      ParameterBounds(i,1) = UniformBounds(i,1);
      ParameterBounds(i,2) = UniformBounds(i,2);
      ParameterBounds(i,3) = 2;
   end;
end;

%
% end of CreateBounds.m
%
