function status = CheckObservedDataFile(DataStructure,DataFileHandle,DataConstructionFile,filename,CurrINI)
% CheckObservedDataFile: Checks if the matlab m-file with data on the observed variables, the exogenous variables,
%                        their names, and the claimed sample are properly setup.
%
% USAGE:
%
%       status = CheckObservedDataFile(DataStructure,DataFileHandle,DataConstructionFile,filename,CurrINI)
%
%
% REQUIRED INPUTS: DataStructure (structure) with fields Y, X, Y.names, Y.data, X.names, X.data, sample
%                                and frequency. This information should be provided by the data construction
%                                file
%
%                  DataFileHandle (handle), a valid handle to the uicontrol where the data construction
%                                file is stored in a string. Maybe empty!
%
%                  DataConstructionFile (string) containing the previous data construction file
%
%                  filename (string) containing the file name of the data construction file that is being tested
%
%                  CurrINI (structure) with initialization information
%
% REQUIRED OUTPUT: status (boolean) taking on 1 if the DataStructure is valid, and 0 otherwise.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: August 1, 2006.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 2-8-2006:   Added the possibility that DataFileHandle is empty.
%
% * 8-12-2006:  The field Y.levels is now also checked.
%
% * 22-2-2007:  Made sure that the field Y.levels only has 0 and 1's.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

status = 0;
if isfield(DataStructure,'Y')==1;
   if isfield(DataStructure.Y,'data')==1;
      if isfield(DataStructure.Y,'names')==1;
         if isfield(DataStructure,'X')==1;
            if isfield(DataStructure.X,'data')==1;
               if isfield(DataStructure.X,'names')==1;
                  if isfield(DataStructure,'sample')==1;
                     if isfield(DataStructure,'frequency')==1;
                        %
                        % All required information is provided by the data constructions file.
                        % Now we check if it is valid
                        %
                        if isempty(DataStructure.Y.data)==0;
                           [n,T] = size(DataStructure.Y.data);
                           if n>T;
                              DataStructure.Y.data = DataStructure.Y.data';
                              [n,T] = size(DataStructure.Y.data);
                           end;
                           num_names = length(DataStructure.Y.names);
                           if n==num_names;
                              [k,T1] = size(DataStructure.X.data);
                              if k>T1;
                                 DataStructure.X.data = DataStructure.X.data';
                                 [k,T1] = size(DataStructure.X.data);
                              end;
                              %
                              % if X.data is non-empty, make sure the number of obs matches
                              %
                              if k>0;
                                 match_T = (T==T1);
                              else;
                                 match_T = 1;
                              end;
                              if match_T==1;
                                 %
                                 % check if the number of exogenous variables matches the number of names
                                 %
                                 if k>0;
                                    match_k = (k==length(DataStructure.X.names));
                                 else;
                                    match_k = 1;
                                 end;
                                 if match_k==1;
                                    %
                                    % check if the sample and frequency are correct
                                    %
                                    if (isnumeric(DataStructure.sample))&(length(DataStructure.sample)==4)&(DataStructure.sample(1)<=DataStructure.sample(3));
                                       CurrSample = floor(DataStructure.sample);
                                       %
                                       % check the frequency
                                       %
                                       CurrFrequency = DataStructure.frequency;
                                       if (ischar(CurrFrequency)==1)&(length(CurrFrequency)>0);
                                          check_freq = CurrFrequency(1,1);
                                          if (strcmp(check_freq,'a')==1)|(strcmp(check_freq,'q')==1)|(strcmp(check_freq,'m')==1);
                                             good_freq = 1;
                                          else;
                                             good_freq = 0;
                                          end;
                                       else;
                                          good_freq = 0;
                                       end;
                                       if good_freq==1;
                                          %
                                          % now we just check that the sample length is correct
                                          %
                                          if strcmp(check_freq,'q')==1;
                                             TestSample = (CurrSample(1)+(CurrSample(2)/4):(1/4):CurrSample(3)+(CurrSample(4)/4));
                                             match_sample = (T==length(TestSample));
                                          elseif strcmp(check_freq,'m')==1;
                                             TestSample = (CurrSample(1)+(CurrSample(2)/12):(1/12):CurrSample(3)+(CurrSample(4)/12));
                                             match_sample = (T==length(TestSample));
                                          else;
                                             TestSample = (CurrSample(1):1:CurrSample(3));
                                             match_sample = (T==length(TestSample));
                                          end;
                                          if match_sample==1;
                                             %
                                             % now we check if the levels part is there and if valid
                                             %
                                             if isfield(DataStructure.Y,'levels')==1;
                                                %
                                                % check if the data is numeric
                                                %
                                                if (isempty(DataStructure.Y.levels)==0)&(isnumeric(DataStructure.Y.levels)==1)&(length(DataStructure.Y.levels)==min(size(DataStructure.Y.data)))&(IsZeroOne(DataStructure.Y.levels)==1);
                                                   %
                                                   % the data construction file checked out OK
                                                   %
                                                   status = 1;
                                                else;
                                                   txt = 'The levels field is badly set up. It should contain a vector of the same length as the number of observed variables, with elements that are 0 if the variable is in first differences and 1 if in levels. YADA will now assume that all variables are in levels.';
                                                   About(txt,'information','Data Construction File - Missing Field',200,500,CurrINI);
                                                   status = 2;
                                                end;
                                             else;
                                                txt = 'The levels field is missing from the Y field. The levels field should contain a vector of the same length as the number of observed variables, with elements that are 0 if the variable is in first differences and 1 if in levels. YADA will now assume that all variables are in levels.';
                                                About(txt,'information','Data Construction File - Missing Field',200,500,CurrINI);
                                                status = 2;
                                             end;
                                          else;
                                             %
                                             % bad sample specification
                                             %
                                             if isempty(DataFileHandle)==0;
                                                set(DataFileHandle,'String',DataConstructionFile);
                                             end;
                                             stat = dos(['del /Q tmp\' filename]);
                                             txt = 'The sample and frequency information regarding sample length doesn''t match the number of observations for the observed variables.';
                                             About(txt,'information','Invalid Data Construction File',120,500,CurrINI);
                                          end;
                                       else;
                                          %
                                          % Bad frequency
                                          %
                                          if isempty(DataFileHandle)==0;
                                             set(DataFileHandle,'String',DataConstructionFile);
                                          end;
                                          stat = dos(['del /Q tmp\' filename]);
                                          txt = 'The frequency field isn''t set up coprrectly.';
                                          About(txt,'information','Invalid Data Construction File',120,500,CurrINI);
                                       end;
                                    else;
                                       if isempty(DataFileHandle)==0;
                                          set(DataFileHandle,'String',DataConstructionFile);
                                       end;
                                       stat = dos(['del /Q tmp\' filename]);
                                       txt = 'The sample field is not setup correctly. It must be a 1x4 vector with integers, where the first entry is not greater than the third.';
                                       About(txt,'information','Invalid Data Construction File',120,500,CurrINI);
                                    end;
                                 else;
                                    if isempty(DataFileHandle)==0;
                                       set(DataFileHandle,'String',DataConstructionFile);
                                    end;
                                    stat = dos(['del /Q tmp\' filename]);
                                    txt = ['The number of exogenous variables (' num2str(k) ') doesn''t match the number of assigned names (' num2str(length(DataStructure.X.names)) ') for these variables.'];
                                    About(txt,'information','Invalid Data Construction File',120,500,CurrINI);
                                 end;
                              else;
                                 %
                                 % the number of observations don't match
                                 %
                                 if isempty(DataFileHandle)==0;
                                    set(DataFileHandle,'String',DataConstructionFile);
                                 end;
                                 stat = dos(['del /Q tmp\' filename]);
                                 txt = ['The number of observations for the observed variables (' num2str(T) ') doesn''t match the number of observations for the exogenous variables (' num2str(T1) ').'];
                                 About(txt,'information','Invalid Data Construction File',120,500,CurrINI);
                              end;
                           else;
                              %
                              % the number of observed variables doesn't match the number of names for these variables
                              %
                              if isempty(DataFileHandle)==0;
                                 set(DataFileHandle,'String',DataConstructionFile);
                              end;
                              stat = dos(['del /Q tmp\' filename]);
                              txt = ['The number of observed variables (' num2str(n) ') doesn''t match the number of assigned names (' num2str(num_names) ') for these variables.'];
                              About(txt,'information','Invalid Data Construction File',120,500,CurrINI);
                           end;
                        else;
                           %
                           % No data for observed variables
                           %
                           if isempty(DataFileHandle)==0;
                              set(DataFileHandle,'String',DataConstructionFile);
                           end;
                           stat = dos(['del /Q tmp\' filename]);
                           txt = ['The file "' filename '" doesn''t include any data on the observed variables.'];
                           About(txt,'information','Invalid Data Construction File',120,500,CurrINI);
                        end;
                     else;
                        %
                        % The field frequency is missing
                        %
                        if isempty(DataFileHandle)==0;
                           set(DataFileHandle,'String',DataConstructionFile);
                        end;
                        stat = dos(['del /Q tmp\' filename]);
                        txt = ['The file "' filename '" doesn''t provide the required field "frequency". Please notice that the data construction file must have the following fields: Y.data (data for observed variables); Y.names (names for observed variables); X.data (data for exogenous variables); X.names (names for exogenous variables); sample (1x4 vector with sample date indices); frequency (annual, quarterly or monthly).'];
                        About(txt,'information','Invalid Data Construction File',200,500,CurrINI);
                     end;
                  else;
                     %
                     % The field sample is missing
                     %
                     if isempty(DataFileHandle)==0;
                        set(DataFileHandle,'String',DataConstructionFile);
                     end;
                     stat = dos(['del /Q tmp\' filename]);
                     txt = ['The file "' filename '" doesn''t provide the required field "sample". Please notice that the data construction file must have the following fields: Y.data (data for observed variables); Y.names (names for observed variables); X.data (data for exogenous variables); X.names (names for exogenous variables); sample (1x4 vector with sample date indices); frequency (annual, quarterly or monthly).'];
                     About(txt,'information','Invalid Data Construction File',200,500,CurrINI);
                  end;
               else;
                  %
                  % The field X.names is missing
                  %
                  if isempty(DataFileHandle)==0;
                     set(DataFileHandle,'String',DataConstructionFile);
                  end;
                  stat = dos(['del /Q tmp\' filename]);
                  txt = ['The file "' filename '" doesn''t provide the required field "X.names". Please notice that the data construction file must have the following fields: Y.data (data for observed variables); Y.names (names for observed variables); X.data (data for exogenous variables); X.names (names for exogenous variables); sample (1x4 vector with sample date indices); frequency (annual, quarterly or monthly).'];
                  About(txt,'information','Invalid Data Construction File',200,500,CurrINI);
               end;
            else;
               %
               % The field X.data is missing
               %
               if isempty(DataFileHandle)==0;
                  set(DataFileHandle,'String',DataConstructionFile);
               end;
               stat = dos(['del /Q tmp\' filename]);
               txt = ['The file "' filename '" doesn''t provide the required field "X.data". Please notice that the data construction file must have the following fields: Y.data (data for observed variables); Y.names (names for observed variables); X.data (data for exogenous variables); X.names (names for exogenous variables); sample (1x4 vector with sample date indices); frequency (annual, quarterly or monthly).'];
               About(txt,'information','Invalid Data Construction File',200,500,CurrINI);
            end;
         else;
            %
            % The field X is missing
            %
            if isempty(DataFileHandle)==0;
               set(DataFileHandle,'String',DataConstructionFile);
            end;
            stat = dos(['del /Q tmp\' filename]);
            txt = ['The file "' filename '" doesn''t provide the required field "X". Please notice that the data construction file must have the following fields: Y.data (data for observed variables); Y.names (names for observed variables); X.data (data for exogenous variables); X.names (names for exogenous variables); sample (1x4 vector with sample date indices); frequency (annual, quarterly or monthly).'];
            About(txt,'information','Invalid Data Construction File',200,500,CurrINI);
         end;
      else;
         %
         % The field Y.names doesn't exist
         %
         if isempty(DataFileHandle)==0;
            set(DataFileHandle,'String',DataConstructionFile);
         end;
         stat = dos(['del /Q tmp\' filename]);
         txt = ['The file "' filename '" doesn''t provide the required field "Y.names". Please notice that the data construction file must have the following fields: Y.data (data for observed variables); Y.names (names for observed variables); X.data (data for exogenous variables); X.names (names for exogenous variables); sample (1x4 vector with sample date indices); frequency (annual, quarterly or monthly).'];
         About(txt,'information','Invalid Data Construction File',200,500,CurrINI);
      end;
   else;
      %
      % The field Y.data doesn't exist
      %
      if isempty(DataFileHandle)==0;
         set(DataFileHandle,'String',DataConstructionFile);
      end;
      stat = dos(['del /Q tmp\' filename]);
      txt = ['The file "' filename '" doesn''t provide the required field "Y.data". Please notice that the data construction file must have the following fields: Y.data (data for observed variables); Y.names (names for observed variables); X.data (data for exogenous variables); X.names (names for exogenous variables); sample (1x4 vector with sample date indices); frequency (annual, quarterly or monthly).'];
      About(txt,'information','Invalid Data Construction File',200,500,CurrINI);
   end;
else;
   %
   % The field Y doesn't exist
   %
   if isempty(DataFileHandle)==0;
      set(DataFileHandle,'String',DataConstructionFile);
   end;
   stat = dos(['del /Q tmp\' filename]);
   txt = ['The file "' filename '" doesn''t provide the required field "Y". Please notice that the data construction file must have the following fields: Y.data (data for observed variables); Y.names (names for observed variables); X.data (data for exogenous variables); X.names (names for exogenous variables); sample (1x4 vector with sample date indices); frequency (annual, quarterly or monthly).'];
   About(txt,'information','Invalid Data Construction File',200,500,CurrINI);
end;

%
% end of CheckObservedDataFile.m
%
