function [status,NumPeriods] = CheckConditioningData(DSGEModel)
% CheckConditioningData: Checks if the conditioning data and the actual data make it possible
%                        to calculate conditional forecasts of the observed variables.
%
% USAGE:
%
%       [status,NumPeriods] = CheckConditioningData(DSGEModel)
%
% REQUIRED INPUT:  DSGEModel (structure) whose fields contain all required information
%                            about the model, such as input files, variable and sample
%                            data. Actual observations are not stored in this file.
%
% REQUIRED OUTPUT: status (boolean) that is 1 if conditional forecasts can be computed.
%                            and 0 otherwise.
%
% OPTIONAL OUTPUT: NumPeriods (integer) with the number of periods over which conditional
%                            forecasts can be made.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: March 8, 2007.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
status = 0;
if nargout==2;
   NumPeriods = 0;
end;
if (isempty(DSGEModel.Z)==1)|(isempty(DSGEModel.ZVariableNames)==1)|(isempty(DSGEModel.K1)==1)|(isempty(DSGEModel.K2)==1)|(isempty(DSGEModel.U)==1);
   return;
end;
%
% check sample size
%
Tz = size(DSGEModel.Z,2);
Tu = size(DSGEModel.U,2);
Tx = size(DSGEModel.PredictedX,2);
%
[FirstPeriod,LastPeriod] = CreateSubSample(DSGEModel);
if (Tx+(DSGEModel.T-LastPeriod)==0)|(Tz<=LastPeriod)|(Tu<LastPeriod);
   return;
end;
status = 1;
if nargout==2;
   NumPeriods = min([Tx+(DSGEModel.T-LastPeriod) Tz-LastPeriod]);
end;

%
% end of CheckConditioningData.m
%
