function Transformation = ChangeTransformInitialValue(Transformation,VariableNames,InitialValues,TimePositions,SwapFcn)
% ChangeTransformationInitialValue: Changes the initial values for the variable dependent transformation functions.
%
% USAGE:
%
%       Transformation = ChangeTransformInitialValue(Transformation,VariableNames,InitialValues,TimePositions,SwapFcn)
%
% REQUIRED INPUT:  Transformation (structure) with field names. The field name to use is determined
%                                 by CurrVariable. This field name has subfields "fcn", "annualizefcn",
%                                 "initial", "x", "invertfcn", "invertinitial", and "invertx".
%
%                  VariableNames (string matrix) of dimension kxK with the names of the variables.
%
%                  InitialValues (matrix) of dimension kxT with the initial values for the individual
%                                 variables.
%
%                  TimePositions (vector) of dimension 1x2 with [T0 T1] being two time periods with T0<=T1.
%                                 These entries are used to adjust the sample of any exogenous variables
%                                 called by the transformation function.
%
%                  SwapFcn (boolean) that indicates if "fcn" should be changed to "partial".
%
% REQUIRED OUTPUT: Transformation (structure) with field names. The field name to use is determined
%                                 by CurrVariable. This field name has subfields "fcn", "annualizefcn",
%                                 "initial", "x", "invertfcn", "invertinitial", and "invertx".
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: July 30, 2008.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 31-7-2008:  Added the input variable "SwapFcn".
%
% * 1-8-2008:   Added some extra tests for the x-matrix size.
%
% * 8-8-2008:   Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

[k,T] = size(InitialValues);
T0 = TimePositions(1);
T1 = TimePositions(2);
if isempty(InitialValues)==1;
   k = size(VariableNames,1);
   T = 0;
end;
for i=1:k;
   CurrVariable = StringTrim(VariableNames(i,:));
   CurrField = getfield(Transformation,CurrVariable);
   %
   % check if we should change the x field.
   %
   if isempty(getfield(CurrField,'x'))==0;
      x = getfield(CurrField,'x');
      if size(x,2)>size(x,1);
         x = x';
      end;
      if (size(x,1)>=T1)&(T0>T);
         CurrField = setfield(CurrField,'x',x(T0-T:T1,:));
      else;
         CurrField = setfield(CurrField,'x',[]);
      end;
   end;
   %
   % change the initial field
   %
   if isempty(InitialValues)==0;
      CurrField = setfield(CurrField,'initial',InitialValues(i,:)');
   end;
   %
   % check if "fcn" should be replaced with "partial"
   %
   if SwapFcn==1;
      CurrField = setfield(CurrField,'fcn',getfield(CurrField,'partial'));
   end;
   %
   Transformation = setfield(Transformation,CurrVariable,CurrField);
end;

%
% end of ChangeTransformInitialValue.m
%
