function NumPeriods = CalcMaxForecastHorizon(MaxForecastHorizon,DataFrequency,EndYear,EndPeriod,SubEndYear,SubEndPeriod)
% CalcMaxForecastHorizon: Computes the maximum forecast horizon based on what is possible and what is
%                         desired.
%
% USAGE:
%
%       NumPeriods = CalcMaxForecastHorizon(MaxForecastHorizon,DataFrequency,EndYear,EndPeriod,SubEndYear,SubEndPeriod)
%
% REQUIRED INPUT:  MaxForecastHorizon (integer), with the maximum desired forecast horizon in years.
%
%                  DataFrequency (string), holding the data frequency 'q', 'm', 'y'.
%
%                  EndYear (string), with the end year of the data, e.g., '1999'.
%
%                  EndPeriod (string), with the end period of the data, e.g., '3'.
%
%                  SubEndYear (string), with the end year of the data used for estimation, e.g., '1996'.
%
%                  SubEndPeriod (string), with the end period of the data used for estimation, e.g., '4'.
%
% REQUIRED OUTPUT: NumPeriods (integer), with the minimum of what is desired and what is possible
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: December 21, 2006.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 8-1-2007:   Made sure that at least 1 period out-of-sample is available for forecasting.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-1-2008:  The variable "MFH_Desired" is no longer used and replaced with the input variable
%               "MaxForecastHorizon".
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

if strcmp(lower(DataFrequency(1,1)),'q')==1;
   EndPos = (4*str2num(StringTrim(EndYear)))+(str2num(StringTrim(EndPeriod))-1);
   SubEndPos = (4*str2num(StringTrim(SubEndYear)))+(str2num(StringTrim(SubEndPeriod))-1);
elseif strcmp(lower(DataFrequency(1,1)),'m')==1;
   EndPos = (12*str2num(StringTrim(EndYear)))+(str2num(StringTrim(EndPeriod))-1);
   SubEndPos = (12*str2num(StringTrim(SubEndYear)))+(str2num(StringTrim(SubEndPeriod))-1);
else;
   EndPos = str2num(StringTrim(EndYear));
   SubEndPos = str2num(StringTrim(SubEndYear));
end;
NumPeriods = min(floor(MaxForecastHorizon),floor(EndPos-SubEndPos));
if NumPeriods<0;
   NumPeriods = 0;
end;

%
% end of CalcMaxForecastHorizon.m
%
