function [EStatus,ExportData,ErrorStr,HeaderStr] = CalcExportTransformData(VariableNames,VariableData,Transformation)
% CalcExportTransformData: Computes export transformation of a matrix with data using the variables individual
%                          export functions.
%
% USAGE:
%
%       [EStatus,ExportData,ErrorStr,HeaderStr] = CalcExportTransformData(VariableNames,VariableData,Transformation)
%
% REQUIRED INPUT:  VariableNames (string matrix) with the names for the variables that should
%                                be transformed with their export functions.
%
%                  VariableData (matrix) of dimension Txk, where T is the number of observations
%                                and k is the number of variables. The latter must be equal to
%                                the number of rows in the string matrix VariableNames.
%
%                  Transformation (structure) with field names. The field name to use is determined
%                                by CurrVariable. This field name has subfields "exportfcn",
%                                "exportinitial", and "exportx" that are used for the inversion.
%
% REQUIRED OUTPUT: EStatus (boolean) that takes on the value 1 if the export transformation were
%                                successfully performed and 0 otherwise.
%
%                  ExportData (matrix) of dimension Txk with the exported data.
%
%                  ErrorStr (string vector) with error messages (if any).
%
%                  HeaderStr (string vector) with a suitable header message (if any).
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: November 9, 2007.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 14-11-2007: Added the input variables "FirstObs" and "LastObs" and took the new fields "exportx.data"
%               and "exportx.numinitialobs" into account. The output variables "ErrorStr" and "HeaderStr"
%               were also added.
%
% * 20-11-2007: The inputs "FirstObs" and "LastObs" have been removed.
%
% * 28-3-2008:  Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
EStatus = 0;
ExportData = [];
ErrorStr = '';
HeaderStr = '';
%
% test if the number of variables match
%
if size(VariableNames,1)~=size(VariableData,2);
   VariableData = VariableData';
   if size(VariableNames,1)~=size(VariableData,2);
      return;
   end;
end;
%
% create a potential output candidate
%
DataMatrix = NaN*ones(size(VariableData));
[T k] = size(DataMatrix);
%
NumTransforms = 0;
for i=1:k;
   %
   % compute the transformation
   %
   z = VariableData(:,i);
   CurrVariable = StringTrim(VariableNames(i,:));
   [y,InvStatus,ErrorStr,HeaderStr] = ExportTransformData(z,Transformation,CurrVariable);
   if (InvStatus==0)&length(y)>=T;
      %
      % make sure y doesn't have too many observations
      %
      y = y(length(y)-T+1:length(y));
      DataMatrix(:,i) = y;
      NumTransforms = NumTransforms+1;
   elseif (InvStatus==0)&length(y)<T;
      DataMatrix(T-length(y)+1:T,i) = y;
      NumTransforms = NumTransforms+1;
   else;
      break;
   end;
end;
%
% check if all exports were successful and load the exported data into the output
% matrix if this is true.
%
if NumTransforms==k;
   EStatus = 1;
   ExportData = DataMatrix;
end;

%
% end of CalcExportTransformData.m
%
