function RawBVARPosteriorPlots(ParamStr,PostSample,NumBurnin,DSGEModel,CurrINI)
% RawBVARPosteriorPlots: Provides plots of the posterior draws over all the draws of certain parameters
%                        of the Bayesian VAR model.
%
% USAGE:
%
%       RawBVARPosteriorPlots(ParamStr,PostSample,NumBurnin,DSGEModel,CurrINI)
%
% REQUIRED INPUT: ParamStr (string vector) taking on the values "Psi", "Pi", or "Omega". These string values
%                          correspond to the steady state parameters, the parameters on lagged endogenous
%                          variables, and the parameters of the residual covariance matrix.
%
%                 PostSample (matrix) with all the posterior draws of the specified groups of parameters.
%
%                 NumBurnin (integer) with the number of burn-in draws.
%
%                 DSGEModel (structure)  whose fields contain all required information
%                          about the model, such as input files, variable and sample
%                          data. Actual observations are not stored in this file.
%
%                 CurrINI (structure) with initialization information.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: February 5, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 6-2-2007:   Removed the inputs p, q, and k.
%
% * 8-2-2007:   Added a second "drawnow" call at the end of the figure loop.
%
% * 9-11-2007:  Updated the documentation.
%
% * 21-11-2007: Updated the documentation.
%
% * 10-1-2008:  Changed the last drawnow call to pause(0.02).
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

[NumIter,NumParam] = size(PostSample);
%
% Try to load the posterior mode estimates
%
ModeFile = [DSGEModel.OutputDirectory '\mode\PosteriorMode-BVAR-' DSGEModel.NameOfModel '-' num2str(DSGEModel.PriorType,'%0.0f') '.' num2str(DSGEModel.OmegaPriorType,'%0.0f') '.mat'];
ErrorStr = '';
try;
   ModeData = load(ModeFile);
catch;
   ErrorStr = ['Unable to load the file "' ModeFile '". Message caught is: ' lasterr];
end;
if isempty(ErrorStr)==0;
   About(ErrorStr,'information','YADA - Bad MAT File',120,500,CurrINI);
   return;
end;
%
% data loaded fine
%
PsiMode = ModeData.Psi;
PiMode = vec(ModeData.Pi);
OmegaMode = ModeData.Omega;
%
% Determine the variable names
%
VariableNamesBVAR = DSGEModel.VariableNames(DSGEModel.BVARY,:);
if isempty(DSGEModel.BVARX)==0;
   XVariableNamesBVAR = DSGEModel.XVariableNames(DSGEModel.BVARX,:);
else;
   XVariableNamesBVAR = '';
end;
p = length(DSGEModel.BVARY);
%
% Directory where the plots are stored
%
GraphDirectory = [DSGEModel.OutputDirectory '\graphs'];
%
% Compute the number of figures (max 16 plots per graph)
%
NumFigs = ceil(NumParam/16);
%
if NumFigs==1;
   [NumRows,NumCols] = VillaniConstructOptimalSubplot(NumParam);
else;
   NumRows = 4;
   NumCols = 4;
end;
NumPlots = 0;
pos = [(CurrINI.scrsz(3)-1000)/2 (CurrINI.scrsz(4)-680)/2 1000 650];
Xvalues = (1:NumIter)';
%
% now we can start the loops
%
IndexOne = 0;
IndexTwo = 1;
IndexThree = 1;
FigParmStr = '';
ModeEstimate = 0;
for i=1:NumFigs;
   if i>1;
      pos(1) = pos(1)+22;
      pos(2) = pos(2)-22;
      if pos(1)+1000>CurrINI.scrsz(3);
         pos(1) = 0;
      end;
      if pos(2)<=32;
         pos(2) = CurrINI.scrsz(4)-700;
      end;
   end;
   FigHandles{i} = figure('Visible','off','Position',pos);
   if NumFigs==1;
      set(FigHandles{i},'Name',['Raw Posterior Draws of ' ParamStr]);
   else;
      set(FigHandles{i},'Name',['Raw Posterior Draws of ' ParamStr ' - Figure ' num2str(i,'%0.0f')]);
   end;
   SetFigureProperties(FigHandles{i},CurrINI);
   %
   k = 0;
   for r=1:NumRows;
      for c=1:NumCols;
         NumPlots = NumPlots+1;
         if NumPlots<=NumParam;
            k = k+1;
            subplot(NumRows,NumCols,k);
            if strcmp(lower(ParamStr),'psi')==1;
               %
               % determine the variables
               %
               if IndexOne==p;
                  IndexOne = 1;
                  IndexTwo = IndexTwo+1;
               else;
                  IndexOne = IndexOne+1;
               end;
               FigParmStr = ['Psi(' StringTrim(VariableNamesBVAR(IndexOne,:)) ',' StringTrim(XVariableNamesBVAR(IndexTwo,:)) ')'];
               ModeEstimate = PsiMode(IndexOne,IndexTwo);
            elseif strcmp(lower(ParamStr),'pi')==1;
               if IndexOne==p;
                  IndexOne = 1;
                  IndexTwo = IndexTwo+1;
                  %
                  % if we should add a lag or not
                  %
                  if IndexTwo>p;
                     IndexTwo = 1;
                     IndexThree = IndexThree+1;
                  end;
               else;
                  IndexOne = IndexOne+1;
               end;
               FigParmStr = ['Pi(' StringTrim(VariableNamesBVAR(IndexOne,:)) ',' StringTrim(VariableNamesBVAR(IndexTwo,:)) ';Lag=' num2str(IndexThree,'%0.0f') ')'];
               ModeEstimate = PiMode(NumPlots,1);
            else;
               if IndexOne==p;
                  IndexOne = IndexTwo+1;
                  IndexTwo = IndexTwo+1;
               else;
                  IndexOne = IndexOne+1;
               end;
               FigParmStr = ['Omega(' StringTrim(VariableNamesBVAR(IndexOne,:)) ',' StringTrim(VariableNamesBVAR(IndexTwo,:)) ')'];
               ModeEstimate = OmegaMode(IndexOne,IndexTwo);
            end;
            %
            % plot the posterior draws
            %
            dh = plot(Xvalues,PostSample(:,NumPlots),'-','Color',[0.75 0 0],'LineWidth',1);
            hold('on');
            SetAxesFonts(CurrINI,'Title',FigParmStr);
            axis('tight');
            ylim = get(gca,'YLim');
            add_on = 0.1*(ylim(2)-ylim(1));
            ylim = [ylim(1)-add_on ylim(2)+add_on];
            %
            set(gca,'YLim',ylim,'XLim',[0 NumIter]);
            %
            % plot the BurnIn value
            %
            bi = plot(NumBurnin*ones(1,2),ylim,'-','Color',[0 0 0]);
            hold('on');
            %
            % plot the posterior mode estimate
            %
            pme = plot([0 NumIter],ModeEstimate*ones(1,2),'-','Color',[0 0 0]);
            hold('off');
            drawnow;
         end;
      end;
   end;
   %
   % save it to disk before
   %
   if DirExist(GraphDirectory)==1;
      if NumFigs>1;
         file = [GraphDirectory '\RawPosteriorDraws-BVAR-' ParamStr '-' num2str(DSGEModel.PriorType,'%0.0f') '.' num2str(DSGEModel.OmegaPriorType,'%0.0f') '-' num2str(NumIter,'%0.0f') '-' num2str(i,'%0.0f') '.fig'];
      else;
         file = [GraphDirectory '\RawPosteriorDraws-BVAR-' ParamStr '-' num2str(DSGEModel.PriorType,'%0.0f') '.' num2str(DSGEModel.OmegaPriorType,'%0.0f') '-' num2str(NumIter,'%0.0f') '.fig'];
      end;
      saveas(FigHandles{i},file);
   end;
   %
   % Fix the toolbar cdata
   %
   SetToolbarImages(FigHandles{i},CurrINI.images);
   %
   % Display the current figure
   %
   set(FigHandles{i},'Visible','on','CloseRequestFcn','delete(gcf); drawnow; pause(0.02);');
   drawnow;
end;

%
% end of RawBVARPosteriorPlots.m
%
