function PrintBVARPosteriorMode(DSGEModel,CurrINI,Psi,Pi,Omega,VariableNamesBVAR,XVariableNamesBVAR,thetaPsi,OmegaPsi,thetaPi,OmegaPi,OverallTightness,CrossEqTightness,HarmonicLagDecay,Amatrix,qDF,NewStartYear,NewStartPeriod,MaxTime,ToleranceValue,MaxIter,NumIter,ConvCrit,logPost);
% PrintBVARPosteriorMode: Prints the results from the posterior mode estimation of the Bayesian VAR to file.
%
% USAGE:
%
%       PrintBVARPosteriorMode(DSGEModel,CurrINI,Psi,Pi,Omega,VariableNamesBVAR,XVariableNamesBVAR,thetaPsi,...
%                              OmegaPsi,thetaPi,OmegaPi,OverallTightness,CrossEqTightness,HarmonicLagDecay,...
%                              Amatrix,qDF,NewStartYear,NewStartPeriod,MaxTime,ToleranceValue,MaxIter,NumIter,ConvCrit);
%
% REQUIRED INPUT: DSGEModel (structure) whose fields contain all required information
%                           about the model, such as input files, variable and sample
%                           data. Actual observations are not stored in this file.
%
%                 CurrINI (structure), with initialization information.
%
%                 Psi (pxq) matrix with posterior mode estimate of the steady state parameters.
%
%                 Pi (pxpk) matrix with posterior mode estimate of the parameters on lagged endogenous variables.
%
%                 Omega (pxp) matrix with posterior mode estimate of the residual covariance matrix.
%
%                 VariableNamesBVAR (string matrix) with the names of the endogenous variables for the
%                           Bayesian VAR model.
%
%                 XVariableNamesBVAR (string matrix) with the names of the exogenous variables for the
%                           Bayesian VAR model.
%
%                 thetaPsi (pq) vector with the prior mean for the steady state.
%
%                 OmegaPsi (pqxpq) matrix with the prior variances (along the diagonal) for the steady state.
%
%                 thetaPi (p^2k) vector with the prior means of vec(Pi). This input is empty if a diffuse prior
%                           has been applied to Pi.
%
%                 OmegaPi (p^2kxp^2k) or (pkxpk) matrix with prior covariances of vec(Pi). If a Minnestota-style
%                           prior has been used, then the dimension is p^2k, while the dimension pk occurs when
%                           a normnal conditional on Omega prior has been used, This input is empty if a diffuse
%                           prior has been applied to Pi.
%
%                 OverallTightness (scalar) with the overall tightness of the prior on Pi.
%
%                 CrossEqTightness (scalar) with the cross-equation tightness of the prior on Pi.
%
%                 HarmonicLagDecay (scalar) with the harmonic lag decay of the prior on Pi.
%
%                 Amatrix (pxp) matrix with the prior data on Omega.
%
%                 qDF (integer) with the number of degrees of freedom of the prior on Omega.
%
%                 NewStartYear (string vector) with the start year for the estimation.
%
%                 NewStartPeriod (string vector) with the start period for the estimation.
%
%                 MaxTime (scalar) with the optimization time.
%
%                 ToleranceValue (scalar) with the tolerance value for the optimization.
%
%                 MaxIter (integer) with the maximum number of iterations.
%
%                 NumIter (integer) with the number of iterations used.
%
%                 ConvCrit (scalar) with the value of the convergence criterion.
%
%                 logPost (scalar) with the value of the log posterior.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: January 18, 2007.
%                        This version: January 21, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 19-1-2007:  Finished work on the function.
%
% * 31-1-2007:  The output file now takes the type of prior used for the Bayesian VAR into account.
%
% * 2-2-2007:   Updated the documentation.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 21-01-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% some useful data parameters.
%
p = length(DSGEModel.BVARY);
k = DSGEModel.BVARLags;
q = length(DSGEModel.BVARX);
%
% Transform some parameters
%
ThetaPsi = reshape(thetaPsi,p,q);
OmegaPsi = reshape(diag(OmegaPsi),p,q);
if isempty(thetaPi)==0;
   ThetaPi = reshape(thetaPi,p,p*k);
   if (DSGEModel.PriorType==2)&(qDF>0);
      OmegaPi = kron(OmegaPi,Amatrix);
   elseif DSGEModel.PriorType~=1;
      OmegaPi = [];
   end;
   if isempty(OmegaPi)==0;
      OmegaPi = reshape(diag(OmegaPi),p,p*k);
   end;
end;
%
ModeTextFile = [DSGEModel.OutputDirectory '\PosteriorModeBVAR-' DSGEModel.NameOfModel '-' num2str(DSGEModel.PriorType,'%0.0f') '.' num2str(DSGEModel.OmegaPriorType,'%0.0f') '.txt'];
%
fid = fopen(ModeTextFile,'wt');
%
% the header information
%
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*                     B A Y E S I A N   V A R   M O D E L                      *\n');
fprintf(fid,'*    P O S T E R I O R   M O D E   E S T I M A T I O N   W I T H   Y A D A     *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% Model info
%
fprintf(fid,'Name of model:                    %s\n',DSGEModel.NameOfModel);
fprintf(fid,'Data construction file:           %s\n',GetFilename(DSGEModel.DataConstructionFile));
fprintf(fid,'Steady state prior file:          %s\n\n',GetFilename(DSGEModel.SteadyStatePriorFile));
%
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*              S A M P L E   &   D A T A   I N F O R M A T I O N               *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% Sample information
%
fprintf(fid,'Estimation sample:                %s:%s -- %s:%s\n',NewStartYear,NewStartPeriod,DSGEModel.SubEndYear,DSGEModel.SubEndPeriod);
fprintf(fid,'Number of lags of the VAR model:  %0.0f\n',k);
fprintf(fid,'Number of endogenous variables:   %0.0f\n',p);
fprintf(fid,'Number of exogenous variables:    %0.0f\n\n',q);
%
% variable names
%
VarNames = StrWrap(vecstr(VariableNamesBVAR,' '),60);
fprintf(fid,'Names of endogenous variables:    %s\n',VarNames(1,:));
for i=2:size(VarNames,1);
   if strcmp(StringTrim(VarNames(i,:)),'')==0;
      fprintf(fid,'                                  %s\n',VarNames(i,:));
   end;
end;
if q>0;
   XVarNames = StrWrap(vecstr(XVariableNamesBVAR,' '),60);
   fprintf(fid,'Names of exogenous variables:     %s\n',XVarNames(1,:));
   for i=2:size(XVarNames,1);
      if strcmp(StringTrim(XVarNames(i,:)),'')==0;
         fprintf(fid,'                                  %s\n',XVarNames(i,:));
      end;
   end;
end;
%
% General information about the prior
%
fprintf(fid,'\n********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*                      P R I O R   I N F O R M A T I O N                       *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
if q>0;
   fprintf(fid,'Psi:                              Normal distribution\n');
end;
if DSGEModel.PriorType==1;
   %
   % we have a Minnesota-style prior on Pi
   %
   fprintf(fid,'Pi:                               Minnesota-style\n');
   fprintf(fid,'Overall tightness:                %0.6f\n',OverallTightness);
   fprintf(fid,'Cross-equation tightness:         %0.6f\n',CrossEqTightness);
   fprintf(fid,'Harmonic lag decay:               %0.6f\n',HarmonicLagDecay);
elseif DSGEModel.PriorType==2;
   %
   % we have a normal conditional on redidual covariance matrix prior on Pi
   %
   fprintf(fid,'Pi:                               Normal conditional on Omega\n');
   fprintf(fid,'Overall tightness:                %0.6f\n',OverallTightness);
   fprintf(fid,'Harmonic lag decay:               %0.6f\n',HarmonicLagDecay);
else;
   %
   % we have a diffuse prior on Pi
   %
   fprintf(fid,'Pi:                               Diffuse\n');
end;
if DSGEModel.OmegaPriorType==1;
   %
   % We have a diffuse prior on Omega
   %
   fprintf(fid,'Omega:                            Diffuse\n');
else;
   %
   % we have an inverse Wishart prior on Omega
   %
   fprintf(fid,'Omega:                            Inverse Wishart\n');
   if DSGEModel.WishartType==1;
      %
      % We use the ML estimate of Omega for the A matrix
      %
      fprintf(fid,'A matrix:                         ML estimate of Omega\n');
   else;
      %
      % we have a common variance tightness hyperparameters
      %
      fprintf(fid,'Variance tightness:               %0.6f\n',Amatrix(1,1));
   end;
   fprintf(fid,'Degrees of freedom:               %0.0f\n',qDF);
end;
%
% optimization information
%
fprintf(fid,'\n********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*               O P T I M I Z A T I O N   I N F O R M A T I O N                *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% optimization information
%
fprintf(fid,'Time:                             %s (hrs:min:sec)\n',SecondsToHoursStr(MaxTime));
fprintf(fid,'Tolerance level:                  %0.10f\n',ToleranceValue);
fprintf(fid,'Maximum number of iterations:     %0.0f\n',MaxIter);
fprintf(fid,'Number of iterations:             %0.0f\n',NumIter);
fprintf(fid,'Convergence value:                %0.10f\n',ConvCrit);
fprintf(fid,'Value of the log posterior:       %0.6f\n\n',logPost);
%
% Parameters
%
prt_val = ['%' num2str(6+CurrINI.decimals,'%0.0f') '.' num2str(CurrINI.decimals,'%0.0f') 'f'];
fprintf(fid,'********************************************************************************\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'*                             P A R A M E T E R S                              *\n');
fprintf(fid,'*                                                                              *\n');
fprintf(fid,'********************************************************************************\n\n');
%
% The steady state
%
if q>0;
   fprintf(fid,'STEADY STATE PARAMETERS - Psi\n\n');
   head_str = '  Eq\Var  ';
   add_head_str = '';
   if size(XVariableNamesBVAR,2)<10;
      add_head_str = SpaceStr(10-size(XVariableNamesBVAR,2));
   end;
   head_prior_mean_str = '          ';
   head_prior_var_str = '          ';
   add_name_str = '';
   if size(VariableNamesBVAR,2)<10;
      add_name_str = SpaceStr(10-size(VariableNamesBVAR,2));
   end;
   prt_str = '%s%s';
   b_prt_str = '          ';
   for i=1:q;
      head_str = [head_str sprintf('       %s%s',add_head_str,XVariableNamesBVAR(i,:))];
      head_prior_mean_str = [head_prior_mean_str '       prior mean'];
      head_prior_var_str = [head_prior_var_str '         variance'];
      prt_str = [prt_str '     ' prt_val];
      b_prt_str = [b_prt_str '     ' prt_val];
   end;
   TopStr = MultiCharStr('=',length(head_str)+2);
   MidStr = MultiCharStr('-',length(head_str)+2);
   fprintf(fid,'%s\n',TopStr);
   fprintf(fid,'%s\n',head_str);
   fprintf(fid,'%s\n',head_prior_mean_str);
   fprintf(fid,'%s\n',head_prior_var_str);
   for i=1:p;
      fprintf(fid,'%s\n',MidStr);
      fprintf(fid,[prt_str '\n'],add_name_str,VariableNamesBVAR(i,:),Psi(i,:)');
      fprintf(fid,[b_prt_str '\n'],ThetaPsi(i,:)');
      fprintf(fid,[b_prt_str '\n'],OmegaPsi(i,:)');
   end;
   fprintf(fid,'%s\n\n',TopStr);
end;
%
% The parameters on lagged values
%
fprintf(fid,'PARAMETERS ON LAGGED ENDOGENOUS VARIABLES - Pi\n\n');
%
for l=1:k;
   Pil = Pi(:,1+p*(l-1):p*l);
   if isempty(thetaPi)==0;
      PilPrior = ThetaPi(:,1+p*(l-1):p*l);
   end;
   if isempty(OmegaPi)==0;
      OmegaPil = OmegaPi(:,1+p*(l-1):p*l);
   end;
   if l==1;
      add_name_str = '';
      if size(VariableNamesBVAR,2)<10;
         add_name_str = SpaceStr(10-size(VariableNamesBVAR,2));
      end;
      head_str = '  Eq\Var  ';
      prt_str = '%s%s';
      if DSGEModel.PriorType~=3;
         head_prior_mean_str = '          ';
         head_prior_var_str = '          ';
         b_prt_str = '          ';
      end;
      for i=1:p;
         head_str = [head_str sprintf('       %s%s',add_name_str,VariableNamesBVAR(i,:))];
         prt_str = [prt_str '     ' prt_val];
         if DSGEModel.PriorType~=3;
            head_prior_mean_str = [head_prior_mean_str '       prior mean'];
            head_prior_var_str = [head_prior_var_str '         variance'];
            b_prt_str = [b_prt_str '     ' prt_val];
         end;
      end;
      TopStr = MultiCharStr('=',length(head_str)+2);
      MidStr = MultiCharStr('-',length(head_str)+2);
   end;
   %
   % ready to print
   %
   fprintf(fid,'%s\n',TopStr);
   fprintf(fid,'Lag: %2.0f\n\n',l);
   fprintf(fid,'%s\n',head_str);
   if (DSGEModel.PriorType~=3)&(l==1);
      fprintf(fid,'%s\n',head_prior_mean_str);
      if isempty(OmegaPi)==0;
         fprintf(fid,'%s\n',head_prior_var_str);
      end;
   end;
   for i=1:p;
      fprintf(fid,'%s\n',MidStr);
      fprintf(fid,[prt_str '\n'],add_name_str,VariableNamesBVAR(i,:),Pil(i,:)');
      if isempty(thetaPi)==0;
         fprintf(fid,[b_prt_str '\n'],PilPrior(i,:)');
      end;
      if isempty(OmegaPi)==0;
         fprintf(fid,[b_prt_str '\n'],OmegaPil(i,:)');
      end;
   end;
   fprintf(fid,'%s\n\n',TopStr);
end;
%
% covariance parameters
%
fprintf(fid,'RESIDUAL COVARIANCE PARAMETERS - Omega\n\n');
%
head_str = '  Eq\Res  ';
prt_str = '%s%s';
if DSGEModel.OmegaPriorType==2;
   head_prior_mean_str = '          ';
   b_prt_str = '          ';
end;
for i=1:p;
   head_str = [head_str sprintf('  %sRes(%s)',add_name_str,VariableNamesBVAR(i,:))];
   prt_str = [prt_str '     ' prt_val];
   if DSGEModel.OmegaPriorType==2;
      head_prior_mean_str = [head_prior_mean_str '       prior mean'];
      b_prt_str = [b_prt_str '     ' prt_val];
   end;
end;
%
fprintf(fid,'%s\n',TopStr);
fprintf(fid,'%s\n',head_str);
if DSGEModel.OmegaPriorType==2;
   fprintf(fid,'%s\n',head_prior_mean_str);
end;
for i=1:p;
   fprintf(fid,'%s\n',MidStr);
   fprintf(fid,[prt_str '\n'],add_name_str,VariableNamesBVAR(i,:),Omega(i,:)');
   if DSGEModel.OmegaPriorType==2;
      fprintf(fid,[b_prt_str '\n'],(1/(qDF-p-1))*Amatrix(i,:)');
   end;
end;
fprintf(fid,'%s\n\n',TopStr);
%
% Some information about the filename and its location
%
fprintf(fid,'Directory for file: %s\n',GetPath(ModeTextFile));
fprintf(fid,'Name of file:       %s\n',GetFilename(ModeTextFile));
fprintf(fid,'Output created on:  %s\n\n',datestr(now,0));
fclose(fid);

%
% end of PrintBVARPosteriorMode.m
%
